import logging
from typing import Any

from fastapi import APIRouter, Depends, Request

from intranet.trip.src.api.auth.permissions import is_yandex_coordinator
from intranet.trip.src.api.depends import get_cache, get_unit_of_work
from intranet.trip.src.cache import Cache
from intranet.trip.src.exceptions import PermissionDenied
from intranet.trip.src.unit_of_work import UnitOfWork
from intranet.trip.src.logic import monitoring


logger = logging.getLogger(__name__)
router = APIRouter()


@router.get('/unistat', response_model=list[Any])
async def unistat(
    uow: UnitOfWork = Depends(get_unit_of_work(read_only=True)),
):
    return await monitoring.unistat(uow)


@router.get('/arq', response_model=dict[str, Any])
async def arq_stats(
    request: Request,
    cache: Cache = Depends(get_cache),
    function: str = None,
):
    if not is_yandex_coordinator(request.state.user):
        raise PermissionDenied(status_code=403)

    return await monitoring.get_arq_stats(
        redis=request.app.state.redis,
        cache=cache,
        function=function,
    )
