import logging

from fastapi import APIRouter, Depends, Request, Response, HTTPException

from intranet.trip.src.api.auth import (
    check_user_perm_by_person,
    has_person_perm,
    has_person_partial_perm,
    LazyRolesMap,
)
from intranet.trip.src.api.decorators import exclude_route
from intranet.trip.src.api.depends import get_unit_of_work
from intranet.trip.src.api.schemas import (
    FakePerson,
    PersonCreate,
    PersonDocumentCreate,
    PersonDocument,
    DocumentId,
    PersonDetails,
    PersonDetailsPublic,
    PersonDetailsUpdate,
    ExtPerson,
    ExtPersonCreate,
    ExtPersonUpdate,
    ExtPersonId,
)
from intranet.trip.src.api.schemas.document import BonusCard, BonusCardCreate, BonusCardId
from intranet.trip.src.config import settings
from intranet.trip.src.enums import ServiceType
from intranet.trip.src.exceptions import PermissionDenied
from intranet.trip.src.logic.persons import (
    PersonalDataGateway,
    get_personal_data_gateway,
)
from intranet.trip.src.logic.ext_persons import (
    create_ext_person,
    regenerate_secret,
    update_ext_person,
)
from intranet.trip.src.unit_of_work import UnitOfWork


logger = logging.getLogger(__name__)
router = APIRouter()


@router.get('/{person_id}', response_model=PersonDetailsPublic)
async def get_person(
    request: Request,
    person_id: int,
    uow: UnitOfWork = Depends(get_unit_of_work(read_only=True)),
):
    user = request.state.user
    pdg: PersonalDataGateway = await get_personal_data_gateway(uow)
    person = await pdg.get_person(person_id)
    roles_map = LazyRolesMap(uow, person_ids=[person_id])
    if not await has_person_partial_perm(user, person, roles_map):
        raise PermissionDenied(log_message='User has not partial permission for person')
    return person


@router.get('/{person_id}/details', response_model=PersonDetails)
async def get_person_details(
    request: Request,
    person_id: int,
    uow: UnitOfWork = Depends(get_unit_of_work(read_only=True)),
):
    pdg: PersonalDataGateway = await get_personal_data_gateway(uow)
    person = await pdg.get_person(person_id)
    if not has_person_perm(request.state.user, person):
        raise PermissionDenied(log_message='User has not read permission for person details')
    return person


@router.put('/{person_id}/details', status_code=204, response_class=Response)
async def person_details_update(
    request: Request,
    person_id: int,
    person_details: PersonDetailsUpdate,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    await check_user_perm_by_person(uow, request.state.user, person_id)

    pdg: PersonalDataGateway = await get_personal_data_gateway(uow)
    await pdg.update_person(person_id, person_details)


@router.post('/{person_id}/documents', status_code=201, response_model=DocumentId)
async def person_document_create(
    request: Request,
    person_id: int,
    document: PersonDocumentCreate,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    person = await uow.persons.get_person(person_id)
    if not has_person_perm(request.state.user, person):
        raise PermissionDenied(log_message=f'User has not read permission for person {person_id}')

    pdg: PersonalDataGateway = await get_personal_data_gateway(uow)
    document_id = await pdg.create_person_document(person, document)
    return {
        'document_id': document_id,
    }


@router.get('/{person_id}/documents', response_model=list[PersonDocument])
async def person_document_list(
    request: Request,
    person_id: int,
    uow: UnitOfWork = Depends(get_unit_of_work(read_only=True)),
):
    await check_user_perm_by_person(uow, request.state.user, person_id)

    pdg: PersonalDataGateway = await get_personal_data_gateway(uow)
    return await pdg.get_person_documents(person_id, request.state.user)


@router.delete('/{person_id}/documents/{document_id}', status_code=204, response_class=Response)
async def person_document_delete(
    request: Request,
    person_id: int,
    document_id: int,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    person = await uow.persons.get_person(person_id)
    if not has_person_perm(request.state.user, person):
        raise PermissionDenied(status_code=403, log_message='not your document')

    pdg: PersonalDataGateway = await get_personal_data_gateway(uow)
    await pdg.delete_person_document(
        person=person,
        document_id=document_id,
    )


@router.get('/{person_id}/bonus_cards', response_model=list[BonusCard])
async def person_bonus_card_list(
    request: Request,
    person_id: int,
    service_type: ServiceType = None,
    uow: UnitOfWork = Depends(get_unit_of_work(read_only=True)),
):
    await check_user_perm_by_person(uow, request.state.user, person_id)

    return await uow.bonus_cards.get_person_bonus_cards(person_id, service_type)


@router.get('/{person_id}/bonus_cards/{bonus_card_id}', response_model=BonusCard)
async def get_bonus_card_detail(
    request: Request,
    person_id: int,
    bonus_card_id: int,
    uow: UnitOfWork = Depends(get_unit_of_work(read_only=True)),
):
    await check_user_perm_by_person(uow, request.state.user, person_id)

    async with uow:
        bonus_card = await uow.bonus_cards.get_bonus_card(bonus_card_id)
    return bonus_card


@router.post('/{person_id}/bonus_cards', status_code=201, response_model=BonusCardId)
async def bonus_card_create(
    request: Request,
    person_id: int,
    bonus_card: BonusCardCreate,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    await check_user_perm_by_person(uow, request.state.user, person_id)

    async with uow:
        bonus_card_id = await uow.bonus_cards.create(
            person_id=person_id,
            **bonus_card.dict(),
        )
        uow.add_job(
            job_name='sync_profiles_to_ihub_task',
            person_ids=[person_id],
            unique=False,
        )
    return {
        'bonus_card_id': bonus_card_id,
    }


@router.delete('/{person_id}/bonus_cards/{bonus_card_id}', status_code=204, response_class=Response)
async def bonus_card_delete(
    request: Request,
    person_id: int,
    bonus_card_id: int,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    person = await uow.persons.get_person(person_id)
    if not has_person_perm(request.state.user, person):
        raise HTTPException(status_code=403, detail='not your bonus_card')

    async with uow:
        bonus_card = await uow.bonus_cards.get_bonus_card(bonus_card_id)
        await uow.bonus_cards.delete(bonus_card_id)

        uow.add_job(
            job_name='sync_profiles_to_ihub_task',
            person_ids=[bonus_card.person_id],
            unique=False,
        )


@router.post('/{person_id}/ext_persons', status_code=201, response_model=ExtPersonId)
async def ext_person_create(
    request: Request,
    person_id: int,
    ext_person: ExtPersonCreate,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    await check_user_perm_by_person(uow, request.state.user, person_id)

    return await create_ext_person(
        uow=uow,
        person_id=person_id,
        ext_person_create=ext_person,
    )


@router.post(
    '/{person_id}/ext_persons/{ext_person_id}/regenerate_secret',
    response_model=ExtPersonId,
)
async def ext_person_regenerate(
    request: Request,
    person_id: int,
    ext_person_id: int,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    await check_user_perm_by_person(uow, request.state.user, person_id)

    return await regenerate_secret(
        uow=uow,
        person_id=person_id,
        ext_person_id=ext_person_id,
    )


# TODO: Эта ручка должна дергаться внешними людьми.
# Но прежде чем делать ее без авторизации, нужно договориться о схеме с СИБ
@router.put('/{person_id}/ext_persons/{ext_person_id}', status_code=200, response_class=Response)
async def ext_person_update(
    person_id: int,
    ext_person_id: int,
    ext_person_in: ExtPersonUpdate,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    try:
        await update_ext_person(uow, ext_person_id, ext_person_in)
    except Exception:
        error_msg = 'Error during updating ext person'
        logger.exception(error_msg)
        raise HTTPException(status_code=400, detail=error_msg)

    return Response(status_code=200)


@router.get('/{person_id}/ext_persons/', response_model=list[ExtPerson])
async def ext_person_list(
    request: Request,
    person_id: int,
    uow: UnitOfWork = Depends(get_unit_of_work(read_only=True)),
):
    await check_user_perm_by_person(uow, request.state.user, person_id)

    return await uow.ext_persons.get_ext_persons(person_id=person_id)


@router.delete('/{person_id}/ext_persons/{ext_person_id}', status_code=204, response_class=Response)
async def ext_person_delete(
    request: Request,
    person_id: int,
    ext_person_id: int,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    await check_user_perm_by_person(uow, request.state.user, person_id)

    async with uow:
        await uow.ext_persons.delete(ext_person_id)


# TODO После тестирование b2b стенда удалилить
@router.post('/fake_create/', status_code=200, response_class=Response)
async def fake_person_create(
    request: Request,
    person_in: FakePerson,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    if not request.state.user.is_coordinator or settings.IS_YA_TEAM:
        raise PermissionDenied(log_message='User has not permission for fake person create')
    async with uow:
        await uow.persons.create(**person_in.dict())


@router.post('/create', status_code=201, response_class=Response)
@exclude_route(is_exclude=settings.IS_YA_TEAM and settings.ENV_TYPE != 'development')
async def person_create(
    request: Request,
    person: PersonCreate,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    uid = request.state.user.uid
    pdg = await get_personal_data_gateway(uow)
    if await pdg.is_person_exists(uid):
        raise PermissionDenied(log_message='Attempt to re-bind a person')
    try:
        await pdg.create_person(uid=uid, email=person.email)
    except Exception as e:
        raise PermissionDenied(log_message=str(e))
