from fastapi import APIRouter, Depends

from intranet.trip.src.api.depends import get_provider_gateway
from intranet.trip.src.api.schemas import (
    AviaSearchResult,
    AviaSearchFilter,
    AviaDetailResponse,
    FilterItem,
    HotelDetailResponse,
    HotelSearchFilter,
    HotelSearchInfo,
    HotelSearchResult,
    ProviderSearchResultCount,
    SearchAviaRequestIn,
    SearchId,
    AviaSearchInfo,
    SearchHotelRequestIn,
    SearchRailRequestIn,
    RailDetailResponse,
    RailSearchResult,
    RailSearchFilter,
    RailSearchInfo,
)
from intranet.trip.src.logic.providers.gateway import ProviderGateway


router = APIRouter()


@router.post(
    '/avia',
    status_code=201,
    responses={201: {'model': SearchId}},
    summary='Создание поискового запроса для авиа',
)
async def create_avia_search(
    search_request_in: SearchAviaRequestIn,
    provider_gateway=Depends(get_provider_gateway),
):
    return await provider_gateway.create_avia_search(search_request_in=search_request_in)


@router.get(
    '/avia/{search_id}',
    status_code=200,
    responses={200: {'model': AviaSearchInfo}},
    summary='Общая информация о поисковом запросе для авиа',
)
async def get_avia_search_info(
    search_id: str,
    provider_gateway=Depends(get_provider_gateway),
):
    return await provider_gateway.get_avia_search_info(search_id=search_id)


@router.get(
    '/avia/{search_id}/count',
    status_code=200,
    responses={200: {'model': ProviderSearchResultCount}},
    summary='Количество результатов поискового запроса для авиа',
)
async def get_avia_search_result_count(
    search_id: str,
    search_filter=Depends(AviaSearchFilter),
    provider_gateway=Depends(get_provider_gateway),
):
    return await provider_gateway.get_search_results_count(
        search_id=search_id,
        search_filter=search_filter,
    )


@router.get(
    '/avia/{search_id}/results',
    status_code=200,
    responses={200: {'model': AviaSearchResult}},
    summary='Результаты поискового запроса для авиа',
)
async def get_avia_search_results(
    search_id: str,
    search_filter=Depends(AviaSearchFilter),
    provider_gateway=Depends(get_provider_gateway),
    limit: int = 20,
    page: int = 1,
):
    return await provider_gateway.get_avia_search_results(
        search_id=search_id,
        search_filter=search_filter,
        limit=limit,
        page=page,
    )


@router.get(
    '/avia/{search_id}/results/{key}',
    status_code=200,
    responses={200: {'model': AviaDetailResponse}},
    summary='Детализация авиа',
)
async def get_avia_search_details(
    search_id: str,
    key: str,
    provider_gateway=Depends(get_provider_gateway),
):
    return await provider_gateway.get_avia_search_details(
        search_id=search_id,
        key=key,
    )


@router.get(
    '/avia/{search_id}/filters',
    status_code=200,
    responses={200: {'model': list[FilterItem]}},
    summary='Получение списка возможных фильтров поискового запроса для авиа',
)
async def get_avia_search_filters(
    search_id: str,
    provider_gateway=Depends(get_provider_gateway),
):
    return await provider_gateway.get_avia_search_filters(
        search_id=search_id,
    )


@router.post(
    '/hotel',
    status_code=201,
    responses={201: {'model': SearchId}},
    summary='Создание поискового запроса для отелей',
)
async def create_hotel_search(
    search_request_in: SearchHotelRequestIn,
    provider_gateway=Depends(get_provider_gateway),
):
    return await provider_gateway.create_hotel_search(search_request_in=search_request_in)


@router.get(
    '/hotel/{search_id}/filters',
    status_code=200,
    responses={200: {'model': list[FilterItem]}},
    summary='Доступные фильтров для поискового запроса',
)
async def get_hotel_search_filters(
    search_id: str,
    provider_gateway: ProviderGateway = Depends(get_provider_gateway),
):
    return await provider_gateway.get_hotel_search_filters(
        search_id=search_id,
    )


@router.get(
    '/hotel/{search_id}/count',
    status_code=200,
    responses={200: {'model': ProviderSearchResultCount}},
    summary='Количество результатов поискового запроса для отелей',
)
async def get_hotel_search_result_count(
    search_id: str,
    search_filter: HotelSearchFilter = Depends(),
    provider_gateway: ProviderGateway = Depends(get_provider_gateway),
):
    return await provider_gateway.get_hotel_search_results_count(
        search_id=search_id,
        search_filter=search_filter,
    )


@router.get(
    '/hotel/{search_id}',
    status_code=200,
    responses={200: {'model': HotelSearchInfo}},
    summary='Общая информация о поисковом запросе для отелей',
)
async def get_hotel_search_info(
    search_id: str,
    provider_gateway: ProviderGateway = Depends(get_provider_gateway),
):
    return await provider_gateway.get_hotel_search_info(search_id=search_id)


@router.get(
    '/hotel/{search_id}/results',
    status_code=200,
    responses={200: {'model': HotelSearchResult}},
    summary='Результаты поискового запроса для отелей',
)
async def get_hotel_search_results(
    search_id: str,
    search_filter: HotelSearchFilter = Depends(),
    provider_gateway: ProviderGateway = Depends(get_provider_gateway),
    limit: int = 20,
    page: int = 1,
):
    return await provider_gateway.get_hotel_search_results(
        search_id=search_id,
        search_filter=search_filter,
        limit=limit,
        page=page,
    )


@router.get(
    '/hotel/{search_id}/results/{key}',
    status_code=200,
    responses={200: {'model': HotelDetailResponse}},
    summary='Детализация отеля',
)
async def get_hotel_search_details(
    search_id: str,
    key: str,
    provider_gateway=Depends(get_provider_gateway),
):
    return await provider_gateway.get_hotel_search_details(
        search_id=search_id,
        key=key,
    )


@router.post(
    '/rail',
    status_code=201,
    responses={201: {'model': SearchId}},
    summary='Создание поискового запроса для поездов',
)
async def create_rail_search(
    search_request_in: SearchRailRequestIn,
    provider_gateway=Depends(get_provider_gateway),
):
    return await provider_gateway.create_rail_search(search_request_in=search_request_in)


@router.get(
    '/rail/{search_id}/results',
    status_code=200,
    responses={200: {'model': RailSearchResult}},
    summary='Результаты поискового запроса для поездов',
)
async def get_rail_search_results(
    search_id: str,
    search_filter=Depends(RailSearchFilter),
    provider_gateway=Depends(get_provider_gateway),
    limit: int = 20,
    page: int = 1,
):
    return await provider_gateway.get_rail_search_results(
        search_id=search_id,
        search_filter=search_filter,
        limit=limit,
        page=page,
    )


@router.get(
    '/rail/{search_id}/count',
    status_code=200,
    responses={200: {'model': ProviderSearchResultCount}},
    summary='Количество результатов поискового запроса для поездов',
)
async def get_rail_search_result_count(
    search_id: str,
    search_filter=Depends(RailSearchFilter),
    provider_gateway=Depends(get_provider_gateway),
):
    return await provider_gateway.get_rail_search_results_count(
        search_id=search_id,
        search_filter=search_filter,
    )


@router.get(
    '/rail/{search_id}',
    status_code=200,
    responses={200: {'model': RailSearchInfo}},
    summary='Общая информация о поисковом запросе для поездов',
)
async def get_rail_search_info(
    search_id: str,
    provider_gateway=Depends(get_provider_gateway),
):
    return await provider_gateway.get_rail_search_info(search_id=search_id)


@router.get(
    '/rail/{search_id}/filters',
    status_code=200,
    responses={200: {'model': list[FilterItem]}},
    summary='Получение списка возможных фильтров поискового запроса для поездов',
)
async def get_rail_search_filters(
    search_id: str,
    provider_gateway=Depends(get_provider_gateway),
):
    return await provider_gateway.get_rail_search_filters(
        search_id=search_id,
    )


@router.get(
    '/rail/{search_id}/results/{key}',
    status_code=200,
    responses={200: {'model': RailDetailResponse}},
    summary='Детализация поезда',
)
async def get_rail_search_train_details(
    search_id: str,
    key: str,
    provider_gateway=Depends(get_provider_gateway),
):
    return await provider_gateway.get_rail_search_train_details(
        search_id=search_id,
        key=key,
    )
