import logging
from intranet.trip.src.enums import ServiceType
from intranet.trip.src.models.service import ServiceProvider

from fastapi import APIRouter, Depends, Request, Response, HTTPException

from intranet.trip.src.api.auth import (
    has_service_read_perm,
    LazyRolesMap,
    LazyManagersMapByTrip,
)
from intranet.trip.src.api.schemas import (
    ServiceCreate,
    Service,
    ServiceId,
    ServiceDocumentIn,
    ServiceReserveIn,
    ServiceUpdateIn,
)
from intranet.trip.src.api.depends import get_unit_of_work
from intranet.trip.src.logic.person_trips import CreateServiceAction
from intranet.trip.src.logic.services import (
    AddDocumentAction,
    ChangeDocumentAction,
    RemoveDocumentAction,
    ReserveAction,
    UpdateAction,
    DeleteAction,
    get_service_actions,
)
from intranet.trip.src.unit_of_work import UnitOfWork


logger = logging.getLogger(__name__)
router = APIRouter()


@router.post('/', status_code=201, response_model=ServiceId)
async def service_create(
    request: Request,
    service_create: ServiceCreate,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    action = await CreateServiceAction.init(
        uow=uow,
        user=request.state.user,
        trip_id=service_create.trip_id,
        person_id=service_create.person_id,
    )
    service_id = await action.check_and_execute(service_create)
    return {
        'service_id': service_id,
    }


@router.get(
    '/providers',
    response_model=list[ServiceProvider],
    summary='Саджест поставщиков услуг'
)
async def service_provider_suggest(
    search_query: str = None,
    service_type: ServiceType = None,
    uow: UnitOfWork = Depends(get_unit_of_work(read_only=True)),
):
    search_query = search_query or ''
    async with uow:
        service_providers = await uow.service_providers.find_provider(
            search_query=search_query,
            service_type=service_type,
        )
    return service_providers


@router.get('/{service_id}', response_model=Service)
async def service_detail(
    request: Request,
    service_id: int,
    uow: UnitOfWork = Depends(get_unit_of_work(read_only=True)),
):
    service = await uow.services.get_service(service_id=service_id)
    lazy_roles_map = LazyRolesMap(uow, person_ids=[service.person.person_id])
    lazy_managers_map = LazyManagersMapByTrip(uow, trip_ids=[service.trip_id])
    if not await has_service_read_perm(
        request.state.user, service, lazy_roles_map, lazy_managers_map,
    ):
        raise HTTPException(status_code=403, detail='not your service')
    service.actions = await get_service_actions(
        uow=uow,
        user=request.state.user,
        service=service,
        roles_map=lazy_roles_map,
    )
    return service


@router.post('/{service_id}/add_document', status_code=204, response_class=Response)
async def service_document_add(
    request: Request,
    service_id: int,
    service_document_in: ServiceDocumentIn,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    action = await AddDocumentAction.init(
        uow=uow,
        user=request.state.user,
        service_id=service_id,
    )
    await action.check_and_execute(service_document_in)


@router.post('/{service_id}/change_document', status_code=204, response_class=Response)
async def service_document_change(
    request: Request,
    service_id: int,
    service_document_in: ServiceDocumentIn,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    action = await ChangeDocumentAction.init(
        uow=uow,
        user=request.state.user,
        service_id=service_id,
    )
    await action.check_and_execute(service_document_in)


@router.post('/{service_id}/remove_document', status_code=204, response_class=Response)
async def service_document_remove(
    request: Request,
    service_id: int,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    action = await RemoveDocumentAction.init(
        uow=uow,
        user=request.state.user,
        service_id=service_id,
    )
    await action.check_and_execute()


@router.post('/{service_id}/reserve', status_code=204, response_class=Response)
async def service_reserve(
    request: Request,
    service_id: int,
    service_reserve_in: ServiceReserveIn,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    action = await ReserveAction.init(
        uow=uow,
        user=request.state.user,
        service_id=service_id,
    )
    await action.check_and_execute(service_reserve_in)


@router.put('/{service_id}', response_model=ServiceId)
async def service_update(
        request: Request,
        service_id: int,
        service_update_in: ServiceUpdateIn,
        uow: UnitOfWork = Depends(get_unit_of_work()),
):
    action = await UpdateAction.init(
        uow=uow,
        user=request.state.user,
        service_id=service_id,
    )
    service_id = await action.check_and_execute(service_update_in)
    return {
        'service_id': service_id,
    }


@router.delete('/{service_id}', status_code=204, response_class=Response)
async def service_delete(
    request: Request,
    service_id: int,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    action = await DeleteAction.init(
        uow=uow,
        user=request.state.user,
        service_id=service_id,
    )
    await action.check_and_execute()
