from datetime import date
from typing import Generic, TypeVar, Optional

from pydantic import BaseModel, Field, root_validator
from pydantic.generics import GenericModel

from intranet.trip.src.lib.utils import unify_values

DataT = TypeVar('DataT')


class PaginatedResponse(GenericModel, Generic[DataT]):
    data: list[DataT]
    count: int
    page: int
    limit: int
    next: Optional[str]
    previous: Optional[str]


class LocalizedString(BaseModel):
    ru: str = None
    en: str = None


class City(BaseModel):
    name: LocalizedString


class Country(BaseModel):
    name: LocalizedString


class RoutePoint(BaseModel):
    city: City
    country: Country
    aeroclub_city_id: int = Field(None, deprecated=True)
    provider_city_id: str
    date: date
    need_hotel: bool = True
    need_transfer: bool = True

    @root_validator(pre=True)
    def set_provider_city(cls, values):
        return unify_values(values, ['provider_city_id', 'aeroclub_city_id'])


class RoutePointCreate(BaseModel):
    city: str
    country: str
    aeroclub_city_id: int = Field(None, deprecated=True)
    provider_city_id: str
    date: date
    need_hotel: bool = True
    need_transfer: bool = True

    @root_validator(pre=True)
    def set_provider_city(cls, values):
        return unify_values(values, ['provider_city_id', 'aeroclub_city_id'])


class TokenResponse(BaseModel):
    csrf_token: str
