from datetime import date
from pydantic import BaseModel, validator, root_validator, constr

from intranet.trip.src.lib.aeroclub.models.common import Name
from intranet.trip.src.enums import (
    DocumentType,
    Citizenship,
    ServiceType,
)
from intranet.trip.src.api.schemas.validators import document_person_name_validate, AlphabetCode


class DocumentId(BaseModel):
    document_id: int


class PersonDocumentBase(BaseModel):
    document_type: DocumentType
    citizenship: Citizenship = Citizenship.RU
    series: constr(max_length=8) = None
    number: constr(max_length=32) = None
    issued_on: date = None
    expires_on: date = None
    first_name: constr(strip_whitespace=True)
    last_name: constr(strip_whitespace=True)
    middle_name: constr(strip_whitespace=True) = None
    external_document_id: str = None

    @validator('expires_on', always=True)
    def field_values_required_for_passport(cls, v, values):
        if values.get('document_type') == DocumentType.external_passport:
            if v is None:
                raise ValueError('field required')
        return v


class PersonDocument(PersonDocumentBase):
    person_id: int
    document_id: int


class ExtPersonDocument(PersonDocumentBase):
    ext_person_id: int
    document_id: int


class PersonDocumentCreate(PersonDocumentBase):
    @validator('series', always=True)
    def field_series_in_passport(cls, v, values):
        required_length_map = {
            DocumentType.passport: 4,
            DocumentType.external_passport: 2,
        }
        if (
            values.get('citizenship') == Citizenship.RU
            and values.get('document_type') in required_length_map.keys()
        ):
            required_length = required_length_map[values.get('document_type')]
            if v is None or len(v) != required_length:
                raise ValueError(f'The passport series must contain {required_length} digits')
        return v

    @validator('number', always=True)
    def field_number_in_passport(cls, v, values):
        if (
            values.get('document_type') == DocumentType.passport
            and values.get('citizenship') == Citizenship.RU
        ):
            if v is None or len(v) != 6:
                raise ValueError("The passport number must contain 6 digits")
        return v

    @validator('first_name', 'last_name', 'middle_name', always=True)
    def name_fields_format(cls, v, values):
        # doesn't validate empty names
        if not v:
            return v

        if (
            values.get('document_type') == DocumentType.passport
            and values.get('citizenship') == Citizenship.RU
        ):
            document_person_name_validate(AlphabetCode.cyrillic, v)
        else:
            document_person_name_validate(AlphabetCode.latin, v)

        return v


class BonusCardBase(BaseModel):
    bonus_card_id: int
    number: str
    service_provider_type: ServiceType
    service_provider_code: str
    name: Name

    @root_validator(pre=True)
    def arrange_names(cls, values):
        if not isinstance(values['name'], dict):
            values['name'] = {
                'ru': values.pop('name', ''),
                'en': values.pop('name_en', ''),
            }
        return values


class BonusCard(BonusCardBase):
    person_id: int


class ExtPersonBonusCard(BonusCardBase):
    ext_person_id: int


class BonusCardId(BaseModel):
    bonus_card_id: int


class BonusCardCreate(BaseModel):
    number: str
    service_provider_type: ServiceType
    service_provider_code: str


class ExtPersonDocumentIn(PersonDocumentCreate):
    pass


class ExtPersonBonusCardIn(BonusCardCreate):
    pass
