from datetime import date
import uuid

from pydantic import BaseModel, root_validator, validator

from intranet.trip.src.api.schemas.document import (
    ExtPersonBonusCard,
    ExtPersonBonusCardIn,
    ExtPersonDocumentIn,
    PersonDocument,
)
from intranet.trip.src.config import settings
from intranet.trip.src.enums import ExtPersonStatus, Gender
from intranet.trip.src.lib.translit.ru_icao import ICAORussianLanguagePack


class PersonBase(BaseModel):
    """
    Общая часть для Person и ExtPerson
    """
    first_name: str = None
    last_name: str = None
    middle_name: str = None
    provider_profile_id: int = None
    email: str = None
    date_of_birth: date = None
    phone_number: str = None
    gender: Gender = None


class FakePerson(BaseModel):
    uid: str
    person_id: int
    company_id: int
    first_name: str
    last_name: str
    middle_name: str
    provider_profile_id: int = None
    date_of_birth: date = None
    first_name_en: str = None
    last_name_en: str = None
    middle_name_en: str = None
    is_limited_access: bool = False
    is_dismissed: bool = False
    login: str = ''
    email: str = 'test@yandex.ru'
    gender: Gender = Gender.male
    phone_number: str = '+79999999999'
    external_uid: str = None
    is_active: bool = True

    @root_validator(pre=True)
    def clear_fake_info(cls, values):
        transliterator = ICAORussianLanguagePack()
        if not values.get('first_name_en'):
            values['first_name_en'] = transliterator.translit(values['first_name'], reversed=True)
        if not values.get('last_name_en'):
            values['last_name_en'] = transliterator.translit(values['last_name'], reversed=True)
        if not values.get('middle_name_en'):
            values['middle_name_en'] = transliterator.translit(values['middle_name'], reversed=True)
        values['external_uid'] = str(uuid.uuid4())
        return values


class Person(PersonBase):
    is_limited_access: bool
    is_dismissed: bool
    person_id: int
    uid: str
    login: str
    is_offline_trip: bool

    @root_validator(pre=True)
    def clear_value_if_dismissed(cls, values):
        if values.get('is_dismissed'):
            for field_name in (
                'first_name',
                'last_name',
                'date_of_birth',
                'phone_number',
                'gender',
            ):
                values[field_name] = None
        return values

    @validator('uid', always=True)
    def get_uid(cls, v, values):
        return str(values['person_id'])


def _clear_values_if_dismissed(values: dict) -> dict:
    if values['is_dismissed']:
        for field_name in (
            'first_name_ac_en',
            'last_name_ac_en',
            'middle_name_ac_en',
            'date_of_birth',
            'first_name',
            'last_name',
            'middle_name',
            'phone_number',
            'gender',
            'provider_profile_id',
        ):
            if field_name in values:
                values[field_name] = None
    return values


class PersonDetailsPublic(BaseModel):
    person_id: int
    uid: str
    login: str
    first_name: str = None
    last_name: str = None
    middle_name: str = None
    first_name_ac_en: str = None
    last_name_ac_en: str = None
    middle_name_ac_en: str = None
    gender: Gender = None
    is_dismissed: bool
    is_limited_access: bool
    is_offline_trip: bool

    has_date_of_birth: bool = False
    has_first_name_en: bool = False
    has_last_name_en: bool = False

    @root_validator(pre=True)
    def clear_value_if_dismissed(cls, values: dict) -> dict:
        return _clear_values_if_dismissed(values)

    @root_validator(pre=True)
    def set_flags(cls, values: dict) -> dict:
        values['has_date_of_birth'] = bool(values.get('date_of_birth'))
        values['has_first_name_en'] = bool(values.get('first_name_ac_en'))
        values['has_last_name_en'] = bool(values.get('last_name_ac_en'))
        return values

    @validator('uid', always=True)
    def get_uid(cls, v, values):
        return str(values['person_id'])


class PersonDetails(PersonDetailsPublic):
    date_of_birth: date = None
    email: str = None
    phone_number: str = None

    @root_validator(pre=True)
    def clear_value_if_dismissed(cls, values):
        return _clear_values_if_dismissed(values)


class PersonDetailsUpdate(BaseModel):
    first_name_ac_en: str = None
    last_name_ac_en: str = None
    middle_name_ac_en: str = None
    date_of_birth: date = None


class PersonCreate(BaseModel):
    email: str


def _get_secret_url(secret: str) -> str:
    return f'{settings.EXT_PERSON_SECRET_URL}?secret={secret}'


class ExtPersonId(BaseModel):
    ext_person_id: int
    secret: str
    secret_url: str = ''

    @validator('secret_url', always=True)
    def get_secret_url(cls, v, values):
        return _get_secret_url(values['secret'])


class ExtPerson(PersonBase):
    ext_person_id: int
    secret: str
    secret_url: str = ''
    name: str
    status: ExtPersonStatus
    documents: list[PersonDocument]
    bonus_cards: list[ExtPersonBonusCard]

    @validator('documents', 'bonus_cards', pre=True, always=True)
    def get_list_fields(cls, v):
        return v or []

    @validator('secret_url', always=True)
    def get_secret_url(cls, v, values):
        return _get_secret_url(values['secret'])


class ExtPersonCreate(BaseModel):
    name: str
    email: str


class ExtPersonUpdate(BaseModel):
    secret: str
    first_name: str
    last_name: str
    middle_name: str = None
    first_name_en: str
    last_name_en: str
    middle_name_en: str = None
    gender: Gender
    date_of_birth: date
    phone_number: str
    email: str
    documents: list[ExtPersonDocumentIn]
    bonus_cards: list[ExtPersonBonusCardIn]
