from datetime import date, datetime
from decimal import Decimal

from pydantic import BaseModel, Field

from intranet.trip.src.api.schemas.service import GeoPosition
from intranet.trip.src.enums import (
    ObjectType,
    SearchStatus,
    FilterValueType,
    ServiceType,
)
from intranet.trip.src.lib.aeroclub.enums import SearchMode


class SearchId(BaseModel):
    """ID поиска """
    search_id: str = Field(..., description='ID поиска у поставщика')


class BaseInfo(BaseModel):
    """Сущность, имеющая тип, ID и название """
    type: ObjectType = Field(..., description='Тип сущности')
    id: str = Field(..., description='ID сущности')
    name: str = Field(..., description='Название сущности')


class Loc(BaseModel):
    """Объект месторасположения """
    type: ObjectType = Field(..., description='Тип расположения')
    name: str = Field(..., description='Название расположения')


class BaseLocation(BaseModel):
    """Базовый класс расположения """
    country: Loc = Field(None, description='Страна')
    city: Loc = Field(None, description='Город')


class AviaLocation(BaseLocation):
    """Расположение авиа """
    airport: Loc = Field(None, description='Аэропорт')
    terminal: Loc = Field(None, description='Терминал')


class RailLocation(BaseLocation):
    """Расположение Ж/Д """
    train_station: Loc = Field(None, description='Железнодорожная станция')


class HotelLocation(BaseLocation):
    """Расположение отеля """


class FilterSelectValue(BaseModel):
    """Значение фильтра """
    target_id: str = Field(..., description='Значение')
    caption: str = Field(..., description='Заголовок')


class FilterItem(BaseModel):
    """Фильтр """
    name: str = Field(..., description='Название фильтра')
    type: FilterValueType = Field(..., description='Тип фильтра')
    values: list[FilterSelectValue] = Field(None, description='Возможные значения фильтра')


class ProviderSearchRequestStatus(BaseModel):
    """Статус поиска """
    status: SearchStatus = Field(..., description='Статус поиска')


class ProviderSearchResultCount(BaseModel):
    """Количество результатов поиска """
    service_type: ServiceType = Field(..., description='Тип услуги')
    count: int = Field(0, description='Количество результатов поиска')


class BaseSearchResult(BaseModel):
    """Базовый результат поиска """
    service_type: ServiceType = Field(..., description='Тип услуги')
    total: int = Field(..., description='Общее количество результатов поиска')
    page: int = Field(None, description='Номер страницы')
    limit: int = Field(None, description='Количество результатов поиска на странице')


class BaseSearchInfo(BaseModel):
    """Базовый класс информации о поиске """
    service_type: ServiceType = Field(..., description='Тип услуги')
    status: SearchStatus = Field(None, description='Статус поиска')


class BaseTransportSearchInfo(BaseSearchInfo):
    """Базовый класс информации о поиске по транспорту """
    location_from: BaseLocation = Field(..., description='Пункт отправления')
    location_to: BaseLocation = Field(..., description='Пункт прибытия')
    departure_on: date = Field(..., description='Дата отправления')


class AviaSearchInfo(BaseTransportSearchInfo):
    """Информация о поиске по авиа """
    service_type: ServiceType.avia = Field(ServiceType.avia, const=True)
    departure_back_on: date = Field(None, description='Дата отправления обратно')


class RailSearchInfo(BaseTransportSearchInfo):
    """Информация о поиске по Ж/Д """
    service_type: ServiceType.rail = Field(ServiceType.rail, const=True)


class HotelSearchInfo(BaseSearchInfo):
    """Информация о поиске по отелям """
    service_type: ServiceType.hotel = Field(default=ServiceType.hotel, const=True)
    check_in: date = Field(..., description='Дата заселения')
    check_out: date = Field(..., description='Дата выселения')
    location: HotelLocation = Field(..., description='Расположение отеля')


class BaseSearchTransportRequestIn(BaseModel):
    """Базовый поисковый запрос по транспорту """
    from_id: str = Field(..., description='ID из саджеста')
    to_id: str = Field(..., description='ID из саджеста')
    departure_on: date = Field(..., description='Дата и время отправления')


class SearchAviaRequestIn(BaseSearchTransportRequestIn):
    """Поисковый запрос по авиа """
    departure_back_on: date = Field(None, description='Дата и время отправления обратно')


class SearchRailRequestIn(BaseSearchTransportRequestIn):
    """Поисковый запрос по Ж/Д """


class SearchHotelRequestIn(BaseModel):
    """Поисковый запрос по отелям """
    target_id: str = Field(..., description='ID из саджеста')
    check_in_on: date = Field(..., description='Дата заселения')
    check_out_on: date = Field(..., description='Дата выселения')
    search_mode: list[SearchMode] = Field(None, description='Режим поиска (АК)')


class TransportBaggage(BaseModel):
    quantity: int = None
    weight: int = None
    dimensions: str = None


class FlightBaggage(BaseModel):
    baggage: TransportBaggage = None
    hand_baggage: TransportBaggage = None

    def __bool__(self):
        return self.baggage is not None or self.hand_baggage is not None


class ProviderSegment(BaseModel):
    """Сегмент перелёта """
    arrival: AviaLocation = Field(..., description='Локация прибытия')
    arrival_at: datetime = Field(..., description='Дата и время прибытия')
    arrival_at_utc: datetime = Field(..., description='Дата и время прибытия в utc')
    arrival_at_timezone_offset: int = Field(..., description='Часовой пояс прибытия')
    departure: AviaLocation = Field(..., description='Локация отправления')
    departure_at: datetime = Field(..., description='Дата и время отправления')
    departure_at_utc: datetime = Field(..., description='Дата и время отправления в utc')
    departure_at_timezone_offset: int = Field(..., description='Часовой пояс отправления')
    seats: int = Field(..., description='Количество мест')
    flight_number: str = Field(..., description='Номер рейса')
    flight_duration: int = Field(None, description='Время перелёта в минутах')
    transfer_duration: int = Field(None, description='Время пересадки в минутах')
    comment: str = Field(None, description='Комментарий')
    baggage: FlightBaggage = None
    flight_class: str = Field(..., description='Класс обслуживания')
    carrier: BaseInfo = Field(..., description='Авиакомпания')
    fare_code: str = Field(..., description='Вид тарифа')
    aircraft: BaseInfo = Field(None, description='Самолёт')


class ProviderLeg(BaseModel):
    """Перелёт """
    segments: list[ProviderSegment] = Field(
        ...,
        description=(
            'Список сегментов билета, '
            'В случае с АК, может содержать только 1 сегмент, '
            'так как нет информации о каждой пересадке'
        ),
    )
    segments_count: int = Field(
        ...,
        description='Количество пересадок. Необходимо, так как у АК нет информации о пересадках',
    )
    route_duration: int = Field(..., description='Время в пути в минутах')


class BaseFlight(BaseModel):
    price: Decimal
    is_refundable: bool = Field(
        None,
        description='Признак возможности возврата',
    )
    is_changeable: bool = Field(
        None,
        description='Признак возможности обмена',
    )
    is_travel_policy_compliant: bool


class Flight(BaseModel):
    """Рейс """
    id: str = Field(..., description='ID рейса')
    price: Decimal = Field(..., description='Стоимость билета')
    is_refundable: bool = Field(None, description='Признак возможности возврата')
    is_changeable: bool = Field(None, description='Признак возможности обмена')
    is_travel_policy_compliant: bool = Field(..., description='Флаг соответствия travel policy')
    legs: list[ProviderLeg] = Field(
        None,
        description=(
            'Список из одного рейса (в случае билета только "туда") '
            'или из двух (Если билет "Туда-обратно")'
        ),
    )


class AviaSearchResult(BaseSearchResult):
    """Результат поиска по авиа """
    service_type: ServiceType.avia = Field(default=ServiceType.avia, const=True)
    data: list[Flight] = Field(..., description='Список рейсов')


class AviaDetailResponse(BaseModel):
    service_type = ServiceType.avia
    flight: BaseFlight
    data: list[ProviderLeg]


class HotelBase(BaseModel):
    """Отель """
    hotel_name: str = Field(..., description='Название')
    stars: int = Field(None, description='Количество звёзд')
    address: str = Field(..., description='Адрес')
    geo_position: GeoPosition = Field(..., description='Координаты отеля')
    location: HotelLocation = Field(..., description='Расположение отеля')
    is_recommended: bool = Field(None, description='Флаг рекомендуемого поставщиком отеля')


class Hotel(HotelBase):
    id: str = Field(..., description='ID отеля')
    image_url: str = Field(None, description='Ссылка на изображение')
    description: str = Field(..., description='Описание')
    currency: str = Field(..., description='Валюта')
    min_price_per_night: Decimal = Field(..., description='Минимальная стоимость за одну ночь')


class HotelSearchResult(BaseSearchResult):
    """Результат поиска по отелям """
    service_type: ServiceType.hotel = Field(default=ServiceType.hotel, const=True)
    data: list[Hotel] = Field(..., description='Список отелей')


class Room(BaseModel):
    """Комната """
    index: int = Field(..., description='ID номера')
    images: list[str] = Field(..., description='Изображения номера')
    description: str = Field(None, description='Описание номера')
    name: str = Field(..., description='Название номера')
    is_meal_included: bool = Field(..., description='Флаг наличия питания')
    meal_names: list[str] = Field(..., description='Названия блюд')
    is_travel_policy_compliant: bool = Field(..., description='Флаг соответствия travel policy')
    currency: str = Field(..., description='Валюта оплаты')
    price_total: Decimal = Field(..., description='Стоимость за весь период')
    price_per_night: Decimal = Field(..., description='Стоимость за одну ночь')
    is_booking_by_request: bool = Field(
        False,
        description='Для бронирования нужно обратиться к агенту через чат.',
    )


class HotelDetail(HotelBase):
    """Детализация отеля """
    images: list[str] = Field(..., description='Изображения номера')
    check_in: date = Field(..., description='Дата заселения')
    check_out: date = Field(..., description='Дата выселения')
    num_of_nights: int = Field(..., description='Количество ночей')
    website: str = Field(None, description='Сайт отеля')


class HotelDetailResponse(BaseModel):
    """Ответ с детализацией отеля """
    service_type: ServiceType.hotel = Field(default=ServiceType.hotel, const=True)
    hotel: HotelDetail = Field(..., description='Общая информация об отеле')
    data: list[Room] = Field(..., description='Список номеров')


class TrainCarriage(BaseModel):
    """Вагон поезда """
    min_price: Decimal = Field(..., description='Минимальная стоимость места')
    max_price: Decimal = Field(None, description='Максимальная стоимость места (АЦ)')
    carriage_type: str = Field(..., description='Тип вагона')
    carriage_owner: str = Field(..., description='Наименование перевозчика')
    place_count: int = Field(..., description='Общее количество мест')
    is_travel_policy_compliant: bool = Field(..., description='Флаг соответствия travel policy')
    travel_policy_violations: list[str] = Field(
        default=None,
        description='Список нарушений travel policy',
    )


class BaseTrain(BaseModel):
    """Базовый класс поезда """
    train_name: str = Field(None, description='Название поезда')
    train_number: str = Field(None, description='Номер поезда')
    train_category: str = Field(None, description='Категория поезда (АЦ)')
    departure: RailLocation = Field(None, description='Локация отправления')
    departure_at: datetime = Field(..., description='Местные дата и время отправления')
    departure_at_utc: datetime = Field(..., description='Дата и время отправления в UTC')
    arrival: RailLocation = Field(None, description='Локация прибытия')
    arrival_at: datetime = Field(..., description='Местные дата и время прибытия')
    arrival_at_utc: datetime = Field(..., description='Дата и время прибытия в UTC')
    ride_duration: int = Field(None, description='Время в пути минутах')


class Train(BaseTrain):
    """Поезд """
    id: str = Field(..., description='ID поезда')
    carriages: list[TrainCarriage] = Field(..., description='Список вагонов')


class RailSearchResult(BaseSearchResult):
    """Результаты поиска по Ж/Д """
    service_type: ServiceType.rail = Field(default=ServiceType.rail, const=True)
    data: list[Train] = Field(..., description='Список поездов')


class TrainCarriagePlace(BaseModel):
    """Детализация одного места в вагоне """
    place_number: int = Field(..., description='Номер места')
    min_price: Decimal = Field(None, description='Минимальная стоимость места (АЦ)')
    max_price: Decimal = Field(None, description='Максимальная стоимость места (АЦ)')
    compartment_number: int = Field(None, description='Номер купе (отсека) (АЦ)')
    compartment_gender: str = Field(None, description='Половой признак купе (отсека)')
    place_type: str = Field(None, description='Тип места (АЦ)')
    place_type_description: str = Field(None, description='Описание типа места (АЦ)')


class TrainCarriageDetail(TrainCarriage):
    """Детализация одного вагона """
    has_electronic_registration: bool = Field(
        default=...,
        description='Признак наличия электронной регистрации',
    )
    carriage_number: int = Field(..., description='Номер вагона')
    service_class_code: str = Field(None, description='Класс обслуживания')
    service_class_description: str = Field(None, description='Описание класса обслуживания')
    services: list[str] = Field(None, description='Список доступных услуг')
    places: list[TrainCarriagePlace] = Field(..., description='Список мест')


class RailDetailResponse(BaseModel):
    """Ответ с детализацией поезда """
    service_type: ServiceType.rail = Field(default=ServiceType.rail, const=True)
    train: BaseTrain = Field(None, description='Общая информация о поезде')
    data: list[TrainCarriageDetail] = Field(..., description='Список вагонов')
