from datetime import time

from fastapi import Query
from pydantic.dataclasses import dataclass

from intranet.trip.src.enums import SearchOrdering
from intranet.trip.src.lib.aeroclub import enums


@dataclass
class BaseSearchFilter:
    order_by: SearchOrdering = Query(
        None,
        description='Сортировать по',
    )
    is_descending: bool = Query(
        None,
        description='Сортировать в обратном порядке',
    )


@dataclass
class AviaSearchFilter(BaseSearchFilter):
    """ Универсальная модель-схема для фильтров по поиску авиа """
    without_transfer: bool = Query(None, description='Без пересадок')
    maximum_transfers_count: int = Query(
        None,
        description='Максимальное количество пересадок',
    )
    has_baggage: bool = Query(None, description='Признак наличия багажа')
    is_restricted_by_travel_policy: bool = Query(
        None,
        description='Признак ограничения travel policy',
    )
    is_changeable: bool = Query(None, description='Признак возможности замены билета')
    is_refundable: bool = Query(None, description='Признак возможности возврата билета')
    # аэроклуб
    departure_there_timespan: list[enums.Time] = Query(None, description='Специфично для Аэроклуб')
    arrival_there_timespan: list[enums.Time] = Query(None, description='Специфично для Аэроклуб')
    departure_back_timespan: list[enums.Time] = Query(None, description='Специфично для Аэроклуб')
    arrival_back_timespan: list[enums.Time] = Query(None, description='Специфично для Аэроклуб')
    # авиацентр
    departure_time_from: time = Query(None, description='Специфично для Авиацентр')
    departure_time_to: time = Query(None, description='Специфично для Авиацентр')
    arrival_time_from: time = Query(None, description='Специфично для Авиацентр')
    arrival_time_to: time = Query(None, description='Специфично для АЦ')
    # аэроклуб filterparameters
    air_companies: list[str] = Query(None, description='Аэроклуб filterparams: аэрокомпании')
    cabin_classes: list[str] = Query(None, description='Аэроклуб filterparams: классы')
    departure_from_there: list[str] = Query(
        None,
        description='Аэроклуб filterparams: аэропорты отправления',
    )
    arrival_to_there: list[str] = Query(
        None,
        description='Аэроклуб filterparams: аэропорты прибытия',
    )
    departure_from_back: list[str] = Query(
        None,
        description='Аэроклуб filterparams: аэропорты обратного отправления',
    )
    arrival_to_back: list[str] = Query(
        None,
        description='Аэроклуб filterparams: аэропорты обратного прибытия',
    )


@dataclass
class HotelSearchFilter(BaseSearchFilter):
    """ Универсальная модель-схема для фильтров по поиску отелям """
    stars: list[str] = Query(None, description='Количество звёзд отеля')
    price_from: int = Query(None, description='Стоимость `от`')
    price_to: int = Query(None, description='Стоимость `до`')
    hotel_types: list[str] = Query(None, description='Типы отелей')
    is_restricted_by_travel_policy: bool = Query(
        None,
        description='Признак ограничения travel policy',
    )
    # aeroclub
    confirmation_types: list[enums.HotelConfirmationType] = Query(
        None,
        description='Аэроклуб: тип бронирования',
    )
    is_recommended: bool = Query(
        None,
        description='Аэроклуб: рекомендованный',
    )
    hotel_name: str = Query(
        None,
        description='Аэроклуб: имя отеля',
    )


@dataclass
class RailSearchFilter(BaseSearchFilter):
    """ Универсальная модель-схема для фильтров по поиску ЖД """
    is_restricted_by_travel_policy: bool = Query(
        None,
        description='Признак ограничения travel policy',
    )
    carriage_owners: list[str] = Query(None, description='Перевозчики (ФПК, Аэроэкспресс, ...)')
    carriage_types: list[str] = Query(None, description='Типы вагонов (Сидячий, Плацкарт, ...)')
    train_names: list[str] = Query(None, description='Названия поездов (Сапсан, Ласточка, ...)')
    train_categories: list[str] = Query(
        default=None,
        description='Типы поездов (Скорый, Пассажирский, ...)',
    )

    # Аэроклуб
    departure_there_timespan: list[enums.Time] = Query(
        default=None,
        description='Аэроклуб: период отправления',
    )
    arrival_there_timespan: list[enums.Time] = Query(
        default=None,
        description='Аэроклуб: период прибытия',
    )
    departure_from_there: list[str] = Query(
        default=None,
        description='Аэроклуб filterparams: id станций отправления',
    )
    arrival_to_there: list[str] = Query(
        default=None,
        description='Аэроклуб filterparams: id станций прибытия',
    )
    is_brand_train: bool = Query(None, description='Аэроклуб: признак фирменного поезда')
    has_electronic_registration: bool = Query(
        default=None,
        description='Аэроклуб: признак электронной регистрации',
    )

    # Авиацентр
    departure_time_from: time = Query(None, description='Авиацентр: дата отправления `от`')
    departure_time_to: time = Query(None, description='Авиацентр: дата отправления `до`')
    arrival_time_from: time = Query(None, description='Авиацентр: дата прибытия `от`')
    arrival_time_to: time = Query(None, description='Авиацентр: дата прибытия `до`')

    from_stations: list[int] = Query(None, description='Авиацентр: станции отправления')
    to_stations: list[int] = Query(None, description='Авиацентр: станции отправления')
