from datetime import date, datetime
from pydantic import BaseModel, validator, conlist, constr, root_validator, Field

from intranet.trip.src.enums import TripStatus, RelationType
from intranet.trip.src.api.schemas.common import RoutePoint, RoutePointCreate
from intranet.trip.src.api.schemas.person import Person
from intranet.trip.src.api.schemas.person_trip import PersonTripLite, PersonTripCreateInTrip
from intranet.trip.src.api.schemas.purpose import Purpose
from intranet.trip.src.lib.utils import unify_values


class ConfDetailsBase(BaseModel):
    conf_date_from: date
    conf_date_to: date
    conference_name: constr(max_length=256)
    price: constr(max_length=256)
    ticket_type: constr(max_length=256) = None
    promo_code: constr(max_length=256) = None
    is_another_city: bool = False
    conference_url: constr(max_length=128) = None
    program_url: constr(max_length=256) = None
    participation_terms: str = None
    cancellation_terms: str = None


class ConfDetails(ConfDetailsBase):
    trip_id: int
    tracker_issue: str = None


class ConfDetailsCreate(ConfDetailsBase):
    pass


class TripId(BaseModel):
    trip_id: int


class TripBase(BaseModel):
    date_from: date
    date_to: date


class Trip(TripBase):
    """Модель для списковой ручки"""
    trip_id: int
    staff_trip_uuid: str = None
    status: TripStatus
    created_at: datetime = None
    updated_at: datetime = None
    city_from: str = None
    city_to: str = None
    country_from: str = None
    country_to: str = None
    person_trips: list[PersonTripLite] = None
    purposes: list[Purpose]
    issue_travel: str = None
    issue_conf: str = None
    author: Person = None
    comment: str = None
    conf_details: ConfDetails = None
    route: list[RoutePoint] = None

    @validator('person_trips', 'purposes', pre=True, always=True)
    def get_list_fields(cls, v):
        return v or []


class TripUpdate(TripBase):
    """Форма редактирования командировки"""
    city_from: constr(max_length=128) = None
    city_to: constr(max_length=128) = None
    country_from: constr(max_length=128) = None
    country_to: constr(max_length=128) = None
    purposes: conlist(int, min_items=1, max_items=2)
    description: str
    with_days_off: bool
    conf_details: ConfDetailsCreate = None
    comment: constr(max_length=4096) = ''
    provider_city_from_id: int = None
    provider_city_to_id: int = None
    aeroclub_city_from_id: int = Field(None, deprecated=True)
    aeroclub_city_to_id: int = Field(None, deprecated=True)

    @validator('date_from')
    def date_from_not_earlier_than_today(cls, v):
        if v < date.today():
            raise ValueError('date_from must not be earlier than today')
        return v

    @validator('date_to')
    def date_to_later_than_date_from(cls, v, values, **kwargs):
        # если date_from отсутствует в values - упал валидатор date_from_not_earlier_than_today
        if 'date_from' not in values:
            return v
        if values['date_from'] > v:
            raise ValueError('date_from must not be later than date_to')
        return v

    @validator('provider_city_from_id', 'provider_city_to_id', always=True)
    def field_required_city_id(cls, v, values):
        if 'conf_details' not in values:
            return v
        if values['conf_details'] is None or values['conf_details'].is_another_city:
            if v is None:
                raise ValueError('field required')
        return v

    @root_validator(pre=True)
    def set_provider_cities(cls, values):
        values = unify_values(values, ['provider_city_from_id', 'aeroclub_city_from_id'])
        values = unify_values(values, ['provider_city_to_id', 'aeroclub_city_to_id'])
        return values


class TripCreate(TripUpdate):
    """Форма создания командировки"""
    person_trips: conlist(PersonTripCreateInTrip, min_items=1)
    route: conlist(RoutePointCreate, min_items=2) = None

    @root_validator(pre=True)
    def set_aeroclub_ids(cls, values):
        route = values.get('route')
        if route:
            first_city_in_route = (
                route[0].get('provider_city_id')
                or route[0].get('aeroclub_city_id')
            )
            last_city_in_route = (
                route[-1].get('provider_city_id')
                or route[-1].get('aeroclub_city_id')
            )
            values['provider_city_from_id'] = first_city_in_route
            values['provider_city_to_id'] = last_city_in_route
            values['aeroclub_city_from_id'] = first_city_in_route
            values['aeroclub_city_to_id'] = last_city_in_route
        return values


class TripFilter(BaseModel):
    """Фильтр командировок в списковой ручке"""
    offset: int = None
    limit: int = None
    date_from__lte: date = None
    date_from__gte: date = None
    provider_city_to_id: str = None
    tracker_issue: str = None
    manager_id: int = None
    person_id: int = None
    relation_type: RelationType = None
    holding_id: int = None
