import re
from enum import Enum

CYRILLIC_NAME_PATTERN = re.compile(r'^[а-яА-ЯёЁ\s-]+$')
LATIN_NAME_PATTERN = re.compile(r'^[a-zA-Z\s-]+$')

LEADING_TRAILING_RESTRICTIONS = ('.', '-')


class AlphabetCode(str, Enum):
    cyrillic = 'cyrillic'
    latin = 'latin'


def _is_cyrillic_name(name: str):
    return CYRILLIC_NAME_PATTERN.match(name) is not None


def _is_latin_name(name: str):
    return LATIN_NAME_PATTERN.match(name) is not None


def document_person_name_validate(code: AlphabetCode, name: str):
    if code == AlphabetCode.cyrillic and not _is_cyrillic_name(name):
        raise ValueError('name must contain only cyrillic characters, space or hyphen')
    elif code == AlphabetCode.latin and not _is_latin_name(name):
        raise ValueError('name must contain only latin characters, space or hyphen')

    # 2.7.6 Запрещён ввод дефиса и точки в начале и конце ФИО в полях «фамилия», «имя», «отчество»
    if (
        name.startswith(LEADING_TRAILING_RESTRICTIONS)
        or name.endswith(LEADING_TRAILING_RESTRICTIONS)
    ):
        raise ValueError('name must start with alpha character')

    return True
