import base64

from pydantic import BaseSettings


class Settings(BaseSettings):
    ENV_TYPE: str = 'default'
    IS_YA_TEAM: bool = True
    DEBUG: bool = True

    SECRET_KEY: str = 'some-very-secret-key'

    TRIP_ROBOT_LOGIN = 'robot-trip'

    POSTGRES_DATABASE: str = 'trip'
    POSTGRES_USER: str = 'trip'
    POSTGRES_PASSWORD: str = 'trip'
    POSTGRES_HOST: str = 'localhost'
    POSTGRES_PORT: int = 5432
    POSTGRES_POOL_MIN: int = 10
    POSTGRES_POOL_MAX: int = 10
    POSTGRES_ACQUIRE_TIMEOUT: int = 1
    POSTGRES_REFRESH_TIMEOUT: int = 5
    POSTGRES_COLLATION: str = 'ru_RU.utf8'

    @property
    def db_pool_config(self):
        return {
            'minsize': self.POSTGRES_POOL_MIN,
            'maxsize': self.POSTGRES_POOL_MAX,
        }

    @property
    def db_settings(self):
        return {
            'database': self.POSTGRES_DATABASE,
            'user': self.POSTGRES_USER,
            'password': self.POSTGRES_PASSWORD,
            'host': self.POSTGRES_HOST,
            'port': self.POSTGRES_PORT,
        }

    @property
    def database_url(self):
        return 'postgresql://{user}:{password}@{host}:{port}/{database}'.format(
            **self.db_settings,
            **self.db_pool_config,
        )

    @property
    def multihost_db_url(self):
        hosts = ','.join(
            f'{host}:{self.db_settings["port"]}'
            for host in self.db_settings['host'].split(',')
        )
        return 'postgresql://{user}:{password}@{hosts}/{database}'.format(
            hosts=hosts,
            **self.db_settings,
        )

    REDIS_HOST: str = 'localhost'
    REDIS_PORT: int = 6379
    REDIS_PASSWORD: str = 'trip'
    REDIS_SENTINEL_MASTER: str = 'trip-test-redis'
    REDIS_CACHE_EXPIRE_AFTER: int = 3600
    REDIS_LOCK_WAIT_TIMEOUT: int = 30  # seconds
    REDIS_LOCK_TIMEOUT: int = 300  # seconds

    @property
    def REDIS_QUEUE_NAME(self):
        return f'arq:{self.DEPLOY_STAGE_ID}_queue'

    TVM_CLIENT_ID: int = 2020751
    TVM_PASSPORT_ID: int = 223
    TVM_PASSPORT_CONTACTS_ID: int = 2029754
    TVM_STAFF_ID: int = 2000053
    TVM_STAFF_API_ID: int = 2001976
    TVM_INTRASEARCH_ID: int = 2001037
    TVM_MESSENGER_ID: int = 2001664
    TVM_BOT_ID: int = 2001666
    TVM_IDM_ID: int = 2001600
    TVM_TRAVEL_AVIA_ID: int = 2000569
    TVM_TRAVEL_TRAIN_ID: int = 2002852
    TVM_DRIVE_ID: int = 2010062
    TVM_OK_ID: int = 2009215
    TVM_LOGBROKER_ID: int = 2001059
    TVM_ZORA_ID: int = 2023123
    TVM_MDS_ID: int = 2017579

    @property
    def tvm_services(self):
        return {
            'trip': self.TVM_CLIENT_ID,
            'passport': self.TVM_PASSPORT_ID,
            'passport-contacts': self.TVM_PASSPORT_CONTACTS_ID,
            'staff': self.TVM_STAFF_ID,
            'staff-api': self.TVM_STAFF_API_ID,
            'intrasearch': self.TVM_INTRASEARCH_ID,
            'messenger': self.TVM_MESSENGER_ID,
            'bot': self.TVM_BOT_ID,
            'idm': self.TVM_IDM_ID,
            'travel_avia': self.TVM_TRAVEL_AVIA_ID,
            'travel_train': self.TVM_TRAVEL_TRAIN_ID,
            'drive': self.TVM_DRIVE_ID,
            'ok': self.TVM_OK_ID,
            'logbroker': self.TVM_LOGBROKER_ID,
            'zora': self.TVM_ZORA_ID,
            'mds': self.TVM_MDS_ID,
        }

    @property
    def allowed_clients(self):
        return [self.tvm_services[client] for client in ('trip', 'bot', 'idm', 'mds')]

    TRIP_HOST: str = 'trip.test.yandex-team.ru'
    OPENAPI_URL: str = '/api/openapi.json'
    UNISTAT_URL: str = '/api/monitoring/unistat'
    EXTERNAL_UNISTAT_URL: str = '/external/monitoring/unistat'
    TOKEN_URL: str = '/api/csrf_token/'
    CREATE_PERSON_URL: str = '/api/persons/create'
    CONFIRM_EMAIL_URL: str = '/api/persons/confirm_email'
    SET_COMPANY_URL: str = '/api/persons/set_company'

    @property
    def trip_base_url(self):
        return f'https://{self.TRIP_HOST}'

    def person_trip_url(self, trip_id: int, person_id: int):
        return f'https://{self.TRIP_HOST}/trips/{trip_id}/person/{person_id}'

    STAFF_HOST: str = 'staff.test.yandex-team.ru'
    STAFF_GET_ASSIGNMENT_TIMEOUTS: int = 8
    STAFF_GET_ASSIGNMENT_RETRIES: int = 3
    STAFF_GET_ASSIGNMENT_WAIT_MULTIPLIER: int = 2

    @property
    def staff_url(self):
        return f'https://{self.STAFF_HOST}/'

    STAFF_API_HOST: str = 'staff-api.test.yandex-team.ru'

    @property
    def staff_api_url(self):
        return f'https://{self.STAFF_API_HOST}/v3/'

    DOCUMENT_API_HOST: str = ''  # FIXME: put passport API document path

    @property
    def document_api_url(self):
        return f'https://{self.DOCUMENT_API_HOST}/'  # FIXME: set path if it need

    PASSPORT_CONTACTS_API_HOST: str = 'address-test.pers.yandex.net'

    @property
    def passport_contacts_api_url(self):
        return f'https://{self.PASSPORT_CONTACTS_API_HOST}'

    SEARCH_HOST: str = 'search-back.test.yandex-team.ru'

    @property
    def search_url(self):
        return f'https://{self.SEARCH_HOST}'

    TRACKER_API_HOST: str = 'st-api.test.yandex-team.ru'
    TRACKER_ROBOT_OAUTH: str = ''

    @property
    def tracker_api_url(self):
        return f'https://{self.TRACKER_API_HOST}/v2/'

    MESSENGER_HOST: str = 'messenger-internal.alpha.yandex.net'
    MESSENGER_GUID_ROBOT_TRIP: str = 'f0100900-aedd-967b-6849-6a83a755cc33'
    BOT_MESSENGER_HOST: str = 'alpha.bp.mssngr.yandex.net'
    MESSENGER_ROBOT_OAUTH: str = ''
    MESSENGER_CHAT_ORG_ID = 34
    MESSENGER_UI_HOST: str = 'messenger.alpha.yandex-team.ru'

    EXT_PERSON_SECRET_URL: str = 'https://trip.yandex-team.ru/ext_persons/'

    @property
    def messenger_url(self):
        return f'https://{self.MESSENGER_HOST}/meta_api/'

    @property
    def bot_messenger_url(self):
        return f'https://{self.BOT_MESSENGER_HOST}/'

    HUB_API_LOGIN: str = ''
    HUB_API_PASSWORD: str = ''

    @property
    def hub_api_token(self):
        s = f'{self.HUB_API_LOGIN}:{self.HUB_API_PASSWORD}'
        b = s.encode("utf-8")
        e = base64.b64encode(b)
        return e.decode("utf-8")

    TRAVEL_AVIA_HOST: str = 'api-gateway.production.avia.yandex.net'

    @property
    def travel_avia_url(self):
        return f'http://{self.TRAVEL_AVIA_HOST}/'

    TRAVEL_TRAIN_HOST: str = 'testing.train-api.rasp.internal.yandex.net'

    @property
    def travel_train_url(self):
        return f'https://{self.TRAVEL_TRAIN_HOST}/'

    DRIVE_API_HOST: str = 'testing.carsharing.yandex.net'
    DRIVE_ORG_PREFIX = 'trip_org_'

    @property
    def drive_api_url(self):
        return f'https://{self.DRIVE_API_HOST}/api/trip/'

    AEROCLUB_API_TOKEN: str = ''
    AEROCLUB_TIMEOUT: int = 300
    AEROCLUB_API_URL: str = 'https://beta-api.aeroclub.ru/'
    HUB_API_URL: str = 'https://beta-integration.aeroclub.ru/hub/'

    AEROCLUB_FS_URL: str = 'https://fs.aeroclub.ru/'

    AEROCLUB_PROFILE_ID = 734379  # Test User
    AEROCLUB_COMPANY_ID = 42436  # Yandex
    AEROCLUB_PAYMENT_TYPE_ID = 2  # 1 - Cash, 2 - Invoice

    # aeroclub sftp options
    AEROCLUB_REGISTRY_HOST: str = 'ftp.aeroclub.ru'
    AEROCLUB_REGISTRY_LOGIN: str = 'yandexfindoc'
    AEROCLUB_REGISTRY_PASSWORD: str = 'password'

    # для удобства тестирования
    AEROCLUB_MOSCOW_ID = 16381
    AEROCLUB_SPB_ID = 15613

    AVIACENTER_API_TOKEN: str = ''
    AVIACENTER_TIMEOUT: int = 60
    AVIACENTER_API_URL: str = 'https://s1-api-colibri.crpo.su/'

    AVIACENTER_COMPANY_ID: int = 2956
    AVIACENTER_TRAVEL_POLICY_ID: int = 295

    ENABLE_STAFF_SYNC: bool = False
    ENABLE_AEROCLUB_SYNC: bool = False  # enable sync with aeroclub
    ENABLE_AEROCLUB_EVENT_HANDLING: bool = False
    ENABLE_AEROCLUB_API_DOCUMENTS_SYNC: bool = False
    ENABLE_EXECUTED_PERSON_TRIPS_SYNC: bool = True
    ENABLE_SERVICE_WORKFLOW: bool = True  # allow processing services in person trip
    ENABLE_CHATS: bool = True

    DEPLOY_STAGE_ID: str = 'stage'

    SKIP_TRIP_APPROVE: bool = False
    DAYS_BEFORE_CLOSING_TRIPS: int = 3

    AEROCLUB_LIMIT_MESSAGES = 200

    # blackbox attributes: https://docs.yandex-team.ru/authdevguide/concepts/DB_About#db-attributes
    BB_ATTRIBUTE_TIMEZONE = '33'
    BB_ATTRIBUTE_LANGUAGE = '34'
    BB_ATTRIBUTE_FIRSTNAME = '27'
    BB_ATTRIBUTE_LASTNAME = '28'

    @property
    def blackbox_attributes(self):
        attributes = [self.BB_ATTRIBUTE_TIMEZONE, self.BB_ATTRIBUTE_LANGUAGE]
        if not self.IS_YA_TEAM:
            attributes += [self.BB_ATTRIBUTE_FIRSTNAME, self.BB_ATTRIBUTE_LASTNAME]
        return ','.join(attributes)

    CONF_MANAGER_LOGIN: str = ''

    ARQ_SERVICES_EXECUTION_JOB_TIMEOUT = 300  # seconds
    ARQ_SERVICES_EXECUTION_RUN_INTERVAL = 5  # minutes

    MESSENGER_GUID_CACHE_EXPIRE_AFTER: int = 3600 * 24  # 1 day

    TRIP_GROUP_EMAIL = ''

    OK_HOST: str = 'ok.test.yandex-team.ru'
    OK_ROBOT_OAUTH: str = ''

    @property
    def ok_url(self):
        return f'https://{self.OK_HOST}/'

    AUTOMATIC_APPROVAL_PURPOSES = (
        3,  # Деловые переговоры с внешними партнерами
        8,  # Логистика
        9,  # Поддержка ИТ-инфраструктуры
        11,  # Инженерно-монтажные работы
        16,  # Организация конференции, семинара или аналогичного мероприятия
        17,  # Рабочая встреча с коллегами
        18,  # Производство, посещение объектов, приемка работ
    )

    MEETING_WITH_COLLEAGS_PURPOSE_ID: int = 17

    ENABLE_AUTOMATIC_APPROVAL: bool = False

    ENABLE_ERROR_BOOSTER: bool = False
    ERROR_BOOSTER_PROJECT_ID: str = ''
    LOGBROKER_ERROR_BOOSTER_TOPIC: str = ''

    ENABLE_PYSCOPG2: bool = True
    ENABLE_CSRF_PROTECTION: bool = True

    BLACKBOX_URL = 'http://blackbox.yandex-team.ru/blackbox'

    YT_PROXY: str = 'hahn'
    YT_PREFIX: str = '//home/oebs-dev/development/travel_expenses/travel_registers/'
    YT_ENABLE_SYNC: bool = False
    YT_TOKEN: str = ''

    YANDEX_HOLDING_ID: int = 1

    ENABLE_ZORA: bool = True
    ZORA_CLIENT_ID: str = 'trip_testing'

    # MDS Section
    MDS_S3_ENDPOINT_URL: str = 'http://s3.mds.yandex.net'
    MDS_S3_BUCKET_NAME: str = 'trip-back-ya-team-test'

    LOG_TRANSACTION_ID = False

    # B2B Finance
    YANDEX_FEE = 0
