import logging

from sqlalchemy import and_, not_, or_, func, select
from intranet.trip.src.enums import EmployeeStatus
from intranet.trip.src.db.gateways.base import DBGateway
from intranet.trip.src.db.tables import person_table
from intranet.trip.src.models.employee import Employee, EmployeeFilter

logger = logging.getLogger(__name__)


class EmployeeGateway(DBGateway):

    table = person_table
    pk = 'person_id'
    model_class = Employee

    async def get_employee(self, company_id: int, employee_id: int) -> Employee:
        return await self._get_one(
            select_list=[
                self.table.c.person_id.label('employee_id'),
                self.table.c.is_active,
                self.table.c.is_dismissed,
                self.table.c.is_coordinator,
                self.table.c.company_id,
                self.table.c.rejected_at,
                # these fields are empty on b2b
                self.table.c.email,
                self.table.c.first_name,
                self.table.c.last_name,
                self.table.c.middle_name,
            ],
            where_clause=and_(
                person_table.c.person_id == employee_id,
                person_table.c.company_id == company_id,
            )
        )

    async def get_list_and_count(
        self,
        company_id: int,
        employee_filter: EmployeeFilter,
        limit: int = None,
        offset: int = None,
    ) -> tuple[list[Employee], int]:
        query = (
            select([
                self.table.c.person_id.label('employee_id'),
                self.table.c.is_active,
                self.table.c.is_dismissed,
                self.table.c.is_coordinator,
                self.table.c.company_id,
                self.table.c.rejected_at,
                # these fields are empty on b2b
                self.table.c.email,
                self.table.c.first_name,
                self.table.c.last_name,
                self.table.c.middle_name,
            ])
            .where(self.table.c.company_id == company_id)
            .order_by('person_id')
        )
        # status
        where_status = []
        for status in employee_filter.statuses or []:
            if status == EmployeeStatus.activated:
                where_status.append(self.table.c.is_active)
            elif status == EmployeeStatus.rejected:
                where_status.append(self.table.c.rejected_at.is_not(None))
            elif status == EmployeeStatus.blocked:
                where_status.append(self.table.c.is_dismissed)
            elif status == EmployeeStatus.wait_activation:
                where_status.append(
                    and_(
                        not_(self.table.c.is_active),
                        not_(self.table.c.is_dismissed),
                        self.table.c.rejected_at.is_(None),
                    )
                )
        if len(where_status) > 0:
            query = query.where(or_(*where_status))
        # apply filters
        is_coordinator = getattr(employee_filter, 'is_coordinator')
        if is_coordinator is not None:
            query = query.where(self.table.c.is_coordinator == is_coordinator)

        count_query = select([func.count()]).select_from(query.subquery())
        count = await self.conn.scalar(count_query)

        if limit is not None:
            query = query.limit(limit)
        if offset is not None:
            query = query.offset(offset)

        rows = await self._fetchall(query)
        return [Employee(**item) for item in rows], count
