# Этот файл не должен импортиться и выполняться.
# Тут код для миграции паспортов со Стаффа.
import re
import logging

from aiopg.sa.connection import SAConnection
from sqlalchemy import select

from intranet.trip.src.db import get_db_engine
from intranet.trip.src.db.tables import staff_passport_table
from intranet.trip.src.db.gateways.base import DBGateway
from intranet.trip.src.enums import Citizenship
from intranet.trip.src.db.gateways.document import (
    PersonDocumentGateway,
    Document,
    DocumentType,
)
from intranet.trip.src.db.gateways.person import (
    PersonGateway,
)
from intranet.trip.src.redis import create_redis_pool

document_types_map = {
    'internal': DocumentType.passport,
    'foreign': DocumentType.external_passport,
}


trash_pattern = re.compile(r'[!"№;%:?\*@#\$%\^,]')


def sanitize(series_number: str) -> str:
    """очищаем строку с серией/номером от всего мусора и делаем upper"""
    series_number = series_number.upper()

    for pattern in ['СЕРИЯ', 'НОМЕР', 'NO']:
        if pattern in series_number:
            series_number = series_number.replace(pattern, '')

    series_number = re.sub(trash_pattern, '', series_number or '')

    return series_number.strip()


def get_passport_key(series: str, number: str) -> str:
    """
    Генерит ключ вида `серия номер`, считаем его уникальным,
    по нему сопоставляем паспорта со стаффовскими данными
    """
    series = sanitize(series or '')
    number = sanitize(number or '')
    return f'{series} {number}'


class StaffDocumentGateway(DBGateway):

    def __init__(self, staff_connection: SAConnection):
        self.conn = staff_connection

    async def _fetch_all_staff_passports(self) -> list:
        query = (
            select([staff_passport_table])
            .where(
                staff_passport_table.c.doc_type.in_(['internal', 'foreign']),
            )
            .order_by(
                staff_passport_table.c.person_id,
                staff_passport_table.c.number,
            )
        )

        result = await self.conn.execute(query)
        return await result.fetchall()

    async def get_all_staff_passports_data(self) -> dict[str, dict]:
        staff_passports = {}
        for passport in await self._fetch_all_staff_passports():
            series, number = craft_fields_from_string(
                series_number=passport['number'],
                passport_type=passport['doc_type'],
                country=passport['issue_country'],
            )
            key = get_passport_key(series, number)
            staff_passports[key] = dict(passport)

        return staff_passports


def craft_fields_from_string(
    series_number: str, passport_type: str = None, country: str = None,
) -> tuple[str, str]:
    """
    пытаемся распарсить строку на серию и номер
    """
    series_number = series_number.strip()
    series_number = sanitize(series_number)
    if citizenship_map.get(country) == Citizenship.RU:
        if passport_type == 'internal':
            # Правила парсинга паспорта РФ: первые 4 символа — серия. Остальные 6 — номер
            series_number = re.sub(' ', '', series_number.strip().lower())
            return series_number[:4], series_number[4:]

        if passport_type == 'foreign':
            # Правила парсинга загранки РФ: первые 2 символа — серия. Остальные — номер
            series_number = re.sub(' ', '', series_number.strip().lower())
            return series_number[:2], series_number[2:]

    # Дефолтное поведение: серия — это всё до первго пробела
    if ' ' in series_number:
        series, number = series_number.split(' ', 1)
        # assert len(series) <= 8, (series, number)
        # assert len(number) <= 16, (series, number)
        return series, number

    # либо первые два символа, если нет пробела. Остальное — номер
    return series_number[:2], series_number[2:]


async def get_staff_gateway():
    import os
    from aiopg.sa import create_engine
    staff_test_db_settings = {
        'database': 'staff4',
        'user': 'tripmigrator',
        'password': '',
        'host': os.getenv('STAFF_DB_HOST'),
        'port': 6432,
        'minsize': 2,
        'maxsize': 2,
    }
    staff_test_db_settings['password'] = os.getenv('STAFF_DB_PASSWORD')

    staff_db_engine = await create_engine(**staff_test_db_settings)
    staff_connection = await staff_db_engine.acquire()

    return StaffDocumentGateway(staff_connection)


def get_what_to_do(
        trip_passports: dict[str, Document],
        staff_passports: dict[str, dict],
        person_ids: list[int],
) -> tuple[list, list, list]:
    create, update, delete = [], [], []

    for key in set(trip_passports) & set(staff_passports):
        if not staff_passports[key]['is_active']:
            delete.append(key)
            continue

        if differs(trip_passports[key], staff_passports[key]):
            update.append(key)

    for key in set(staff_passports) - set(trip_passports):
        if staff_passports[key]['is_active'] and staff_passports[key]['person_id'] in person_ids:
            create.append(key)

    logging.info('create %s, update %s, delete %s', len(create), len(update), len(delete))
    return create, update, delete


async def sync_passports():
    engine = await get_db_engine()
    async with engine.acquire() as conn:
        redis = await create_redis_pool()
        trip_persons_gateway = PersonGateway(conn)
        trip_documents_gateway = PersonDocumentGateway(conn)
        staff_documents_gateway = await get_staff_gateway()

        trip_person_ids: list[int] = await trip_persons_gateway.get_all_ids()
        trip_passports = {
            get_passport_key(doc.series, doc.number): doc
            for doc in await trip_documents_gateway.get_all_passports_for_staff_sync()
        }
        staff_passports: dict = await staff_documents_gateway.get_all_staff_passports_data()

        create, update, delete = get_what_to_do(trip_passports, staff_passports, trip_person_ids)
        print(f'CREATE: {create}\n\nUPDATE: {update}\n\nDELETE: {delete}\n')

        person_ids_to_push = []

        for key in create:
            passport_data = staff_passports[key]
            person_id = passport_data['person_id']
            series, number = key.split(' ', 1)
            passport_data = {
                'document_type': document_types_map[passport_data['doc_type']],
                'series': series,
                'number': number,
                'issued_by': passport_data['issued_by'],
                'issued_on': passport_data['issue_date'],
                'expires_on': passport_data['expire_date'],
                'citizenship': citizenship_map.get(
                    passport_data['issue_country'], Citizenship.OTHER,
                ),
                'first_name': passport_data['first_name'],
                'last_name': passport_data['last_name'],
                'middle_name': passport_data['middle_name'],
                'description': passport_data['description'],
            }
            if (
                passport_data['document_type'] == DocumentType.external_passport
                and passport_data['expires_on'] is None
            ):
                continue
            try:
                await trip_documents_gateway.create(person_id=person_id, **passport_data)
                person_ids_to_push.append(person_id)
            except Exception as ex:
                print(f'ERROR creating {passport_data} [{str(ex)}]')

        for key in update:
            passport_data = staff_passports[key]
            trip_passport = trip_passports[key]
            series, number = key.split(' ', 1)
            person_id = passport_data['person_id']
            passport_data = {
                'document_type': document_types_map[passport_data['doc_type']],
                'issued_by': passport_data['issued_by'],
                'issued_on': passport_data['issue_date'],
                'expires_on': passport_data['expire_date'],
                'citizenship': citizenship_map.get(
                    passport_data['issue_country'], Citizenship.OTHER,
                ),
                'first_name': passport_data['first_name'],
                'last_name': passport_data['last_name'],
                'middle_name': passport_data['middle_name'],
                'description': passport_data['description'],
            }
            if (
                passport_data['document_type'] == DocumentType.external_passport
                and passport_data['expires_on'] is None
            ):
                continue
            try:
                await trip_documents_gateway.update(trip_passport.document_id, **passport_data)
                person_ids_to_push.append(person_id)
            except Exception as ex:
                print(f'ERROR updating {trip_passport.document_id}: {passport_data} [{str(ex)}]')

        for key in delete:
            trip_passport = trip_passports[key]
            await trip_documents_gateway.delete(
                person_id=trip_passport.person_id,
                document_id=trip_passport.document_id,
            )
            person_ids_to_push.append(trip_passport.person_id)

    person_ids_to_push = await trip_persons_gateway.get_person_ids_for_hub_sync(person_ids_to_push)
    print('Person_ids to push to ihub', person_ids_to_push)

    await redis.enqueue_job(
        'sync_profiles_to_ihub_task',
        person_ids=person_ids_to_push,
        unique=False,
    )


def differs(trip_passport: Document, staff_passport: dict) -> bool:
    """
    Ввозвращает True если надо обновить паспорт в trip'е.
    """
    return (
        trip_passport.document_type != document_types_map[staff_passport['doc_type']]
        or trip_passport.first_name != staff_passport['first_name']
        or trip_passport.last_name != staff_passport['last_name']
        or trip_passport.middle_name != staff_passport['middle_name']
        or trip_passport.issued_on != staff_passport['issue_date']
        or trip_passport.expires_on != staff_passport['expire_date']
        or trip_passport.issued_by != staff_passport['issued_by']
        or trip_passport.description != staff_passport['description']
        or trip_passport.citizenship != citizenship_map.get(
            staff_passport['issue_country'],
            Citizenship.OTHER,
        )
    )


citizenship_map = {
    # russia
    'rf': Citizenship.RU,
    'rossia': Citizenship.RU,
    'rus': Citizenship.RU,
    'russain federation': Citizenship.RU,
    'russia': Citizenship.RU,
    'russia federation': Citizenship.RU,
    'russian': Citizenship.RU,
    'russian fedaration': Citizenship.RU,
    'russian federation': Citizenship.RU,
    'russian federtion': Citizenship.RU,
    'russias federation': Citizenship.RU,
    'ussr': Citizenship.RU,
    'россии': Citizenship.RU,
    'российская федерациция': Citizenship.RU,
    'российская федерация': Citizenship.RU,
    'российская федерация / russian federation': Citizenship.RU,
    'российская федерация / russioan federation': Citizenship.RU,
    'российская федерация/russian federation': Citizenship.RU,
    'российскя федерация': Citizenship.RU,
    'россиская федерация': Citizenship.RU,
    'россия': Citizenship.RU,
    'Россия': Citizenship.RU,
    'росссия': Citizenship.RU,
    'рф': Citizenship.RU,
    'санкт-петербург': Citizenship.RU,
    'ссср': Citizenship.RU,
    '016263': Citizenship.RU,
    '025369': Citizenship.RU,
    '084740': Citizenship.RU,
    '163851': Citizenship.RU,
    '297099': Citizenship.RU,
    '311533': Citizenship.RU,
    '312409': Citizenship.RU,
    '315580': Citizenship.RU,
    '349084': Citizenship.RU,
    '354604': Citizenship.RU,
    '357565': Citizenship.RU,
    '370040': Citizenship.RU,
    '429940': Citizenship.RU,
    '433941': Citizenship.RU,
    '498434': Citizenship.RU,
    '626286': Citizenship.RU,
    '644076, омская обл., г омск, ул. 75 гвардейской бригады, д 1в, кв 139': Citizenship.RU,
    '699995': Citizenship.RU,
    '727925': Citizenship.RU,
    '821469': Citizenship.RU,
    '839216': Citizenship.RU,
    '883416': Citizenship.RU,
    '988050': Citizenship.RU,
    # other
    'belarus': Citizenship.BY,
    'republic of belarus': Citizenship.BY,
    'беларусь': Citizenship.BY,
    'рб': Citizenship.BY,
    'республика беларусь': Citizenship.BY,

    'ukraine': Citizenship.UA,
    'ukranie': Citizenship.UA,

    'kazakhstan': Citizenship.KZ,
    'казахстан': Citizenship.KZ,
    'республика казахстан': Citizenship.KZ,

    'kyrgyz republic': Citizenship.KG,
    'киргизия': Citizenship.KG,
    'кыргызская республика': Citizenship.KG,
    'кыргызстан': Citizenship.KG,

    'azerbaijan': Citizenship.OTHER,
    'hungary': Citizenship.OTHER,
    'israel': Citizenship.OTHER,
    'израиль': Citizenship.OTHER,
    'italy': Citizenship.OTHER,
    'nigeria': Citizenship.OTHER,
    'republic of korea': Citizenship.OTHER,
    'republic of turkey': Citizenship.OTHER,
    'uzbekistan': Citizenship.OTHER,
    'босния и герцеговина': Citizenship.OTHER,
    'вв': Citizenship.OTHER,
}
