from sqlalchemy import select, and_, or_

from intranet.trip.src.db.utils import json_column, json_agg_column
from intranet.trip.src.db.tables import (
    purpose_table,
    service_table,
    trip_table,
    trip_purpose_table,
    conf_details_table,
    person_trip_table,
    person_document_table,
    person_trip_purpose_table,
    person_conf_details_table,
    travel_details_table,
    person_trip_document_table,
    trip_route_point_table,
    person_trip_route_point_table,
)
from intranet.trip.src.db.subqueries.common import get_person_column, get_person_with_company_column
from intranet.trip.src.enums import DocumentType


def get_documents_full_column():
    """
    Только паспорта
    """
    document_types = [
        DocumentType.passport.name,
        DocumentType.external_passport.name,
    ]
    subquery = (
        select([person_document_table])
        .select_from(
            person_trip_document_table.join(
                person_document_table,
                person_trip_document_table.c.document_id == person_document_table.c.document_id,
            )
        )
        .where(person_trip_table.c.trip_id == person_trip_document_table.c.trip_id)
        .where(person_trip_table.c.person_id == person_trip_document_table.c.person_id)
        .where(person_document_table.c.document_type.in_(document_types))
        .where(person_document_table.c.is_deleted.is_(False))
        .correlate(person_trip_table)
    )
    return json_agg_column('documents', subquery)


def get_person_trip_purposes_column():
    subquery = (
        select([
            purpose_table,
        ])
        .select_from(
            person_trip_purpose_table.join(
                purpose_table,
                person_trip_purpose_table.c.purpose_id == purpose_table.c.purpose_id,
            )
        )
        .where(person_trip_table.c.trip_id == person_trip_purpose_table.c.trip_id)
        .where(person_trip_table.c.person_id == person_trip_purpose_table.c.person_id)
        .correlate(person_trip_table)
    )
    return json_agg_column('purposes', subquery)


def get_travel_details_column():
    subquery = (
        select([
            travel_details_table,
        ])
        .where(person_trip_table.c.trip_id == travel_details_table.c.trip_id)
        .where(person_trip_table.c.person_id == travel_details_table.c.person_id)
        .correlate(person_trip_table)
    )
    return json_column('travel_details', subquery)


def get_person_trips_column():
    subquery = (
        select([
            person_trip_table,
            get_person_column(person_trip_table, label='person'),
            get_documents_full_column(),

            get_person_trip_purposes_column(),

            get_travel_details_column(),
            get_person_conf_details_column(),
        ])
        .where(trip_table.c.trip_id == person_trip_table.c.trip_id)
        .order_by(person_trip_table.c.created_at)
        .correlate(trip_table)
    )
    return json_agg_column('person_trips', subquery)


def get_person_trip_column():
    subquery = (
        select([
            person_trip_table,
            get_person_column(person_trip_table, label='person'),
        ])
        .where(service_table.c.trip_id == person_trip_table.c.trip_id)
        .where(service_table.c.person_id == person_trip_table.c.person_id)
        .correlate(service_table)
    )
    return json_column('person_trip', subquery)


def get_trip_purposes_column():
    subquery = (
        select([
            purpose_table,
        ])
        .select_from(
            trip_purpose_table.join(
                purpose_table,
                trip_purpose_table.c.purpose_id == purpose_table.c.purpose_id,
            )
        )
        .where(trip_table.c.trip_id == trip_purpose_table.c.trip_id)
        .correlate(trip_table)
    )
    return json_agg_column('purposes', subquery)


def get_conf_details_column():
    subquery = (
        select([
            conf_details_table,
        ])
        .where(trip_table.c.trip_id == conf_details_table.c.trip_id)
        .correlate(trip_table)
    )
    return json_column('conf_details', subquery)


def get_person_conf_details_column():
    subquery = (
        select([
            person_conf_details_table,
        ])
        .where(person_trip_table.c.trip_id == person_conf_details_table.c.trip_id)
        .where(person_trip_table.c.person_id == person_conf_details_table.c.person_id)
        .correlate(person_trip_table)
    )
    return json_column('conf_details', subquery)


def get_documents_column():
    subquery = (
        select([
            person_trip_document_table,
        ])
        .where(person_trip_table.c.trip_id == person_trip_document_table.c.trip_id)
        .where(person_trip_table.c.person_id == person_trip_document_table.c.person_id)
        .correlate(person_trip_table)
    )
    return json_agg_column('documents', subquery)


def get_person_trips_with_details_column(person_id: int = None, manager_id: int = None):
    """
    Subquery for person-trips related with Trip by trip_id
    if person_id and manager_id are set at the same call, then use:
        person_trip.person_id == person_id OR person_trip.manager_id == manager_id

    :param person_id: if not None then filter person_trip by person_trip.person_id
    :param manager_id: if not None then filter person_trip by person_trip.manager_id
    :return:
    """
    subquery = (
        select([
            person_trip_table,
            get_person_column(person_trip_table, label='person'),
            get_travel_details_column(),
            get_person_conf_details_column(),
            trip_table.c.author_id.label('trip_author_id'),
            get_person_with_company_column(correlate_table=person_trip_table),
            trip_table.c.city_to,
            trip_table.c.country_to,
            trip_table.c.provider_city_to_id,
            trip_table.c.aeroclub_city_to_id,
            get_person_trip_route_points_column(),
        ])
        .where(trip_table.c.trip_id == person_trip_table.c.trip_id)
    )
    where_parts = []
    if person_id:
        where_parts.append(person_trip_table.c.person_id == person_id)
    if manager_id:
        where_parts.append(person_trip_table.c.manager_id == manager_id)
    if where_parts:
        subquery = subquery.where(or_(*where_parts))
    return json_agg_column('person_trips', subquery.correlate(trip_table))


def get_services_column():
    subquery = (
        select([
            service_table,
            trip_table.c.author_id.label('trip_author_id'),
            person_trip_table.c.is_approved.label('is_person_trip_approved'),
        ])
        .select_from(
            service_table
            .join(trip_table, onclause=service_table.c.trip_id == trip_table.c.trip_id)
        )
        .where(person_trip_table.c.trip_id == service_table.c.trip_id)
        .where(person_trip_table.c.person_id == service_table.c.person_id)
        .correlate(person_trip_table)
    )
    return json_agg_column('services', subquery)


def get_route_points_query(table, where_clause, correlate_table):
    subquery = (
        select([
            table.c.city,
            table.c.country,
            table.c.date,
            table.c.provider_city_id,
            table.c.aeroclub_city_id,
            table.c.need_hotel,
            table.c.need_transfer,
        ])
        .where(where_clause)
        .order_by(table.c.date)
        .correlate(correlate_table)
    )
    return json_agg_column('route', subquery)


def get_trip_route_points_column():
    return get_route_points_query(
        table=trip_route_point_table,
        where_clause=trip_table.c.trip_id == trip_route_point_table.c.trip_id,
        correlate_table=trip_table,
    )


def get_person_trip_route_points_column():
    return get_route_points_query(
        table=person_trip_route_point_table,
        where_clause=and_(
            person_trip_table.c.trip_id == person_trip_route_point_table.c.trip_id,
            person_trip_table.c.person_id == person_trip_route_point_table.c.person_id,
        ),
        correlate_table=person_trip_table,
    )
