import datetime
import uuid

from sqlalchemy import (
    Table,
    Column,
    ForeignKey,
    MetaData,
    PrimaryKeyConstraint,
    ForeignKeyConstraint,
    types,
    Index,
)
from sqlalchemy.dialects.postgresql import UUID

from intranet.trip.src import enums


convention = {
    'all_column_names': lambda constraint, table: '_'.join([
        column.name for column in constraint.columns.values()
    ]),
    'ix': 'ix__%(table_name)s__%(all_column_names)s',
    'uq': 'uq__%(table_name)s__%(all_column_names)s',
    'ck': 'ck__%(table_name)s__%(constraint_name)s',
    'fk': 'fk__%(table_name)s__%(all_column_names)s__%(referred_table_name)s',
    'pk': 'pk__%(table_name)s'
}

metadata = MetaData(naming_convention=convention)


class TimestampedTable(Table):
    def __new__(cls, *args, **kwargs):
        assert isinstance(args[0], str)
        assert isinstance(args[1], MetaData)
        place = 2
        while place < len(args):
            if isinstance(args[place], Column) and args[place].primary_key:
                place += 1
            else:
                break

        created_at = Column(
            'created_at',
            types.DateTime(timezone=True),
            default=datetime.datetime.now,
        )
        updated_at = Column(
            'updated_at',
            types.DateTime(timezone=True),
            default=datetime.datetime.now,
            onupdate=datetime.datetime.now,
        )

        args = [*args[:place], created_at, updated_at, *args[place:]]
        return Table.__new__(cls, *args, **kwargs)


def route_point_base_columns():
    return (
        Column('city', types.String(64), nullable=False),
        Column('country', types.String(64), nullable=False),
        Column('date', types.Date, nullable=False),
        Column('provider_city_id', types.String, nullable=True),
        Column('aeroclub_city_id', types.Integer, nullable=True),
        Column('is_start_city', types.Boolean, nullable=False, default=False),
        Column('need_hotel', types.Boolean, nullable=False, default=True),
        Column('need_transfer', types.Boolean, nullable=False, default=True),
    )


# табличка в базе staff для синка оттуда документов
staff_passport_table = Table(
    'person_passport',
    MetaData(),
    # Column('id', types.Integer, primary_key=True),
    Column('description', types.String(256), nullable=False),
    Column('doc_type', types.String(16), nullable=False),
    Column('issue_country', types.String(128), nullable=False),
    # Column('country_code', types.String(16), nullable=False),
    Column('number', types.String(64), nullable=False),

    Column('issue_date', types.Date, nullable=True),
    Column('expire_date', types.Date, nullable=True),

    Column('issued_by', types.String(256), nullable=False),

    Column('first_name', types.String(50), nullable=False),
    Column('last_name', types.String(100), nullable=False),
    Column('middle_name', types.String(100), nullable=False),

    Column('is_active', types.Boolean, nullable=False),
    Column('created_at', types.DateTime, nullable=False),
    Column('modified_at', types.DateTime, nullable=False),

    Column('person_id', types.Integer, nullable=False),
)

# Цель командировки
purpose_table = Table(
    'list_purpose',
    metadata,
    Column('purpose_id', types.Integer, primary_key=True),
    Column('name', types.String(128), nullable=False, default=''),
    Column('name_en', types.String(128), nullable=False, default=''),
    Column('kind', types.Enum(enums.PurposeKind), nullable=False, default=enums.PurposeKind.any),
    Column('aeroclub_grade', types.Integer, nullable=False, default=0)
)


# Город
city_table = Table(
    'list_city',
    metadata,
    Column('city_id', types.Integer, primary_key=True),
    Column('code', types.String(16), nullable=False),
)


# Холдинг
holding_table = Table(
    'list_holding',
    metadata,
    Column('holding_id', types.Integer, primary_key=True),
    Column('name', types.String(64), nullable=False, default=''),
)


# Компания (юр.лицо внутри холдинга)
company_table = Table(
    'list_company',
    metadata,
    Column('company_id', types.Integer, primary_key=True),
    Column('name', types.String(64), nullable=False, default=''),
    Column('aeroclub_company_id', types.Integer, nullable=True),
    Column('aeroclub_company_uid', types.String(64), nullable=True),
    Column('country', types.Enum(enums.Citizenship), nullable=True),
    Column('provider', types.Enum(enums.Provider), nullable=False),
    Column('default_active', types.Boolean, nullable=False, default=True),
    Column('holding_id', ForeignKey('list_holding.holding_id'), nullable=False),
)


# Справочник поставщиков услуг. Актуально только для Аэроклуба
service_provider_table = Table(
    'list_service_provider',
    metadata,
    Column('service_type', types.Enum(enums.ServiceType), nullable=False, primary_key=True),
    Column('code', types.String(8), nullable=False, primary_key=True),
    Column('name', types.String(40), nullable=False, default=''),
    Column('name_en', types.String(40), nullable=False, default=''),

    PrimaryKeyConstraint('service_type', 'code'),
)


# Сотрудник компании
person_table = TimestampedTable(
    'person',
    metadata,
    Column('person_id', types.Integer, primary_key=True),

    Column('uid', types.String(length=64), nullable=False),
    # uid для синка с аэроклубом
    Column('external_uid', types.String(length=64), nullable=False, default=''),
    Column('provider_profile_id', types.Integer, nullable=True, default=None),
    Column('login', types.String(length=64), nullable=False, default=''),
    Column('first_name', types.String(length=64), nullable=False),
    Column('last_name', types.String(length=64), nullable=False),
    Column('middle_name', types.String(length=64), nullable=True),
    Column('middle_name_en', types.String(length=64), nullable=True),
    Column('first_name_en', types.String(length=64), nullable=True),
    Column('last_name_en', types.String(length=64), nullable=True),

    # поля для отправки корректного ФИО в hub
    Column('first_name_ac', types.String(length=64), nullable=True),
    Column('last_name_ac', types.String(length=64), nullable=True),
    Column('middle_name_ac', types.String(length=64), nullable=True),
    Column('first_name_ac_en', types.String(length=64), nullable=True),
    Column('last_name_ac_en', types.String(length=64), nullable=True),
    Column('middle_name_ac_en', types.String(length=64), nullable=True),

    Column('gender', types.Enum(enums.Gender), nullable=True),
    Column('date_of_birth', types.Date, nullable=True),

    Column('phone_number', types.String(length=32), nullable=True),
    Column('email', types.String(length=64), nullable=True),

    Column('company_id', ForeignKey('list_company.company_id'), nullable=True),

    Column('is_coordinator', types.Boolean, nullable=False, default=False),
    Column('is_dismissed', types.Boolean, nullable=False, default=False),
    Column('is_active', types.Boolean, nullable=False, default=False),
    Column('is_limited_access', types.Boolean, nullable=False, default=False),

    Column('email_confirmed_at', types.DateTime(timezone=True), nullable=True),
    Column('rejected_at', types.DateTime(timezone=True), nullable=True),

    Column('support_id', ForeignKey('person.person_id'), nullable=True),
    Column('chat_id', types.String(length=64), nullable=True),

    Index('person_table__login', 'login'),
    Index('person_table__external_uid', 'external_uid'),
    Index('person_table__provider_profile_id', 'provider_profile_id'),
)


# +1 сотрудника компании
ext_person_table = TimestampedTable(
    'ext_person',
    metadata,
    Column('ext_person_id', types.Integer, primary_key=True),
    Column('person_id', ForeignKey('person.person_id'), nullable=False),
    # TODO: лучше бы назвать alias, чтобы не было путаницы
    Column('name', types.String(64), nullable=False),
    Column('secret', types.String(64), nullable=True),
    Column('status', types.Enum(enums.ExtPersonStatus), nullable=False),

    # Общие поля с person_table
    Column('email', types.String(256), nullable=False, default=''),
    Column('provider_profile_id', types.Integer, nullable=True),
    Column('external_uid', types.String(length=64), nullable=False, default=''),
    Column('gender', types.Enum(enums.Gender), nullable=True),
    Column('date_of_birth', types.Date, nullable=True),
    Column('phone_number', types.String(length=32), nullable=True),
    # разный смысл у полей из-за _ac
    Column('first_name', types.String(length=64), nullable=False, default=''),
    Column('last_name', types.String(length=64), nullable=False, default=''),
    Column('middle_name', types.String(length=64), nullable=True, default=''),
    Column('middle_name_en', types.String(length=64), nullable=True, default=''),
    Column('first_name_en', types.String(length=64), nullable=True, default=''),
    Column('last_name_en', types.String(length=64), nullable=True, default=''),

    Index('ext_person_table__external_uid', 'external_uid'),
    Index('ext_person_table__provider_profile_id', 'provider_profile_id'),
)


# Связь двух сотрудников компании
person_relationship_table = Table(
    'person_relationship',
    metadata,
    Column('role', types.Enum(enums.PersonRole), nullable=False, primary_key=True),
    Column('owner_id', ForeignKey('person.person_id'), nullable=False, primary_key=True),
    Column('dependant_id', ForeignKey('person.person_id'), nullable=False, primary_key=True),

    Column('is_direct', types.Boolean, default=False),
)


# Групповая командировка
trip_table = TimestampedTable(
    'trip',
    metadata,
    Column('trip_id', types.Integer, primary_key=True),
    Column('staff_trip_uuid', types.String(length=36), nullable=True, default=None),

    Column('status', types.Enum(enums.TripStatus), nullable=False, default=enums.TripStatus.new),

    Column('city_from', types.String(length=128), nullable=True),
    Column('city_to', types.String(length=128), nullable=True),

    Column('country_from', types.String(length=128), nullable=True),
    Column('country_to', types.String(length=128), nullable=True),

    Column('date_from', types.Date, nullable=False),
    Column('date_to', types.Date, nullable=False),

    # TODO: поправить на tracker_issue
    Column('issue_travel', types.String(length=16)),

    Column('author_id', ForeignKey('person.person_id'), nullable=False),
    Column('comment', types.String(length=4096), nullable=False, default=''),

    Column('aeroclub_city_from_id', types.Integer, nullable=True),
    Column('aeroclub_city_to_id', types.Integer, nullable=True),
    Column('provider_city_from_id', types.Integer, nullable=True),
    Column('provider_city_to_id', types.Integer, nullable=True),

    Column('chat_id', types.String(length=64), nullable=True),

    Index('trip__issue_travel', 'issue_travel'),
    Index('trip__date_from', 'date_from'),
)


# Персональная командировка
person_trip_table = TimestampedTable(
    'person_trip',
    metadata,

    Column('trip_id', ForeignKey('trip.trip_id'), primary_key=True),
    Column('person_id', ForeignKey('person.person_id'), primary_key=True),
    Column('provider', types.Enum(enums.Provider), nullable=False),

    Column('city_from', types.String(length=128), nullable=True),
    Column('country_from', types.String(length=128), nullable=True),

    Column('status', types.Enum(enums.PTStatus), nullable=False),
    # Получен approve в тикете
    Column('is_approved', types.Boolean, nullable=True, default=False),
    # Авторизовано в Aeroclub
    Column('is_authorized', types.Boolean, nullable=True, default=False),
    Column('description', types.Text, nullable=True),

    Column('gap_date_from', types.Date, nullable=True),
    Column('gap_date_to', types.Date, nullable=True),
    Column('is_hidden', types.Boolean, nullable=False, default=False),
    Column('is_offline', types.Boolean, nullable=False, default=False),

    Column('with_days_off', types.Boolean, nullable=False),
    Column('compensation_type', types.Enum(enums.TripDaysOffCompensations), nullable=True),

    Column('need_visa', types.Boolean, nullable=False, default=False),

    Column('aeroclub_journey_id', types.Integer, nullable=True),
    Column('aeroclub_trip_id', types.Integer, nullable=True),

    Column('provider_city_from_id', types.Integer, nullable=True),
    Column('aeroclub_city_from_id', types.Integer, nullable=True),

    Column('aeroclub_last_message_id', types.Integer, nullable=True),

    Column('chat_id', types.String(length=64), nullable=True),
    Column('manager_chat_id', types.String(length=64), nullable=True),

    Column('manager_id', ForeignKey('person.person_id'), primary_key=False),

    Index('aeroclub__person_trip_id', 'aeroclub_journey_id', 'aeroclub_trip_id'),
    Index('person_trip__aeroclub_city_from_id', 'aeroclub_city_from_id'),
    Index('person_trip__provider_city_from_id', 'provider_city_from_id'),
)


# Связь цели и групповой командировки
trip_purpose_table = Table(
    'link_trip_purpose',
    metadata,
    Column('trip_id', ForeignKey('trip.trip_id'), nullable=False),
    Column('purpose_id', ForeignKey('list_purpose.purpose_id'), nullable=False),

    PrimaryKeyConstraint('trip_id', 'purpose_id'),
)


# Связь цели и персональной командировки
person_trip_purpose_table = Table(
    'link_persontrip_purpose',
    metadata,
    Column('trip_id', types.Integer, nullable=False),
    Column('person_id', types.Integer, nullable=False),
    Column('purpose_id', ForeignKey('list_purpose.purpose_id'), nullable=False),

    PrimaryKeyConstraint('trip_id', 'person_id', 'purpose_id'),
    ForeignKeyConstraint(
        ('trip_id', 'person_id'),
        ('person_trip.trip_id', 'person_trip.person_id'),
    ),
)


# Документы сотрудника
person_document_table = TimestampedTable(
    'person_document',
    metadata,
    Column('document_id', types.Integer, primary_key=True),
    Column('person_id', ForeignKey('person.person_id'), nullable=True),
    Column('ext_person_id', ForeignKey('ext_person.ext_person_id'), nullable=True),

    Column('document_type', types.Enum(enums.DocumentType), nullable=False),

    Column('series', types.String(length=8), nullable=True),
    Column('number', types.String(length=32), nullable=True),
    Column('issued_by', types.String(256), nullable=True),
    Column('issued_on', types.Date, nullable=True),
    Column('expires_on', types.Date, nullable=True),
    Column('citizenship', types.Enum(enums.Citizenship), nullable=False, default=enums.Citizenship.RU),

    Column('first_name', types.String(length=50), nullable=False),
    Column('last_name', types.String(length=50), nullable=False),
    Column('middle_name', types.String(length=50), nullable=True),
    Column('description', types.String(256), nullable=True),
    Column('is_deleted', types.Boolean, nullable=False, default=False),
    Column('external_document_id', types.String(length=256), nullable=True),
    Column('provider_document_id', types.Integer, nullable=True),
)


# Документы, добавленные в персональную командировку
person_trip_document_table = Table(
    'link_persontrip_document',
    metadata,
    Column('trip_id', types.Integer, nullable=False),
    Column('person_id', types.Integer, nullable=False),
    Column('document_id', ForeignKey('person_document.document_id'), nullable=False),

    PrimaryKeyConstraint('trip_id', 'document_id'),
    ForeignKeyConstraint(
        ('trip_id', 'person_id'),
        ('person_trip.trip_id', 'person_trip.person_id'),
    ),
)


# Бонусные карты сотрудника
person_bonus_card_table = TimestampedTable(
    'bonus_card',
    metadata,
    Column('bonus_card_id', types.Integer, primary_key=True),
    Column('person_id', ForeignKey('person.person_id'), nullable=True),
    Column('ext_person_id', ForeignKey('ext_person.ext_person_id'), nullable=True),
    Column('number', types.String(64), nullable=False),
    Column('service_provider_type', types.Enum(enums.ServiceType), nullable=False),
    Column('service_provider_code', types.String(8), nullable=False),

    ForeignKeyConstraint(
        ('service_provider_type', 'service_provider_code'),
        ('list_service_provider.service_type', 'list_service_provider.code'),
    ),
)


# Точка маршрута в групповой командировке
trip_route_point_table = TimestampedTable(
    'trip_route_point',
    metadata,
    Column('point_id', types.Integer, primary_key=True),
    Column('trip_id', ForeignKey('trip.trip_id'), nullable=False),

    *route_point_base_columns(),
)


# Точка маршрута в перональной командировке
person_trip_route_point_table = TimestampedTable(
    'person_trip_route_point',
    metadata,
    Column('point_id', types.Integer, primary_key=True),
    Column('trip_id', types.Integer, nullable=False),
    Column('person_id', types.Integer, nullable=False),

    *route_point_base_columns(),

    ForeignKeyConstraint(
        ('trip_id', 'person_id'),
        ('person_trip.trip_id', 'person_trip.person_id'),
    ),
)


# Дополнительная информация персональной командировки типа "командировка"
travel_details_table = TimestampedTable(
    'trip_travel_details',
    metadata,
    Column('trip_id', ForeignKey('trip.trip_id'), primary_key=True),
    Column('person_id', ForeignKey('person.person_id'), primary_key=True),

    Column('city_id', ForeignKey('list_city.city_id'), nullable=False),

    Column('tracker_issue', types.String(length=16), nullable=True),
    Column('is_created_on_provider', types.Boolean, nullable=False),

    # Нужна помощь в оформлении визы
    Column('need_visa_assistance', types.Boolean, nullable=False, default=False),

    Column('taxi_date_from', types.Date(), nullable=True),
    Column('taxi_date_to', types.Date(), nullable=True),
    Column('is_taxi_activated', types.Boolean(), nullable=False, default=False),
    Column('is_drive_activated', types.Boolean(), nullable=False, default=False),
    Column('drive_org_id', types.String(length=64), nullable=True),
    Column('drive_user_id', types.String(length=64), nullable=True),

    Column('taxi_access_phone', types.String(14), nullable=True),
    Column('comment', types.String(length=256), nullable=False, default=''),

    # добавить comment
    PrimaryKeyConstraint('trip_id', 'person_id'),
    ForeignKeyConstraint(
        ('trip_id', 'person_id'),
        ('person_trip.trip_id', 'person_trip.person_id')
    ),

    Index('trip_travel_details__tracker_issue', 'tracker_issue'),
)


# Дополнительная информация групповой командировки типа "конференция"
conf_details_table = TimestampedTable(
    'trip_conf_details',
    metadata,
    Column('trip_id', ForeignKey('trip.trip_id'), primary_key=True),

    Column('conference_name', types.String(length=256)),
    Column('tracker_issue', types.String(length=16)),

    Column('price', types.String(256), nullable=True),
    Column('promo_code', types.String(256), nullable=True),

    Column('conf_date_from', types.Date, nullable=False),
    Column('conf_date_to', types.Date, nullable=False),

    Column('conference_url', types.String(128), nullable=True),
    Column('program_url', types.String(256), nullable=True),

    Column('participation_terms', types.Text, nullable=True),
    Column('cancellation_terms', types.Text, nullable=True),

    Column('ticket_type', types.String(256), nullable=True),
    Column('is_another_city', types.Boolean(), nullable=False, default=False),

    Index('trip_conf_details__tracker_issue', 'tracker_issue'),
)


# Дополнительная информация персональной командировки типа "конференция"
person_conf_details_table = TimestampedTable(
    'person_trip_conf_details',
    metadata,
    Column('trip_id', ForeignKey('trip.trip_id'), primary_key=True),
    Column('person_id', ForeignKey('person.person_id'), primary_key=True),

    Column(
        'role', types.Enum(enums.ConferenceParticiationType),
        nullable=False,
        default=enums.ConferenceParticiationType.speaker,
    ),

    Column('presentation_topic', types.String(length=128), nullable=True),
    Column('is_hr_approved', types.Boolean, nullable=False, default=False),
    Column('is_paid_by_host', types.Boolean, nullable=False, default=False),

    Column('tracker_issue', types.String(length=16)),

    Column('price', types.String(256), nullable=True),
    Column('promo_code', types.String(256), nullable=True),
    Column('discount', types.String(256), nullable=True),

    Column('badge_position', types.String(128), nullable=True),
    Column('badge_name', types.String(128), nullable=True),

    Column('comment', types.String(length=256), nullable=False, default=''),

    Column('is_another_city', types.Boolean(), nullable=False, default=False),

    ForeignKeyConstraint(
        ('trip_id', 'person_id'),
        ('person_trip.trip_id', 'person_trip.person_id')
    ),

    Index('person_trip_conf_details__tracker_issue', 'tracker_issue'),
)


# Услуга в персональной командировке (авиа, жд, отель)
service_table = TimestampedTable(
    'service',
    metadata,
    Column('service_id', types.Integer, primary_key=True),
    Column('trip_id', types.Integer, nullable=False),
    Column('person_id', types.Integer, nullable=False),

    Column('type', types.Enum(enums.ServiceType), nullable=False),
    Column('ordering', types.Integer, nullable=False, default=0),  # cортировка в списке услуг
    Column(
        'status',
        types.Enum(enums.ServiceStatus),
        nullable=False,
        default=enums.ServiceStatus.draft,
    ),
    Column('boarding_pass', types.String(128)),  # - Посадочный талон (строка, линк на хранилище файлов)

    Column('is_broken', types.Boolean, nullable=True, default=False),
    Column('provider_service_id', types.Integer, nullable=True),
    Column('provider_order_id', types.Integer, nullable=True),
    Column('provider_document_id', types.Integer, nullable=True),
    Column('is_from_provider', types.Boolean, default=False),
    Column('is_authorized', types.Boolean, default=False),
    Column('seat_number', types.Integer, nullable=True),
    Column('in_process_of_cancelling', types.Boolean, nullable=False, default=False),
    Column('document_id', ForeignKey('person_document.document_id'), nullable=True),

    ForeignKeyConstraint(
        ('trip_id', 'person_id'),
        ('person_trip.trip_id', 'person_trip.person_id'),
    ),
)


# Транзакции для отображения в Балансе
billing_transactions_table = TimestampedTable(
    'billing_transaction',
    metadata,
    Column('transaction_id', UUID(as_uuid=True), default=uuid.uuid4),
    Column('company_id', ForeignKey('list_company.company_id'), nullable=False),
    Column('trip_id', ForeignKey('trip.trip_id'), nullable=False),
    Column('service_id', ForeignKey('service.service_id'), nullable=True),
    Column('general_service_id', types.String(128), nullable=True),  # номер билета, страховки, etc
    Column('service_type', types.Enum(enums.TransactionServiceType), nullable=False),
    Column('person_id', ForeignKey('person.person_id'), nullable=False),
    Column('status', types.Enum(enums.TransactionStatus), nullable=False),
    Column('type', types.Enum(enums.TransactionType), nullable=False),
    Column('original_transaction_id', UUID(as_uuid=True), nullable=True, default=None),
    Column('author_id', ForeignKey('person.person_id'), nullable=True, default=None),
    Column('execution_date', types.Date),
    Column('invoice_date', types.Date),
    Column('price', types.DECIMAL, nullable=False),
    Column('provider_fee', types.DECIMAL, nullable=False),
    Column('yandex_fee', types.DECIMAL, nullable=False),
    Column('currency', types.Enum(enums.CurrencyType), default=enums.CurrencyType.rub, nullable=False),
    Column('is_penalty', types.Boolean, default=False, nullable=False),

    Column('is_obsolete', types.Boolean, default=False, nullable=False),

    Index(
        'only_one_not_obsolete_transaction',
        'transaction_id',
        unique=True,
        postgresql_where='NOT is_obsolete',
    ),
)


# Домены компаний
company_domain_table = TimestampedTable(
    'company_domain',
    metadata,
    Column('company_id', ForeignKey('list_company.company_id'), nullable=True, primary_key=True),
    Column('domain', types.String(length=128), primary_key=True),
)


# Метаданные файла реестра
registry_metadata_table = TimestampedTable(
    'registry_meta',
    metadata,
    Column('registry_id', types.Integer, primary_key=True),
    Column('s3_key', types.String(512), nullable=False, unique=True),

    Column('filename', types.String(256), nullable=True),
    Column('size', types.Integer, nullable=True),
    Column('status', types.Enum(enums.RegistryStatus), nullable=False, default=enums.RegistryStatus.created),

    # Для выгрузки реестра из s3
    Index('registry_meta__s3_key', 's3_key'),
    # Для поиска не отправленных реестров за день
    Index('registry_meta__status_date', 'created_at', 'status'),
)

# Депозиты компаний-клиентов
billing_deposit_table = TimestampedTable(
    'billing_deposit',
    metadata,
    Column('deposit_id', UUID(as_uuid=True), default=uuid.uuid4, primary_key=True, nullable=False),
    Column('charge_date', types.Date, nullable=False),
    Column('company_id', ForeignKey('list_company.company_id'), nullable=False),
    Column('amount', types.DECIMAL, nullable=False),
    Column('author_id',  ForeignKey('person.person_id'), nullable=False),
)
