from decimal import Decimal
from pydantic import BaseModel, validator

from intranet.trip.src.lib.aeroclub import enums
from intranet.trip.src.lib.aeroclub.models.common import (
    ACAddress,
    Name,
    ACCity,
    CodeName,
    ACGeoPosition,
    GeoObject,
)


# AVIA
class ACAirline(CodeName):
    code_ru: str = None
    registration_links: str = None


class SegmentLite(BaseModel):
    flight_number: str = None
    marketing_airline: ACAirline = None


class ACBaggage(BaseModel):
    type: str
    quantity: int = None
    weight_measurement: str = None


class Segment(BaseModel):
    """Сегмент авиа-перелета"""
    id: int = None
    duration: str = None
    booking_class: CodeName = None
    flight_number: str = None
    airline: ACAirline = None
    baggage: ACBaggage = None
    class_of_service: CodeName = None
    arrival_at_utc: str = None
    arrival_at: str = None
    departure_at_utc: str = None
    departure_at: str = None
    arrival_to: GeoObject = None
    departure_from: GeoObject = None
    marketing_airline_code: str = None
    # flight_status
    actual_arrival_at_utc: str = None
    actual_arrival_at: str = None
    actual_departure_at_utc: str = None
    actual_departure_at: str = None
    # aircraft: Name
    aircraft: CodeName = None


class LegLite(BaseModel):
    departure_from: ACCity = None
    arrival_to: ACCity = None
    departure_at: str = None
    departure_at_utc: str = None
    arrival_at: str = None
    arrival_at_utc: str = None
    cabin_class: str = None
    transfers: int = None
    duration: str = None
    segments: list[SegmentLite] = None
    # baggage: Baggage


class Leg(LegLite):
    id: int = None
    departure_from: GeoObject = None
    arrival_to: GeoObject = None
    segments: list[Segment] = None


# RAIL
class CarriageInfo(BaseModel):
    min_price: Decimal = None
    type: str = None  # enum
    carriage_owner: str = None
    seat_count: int = None


class Place(BaseModel):
    number: int = None
    compartment_type: str = None


class Carriage(BaseModel):
    """Данные о вагоне ЖД"""
    index: int
    is_travel_policy_compliant: bool = None
    travel_policy_violation_reason: str = None
    carriage_type: str = None  # enum
    carriage_owner: str = None
    carriage_num: str = None
    places: list[Place] = None
    is_electronic_registration_allowed: bool = None
    carriage_class_options: list[str] = None  # enum
    # services_description
    # class: CodeName
    places_summary: dict = None
    min_price_for_class: Decimal = None
    max_price_for_class: Decimal = None
    is_two_floor: bool = None
    # schema: str


# HOTEL
class HotelTariff(BaseModel):
    price_per_night: Decimal = None
    payment_place: str = None  # enum?
    total_price: Decimal = None
    service_fee: Decimal = None
    is_corporate_tariff: bool = None
    confirmation_type: str = None  # enum
    index: int = None


class Meal(BaseModel):
    """Питание в самолете"""
    code: str = None
    name: str = None
    meal_type: str = None  # enum
    price: Decimal = None


class Image(BaseModel):
    url: str = None
    preview_url: str = None


class ACRoom(BaseModel):
    """Номер в отеле"""
    room_name: str = None
    total_price: Decimal = None
    actual_checkin_at: str = None
    actual_checkin_at_utc: str = None
    actual_checkout_at: str = None
    actual_checkout_at_utc: str = None
    service_type: str = None  # enum
    additional_room_info: str = None
    is_pay_on_site: bool = None
    asim_room_commission: Decimal = None
    booking_code: str = None
    room_rate_id: str = None
    sale_type: str = None  # enum
    meals: list[Meal] = None
    currency: str = None
    tariff_per_night: Decimal = None
    index: int = None
    tariff_expected_cancellation_penalty: Decimal = None
    tariff_vat: Decimal = None
    tariff_additional_fee_percent: Decimal = None
    tariff_discount_percent: Decimal = None
    tariff_rack_rate: Decimal = None
    allocation_is_limited: bool = None
    check_in_at: str = None
    check_in_at_utc: str = None
    check_out_at: str = None
    check_out_at_utc: str = None
    room_id: str = None
    tariff_total: Decimal = None
    tariff_adult: Decimal = None
    is_travel_policy_compliant: bool = None
    tariff_service_fee: Decimal = None
    is_corporate_tariff: bool = None
    supplier_unique_identifier: str = None
    supplier_room_id: str = None
    corporate_tariff_code: str = None
    allocation_adults: int = None
    reservation_cancellation_policy: str = None
    travel_policy_violation_reason: str = None
    is_hidden_price: bool = None


class SearchResultCount(BaseModel):
    count: int = 0
    expires_at_utc: str = None


class SearchResultVariantsCount(BaseModel):
    expires_at_utc: str = None
    variants_count: int
    variants_count_avia: int
    variants_count_hotel: int
    variants_count_rail: int
    variants_count_rail_round_trip: int


class SearchResultBase(BaseModel):
    service_type: enums.ACServiceType = None
    key: str = None
    is_corporate_tariff: bool = None
    raw: dict


class SearchResultInList(SearchResultBase):
    option_number: int
    sequence_number: int = None
    currency: str = None

    # avia and rail fields
    arrival_to: ACCity = None
    departure_from: ACCity = None
    tariff_total: Decimal = None
    tariff_service_fee: Decimal = None

    # avia fields
    legs: list[LegLite] = None
    # mini_rules: list[dict]

    # rail fields
    client_id: int = None
    profile_grade_id: int = None
    departure_at: str = None
    departure_at_utc: str = None
    is_local_departure_at: int = None
    arrival_at: str = None
    arrival_at_utc: str = None
    is_local_arrival_at: bool = None
    train_code: str = None
    train_name: str = None
    carriages_info: list[CarriageInfo] = None
    electronic_registration: bool = None
    duration: str = None

    @validator('arrival_to', 'departure_from', pre=True)
    def extract_city(cls, v):
        """
        Почему-то у rail города приходят в другом формате
        """
        if v and 'city' in v:
            return v['city']
        return v

    # hotel fields
    hotel: CodeName = None
    city: ACCity = None
    is_direct_contract: bool = None
    stars: int = None
    geo_position: ACGeoPosition = None
    image_url: str = None
    is_top_hotel: bool = None
    start_on: str = None
    end_on: str = None
    tariffs: list[HotelTariff] = None
    address: Name = None


class SearchResultDetailBase(SearchResultBase):
    start_on: str = None
    end_on: str = None
    start_on_utc: str = None
    end_on_utc: str = None
    tariff_total: Decimal = None
    is_travel_policy_compliant: bool = None
    tariff_tax: Decimal = None
    corporate_tariff_code: str = None
    reservation_cancellation_policy: str = None
    grade_custom_list_property_value_id: int = None
    travel_policy_violation_reason: str = None
    tariff_service_fee: Decimal = None


class _AviaDetailFields(BaseModel):
    legs: list[Leg] = None
    validating_airline_code: str = None
    is_segment_discount_variant: bool = None
    is_segment_discount_as_exact_value: bool = None
    # mini_rules


class _RailDetailFields(BaseModel):
    departure_from: ACCity = None
    arrival_to: ACCity = None
    departure_at: str = None
    departure_at_utc: str = None
    is_local_departure_at: bool = None
    arrival_at: str = None
    arrival_at_utc: str = None
    duration: str = None
    is_local_arrival_at: bool = None
    train_code: str = None
    train_name: str = None
    train_category: str = None  # enum
    branded: bool = None
    currency: str = None
    carriage_details: list[Carriage] = None


class _HotelDetailFields(BaseModel):
    hotel: CodeName = None
    city: ACCity = None
    stars: int = None
    rooms: list[ACRoom] = None
    address: ACAddress = None
    geo_position: ACGeoPosition = None
    images: list[str] = None
    hotel_images: list[Image] = None
    web_site: str = None
    is_recommended: bool = None
    rating: int = None
    is_top_hotel: bool = None
    # recommendations


class AviaSearchResult(SearchResultDetailBase, _AviaDetailFields):
    """
    Результат поиска по авиа
    """


class ACRailSearchResult(SearchResultDetailBase, _RailDetailFields):
    """
    Результат поиска по ЖД-билетам
    """


class ACHotelSearchResult(SearchResultDetailBase, _HotelDetailFields):
    """
    Результат поиска по отелям
    """


class SearchResult(
        SearchResultDetailBase,
        _AviaDetailFields,
        _RailDetailFields,
        _HotelDetailFields,
):
    """
    Результат поиска, который содержит в себе все возможные поля всех типов услуг
    """
