from enum import Enum
import logging
from typing import Any, Mapping

import aiohttp
from fastapi.encoders import jsonable_encoder
from tenacity import TryAgain

from intranet.trip.src.config import settings
from intranet.trip.src.lib.aviacenter.exceptions import AviaCenterError
from intranet.trip.src.lib.aviacenter.fields import boolint
from intranet.trip.src.lib.aviacenter.models import (
    AviaBookIn,
    AviaSearchFilter,
    AviaSearchRequest,
    AviaSort,
    HotelBookIn,
    HotelSearchFilter,
    HotelSearchRequest,
    HotelSort,
    TrainBookIn,
    TrainSearchFilter,
    TrainSearchRequest,
    TrainSort,
)
from intranet.trip.src.lib.provider import BaseServiceProviderClient

logger = logging.getLogger(__name__)


def flatten_dict(dictionary, prefix=''):
    """
    Список простых типов мы не раскрываем по индексам,
    потому что это делает сам aiohttp, дублируя значения в списке в multidict
    """
    res = {}
    for key, value in dictionary.items():
        new_prefix = prefix + f'[{key}]' if prefix else key
        if value is None:
            continue
        if isinstance(value, dict):
            res.update(flatten_dict(value, prefix=new_prefix))
        elif isinstance(value, list):
            if not value:
                continue
            if isinstance(value[0], dict):
                for i, item in enumerate(value):
                    res.update(flatten_dict(item, prefix=new_prefix + f'[{i}]'))
            else:
                res[new_prefix + '[]'] = value
        else:
            res[new_prefix] = value
    return res


class AviaCenter(BaseServiceProviderClient):
    """
    https://s1-api-colibri.crpo.su/api/documentation?access_level=Client
    https://yav.yandex-team.ru/secret/sec-01fkgg99550f6sczgver3tx0yw
    """
    CLIENT_NAME = 'aviacenter'
    RETRY_CODES = {
        499,
        502,
        503,
        504,
    }

    def get_auth_headers(self, *args, **kwargs) -> dict:
        """
        Авиторизация через параметр запроса
        """
        return {}

    def _cast_param(self, param_value):
        if isinstance(param_value, bool):
            return int(param_value)
        elif isinstance(param_value, Enum):
            return param_value.value
        elif isinstance(param_value, (list, tuple)):
            return [self._cast_param(val) for val in param_value]
        else:
            return str(param_value)

    def _prepare_params(self, params: dict | None) -> Mapping:
        params = params or {}
        result = {}

        params = flatten_dict(params)

        for key, value in params.items():
            result[key] = self._cast_param(value)

        result['auth_key'] = settings.AVIACENTER_API_TOKEN  # TODO Пофиксить в BTRIP-2148
        return result

    async def parse_response(self, response: aiohttp.ClientResponse, **kwargs) -> dict:
        if response.status in self.RETRY_CODES:
            raise TryAgain()
        try:
            response.raise_for_status()
        except aiohttp.ClientResponseError:
            message = (
                f'Got status: {response.status}, for request: '
                f'{response.method} {response.url}'
            )
            raise AviaCenterError(content=message, status_code=response.status)

        data = await response.json()

        if data['success'] is False:
            message = data['data']['message']
            pid = data['pid']
            error_code = data['code']
            logger.error(
                'Bad response from Aviacenter: %s, pid: %s, error_code: %s',
                message, pid, error_code
            )
            raise AviaCenterError(
                content=message,
                status_code=response.status,
                error_code=error_code,
                pid=pid,
            )

        return data['data']

    async def get_companies(self, limit: int, offset: int = 0) -> dict[str, Any]:
        """
        Получить все доступные компании (юр.лица Яндекса)
        """
        return await self._make_request(
            method='get',
            path='/companies/',
            params={
                'limit': limit,
                'offset': offset,
            },
        )

    async def get_airports(
        self,
        query: str = None,
        city_iata: str = None,
        lang: str = None,
    ) -> dict[str, Any]:
        """
        Саджест по аэропортам
        """
        params = {}
        if query is not None:
            params['part'] = query

        if city_iata is not None:
            params['city_iata'] = city_iata

        if lang is not None:
            params['lang'] = lang

        return await self._make_request(
            method='get',
            path='/avia/airports',
            params=params,
        )

    async def get_cities(self, query: str) -> dict[str, Any]:
        """
        Саджест по городам
        """
        params = {
            'part': query,
        }
        return await self._make_request(
            method='get',
            path='/cities',
            params=params,
        )

    async def create_avia_search(
            self,
            search_request: AviaSearchRequest,
            lang: str = None,
    ) -> dict[str, Any]:
        """
        Создать поиск по авиа
        """
        params = search_request.dict()
        if lang is not None:
            params['lang'] = lang
        return await self._make_request(
            method='get',
            path='/avia/search/',
            params=params,
        )

    async def get_avia_search_results(
            self,
            company_id: int,
            sro: str,
            travel_policy_id: int = None,
            is_async: boolint = 0,
            language: str = None,
            page: int = 1,
            limit: int = 20,
            search_filter: AviaSearchFilter = None,
            sort: list[AviaSort] = None,
    ) -> dict[str, Any]:
        """
        Получить информацию о результате поиска авиа

        :param company_id: id юр.лица яндекса в Авиацентре
        :param sro: это поле мы получаем в ответе create_avia_search
        :param travel_policy_id: id тревел-политики
        :param page: int номер страницы
        :param limit: int максимальное число на странице
        :param language: str язык на котором хотим получить данные
        :param is_async: 0 или 1 флаг для асинхронности
        :param search_filter: объект с параметрами фильтрации результатов поиска
        :param sort: список полей для сортировки в порядке приоритета
        """
        params = {
            'company_id': company_id,
            'sro': sro,
            'limit': limit,
            'offset': limit * (page - 1),
            'is_async': is_async,
        }
        if language is not None:
            params['lang'] = language

        if travel_policy_id:
            params['travel_policy_id'] = travel_policy_id

        if search_filter:
            if search_filter.sort and sort is None:
                sort = search_filter.sort
                search_filter.sort = None
            params['post_filter'] = jsonable_encoder(search_filter, exclude_none=True)

        if sort:
            params['sort'] = jsonable_encoder(sort, exclude_none=True)

        return await self._make_request(
            method='get',
            path='/avia/search-result/',
            params=params,
        )

    async def create_hotel_search(
            self,
            search_request: HotelSearchRequest,
            lang: str = None,
    ) -> dict[str, Any]:
        """
        Создать поиск по отелям
        """
        params = search_request.dict()
        if lang is not None:
            params['lang'] = lang
        return await self._make_request(
            method='get',
            path='/hotel/search-id/',
            params=params,
        )

    async def get_hotel_search_results(
            self,
            company_id: int,
            search_id: str,
            travel_policy_id: int = None,
            is_async: boolint = 0,
            language: str = None,
            page: int = 1,
            limit: int = 20,
            search_filter: HotelSearchFilter = None,
            sort: list[HotelSort] = None,
    ) -> dict[str, Any]:
        """
        Получить информацию о результате поиска отелей

        :param company_id: id юр.лица яндекса в Авиацентре
        :param search_id: это поле мы получаем в ответе create_hotel_search
        :param travel_policy_id: id тревел-политики
        :param page: int номер страницы
        :param limit: int максимальное число на странице
        :param language: str язык на котором хотим получить данные
        :param is_async: 0 или 1 флаг для асинхронности
        :param search_filter: объект с параметрами фильтрации результатов поиска
        :param sort: список полей для сортировки в порядке приоритета
        """
        params = {
            'company_id': company_id,
            'search_id': search_id,
            'limit': limit,
            'offset': limit * (page - 1),
            'is_async': is_async,
        }
        if language is not None:
            params['lang'] = language

        if travel_policy_id:
            params['travel_policy_id'] = travel_policy_id

        if search_filter:
            if search_filter.sort and sort is None:
                sort = search_filter.sort
                search_filter.sort = None
            params['post_filter'] = jsonable_encoder(search_filter, exclude_none=True)

        if sort:
            params['sort'] = jsonable_encoder(sort, exclude_none=True)

        return await self._make_request(
            method='get',
            path='/hotel/search/',
            params=params,
        )

    async def get_hotel_search_view(self, company_id: int, search_id: str) -> dict[str, Any]:
        """
        Получение информации об отеле из поиска (список комнат и тарифов)

        :param company_id: id юр.лица яндекса в Авиацентре
        :param search_id: это поле мы получаем в ответе get_hotel_search_results внутри hotels[]
        """
        params = {
            'company_id': company_id,
            'search_id': search_id,
        }
        return await self._make_request(
            method='get',
            path='/hotel/search/view',
            params=params,
        )

    async def get_fare_families(
            self,
            company_id: int,
            tid: int,
            travel_policy_id: int = None,
    ) -> dict[str, Any]:
        """
        Получить семейство тарифов перелета

        :param company_id: id юр.лица яндекса в Авиацентре
        :param tid: id результата поиска (data['flights'][0]['id'] ручки get_avia_search_results)
        :param travel_policy_id: id тревел-политики
        """
        params = {
            'company_id': company_id,
            'tid': tid,
        }
        if travel_policy_id:
            params['travel_policy_id'] = travel_policy_id
        return await self._make_request(
            method='get',
            path='/avia/fare-families/',
            params=params,
        )

    async def get_flight_info(
            self,
            company_id: int,
            tid: str,
            travel_policy_id: int = None,
    ) -> dict[str, Any]:
        """
        Получить детальную информацию о перелете из результата поиска

        :param company_id: id юр.лица яндекса в Авиацентре
        :param tid: id результата поиска (data['flights'][0]['id'] ручки get_avia_search_results)
        :param travel_policy_id: id тревел-политики
        """
        params = {
            'company_id': company_id,
            'tid': tid,
        }
        if travel_policy_id:
            params['travel_policy_id'] = travel_policy_id
        return await self._make_request(
            method='get',
            path='/avia/flight-info/',
            params=params,
        )

    async def get_rules(self, tid: str) -> dict[str, Any]:
        """
        Получить правила перелета/возврата/обмена
        """
        return await self._make_request(
            method='get',
            path='/avia/rules/',
            params={
                'tid': tid,
            },
        )

    async def get_travel_policies(self, company_id) -> list[dict[str, Any]]:
        """
        Получить правила перелета/возврата/обмена
        """
        return await self._make_request(
            method='get',
            path=f'/company/{company_id}/travel_policies',
            params={},
        )

    async def book_avia(self, avia_book_in: AviaBookIn) -> dict[str, Any]:
        """
        Бронирование Авиа
        """
        # Используем jsonable_encoder, чтобы поля типа date превратились в строку
        data = jsonable_encoder(avia_book_in, exclude_none=True)
        return await self._make_request(
            method='post',
            path='/avia/book',
            json=data,
        )

    async def book_hotel(self, hotel_book_in: HotelBookIn) -> dict[str, Any]:
        """
        Бронирование Отеля
        """
        # Используем jsonable_encoder, чтобы поля типа date превратились в строку
        data = jsonable_encoder(hotel_book_in, exclude_none=True)
        return await self._make_request(
            method='post',
            path='/hotel/book',
            json=data,
        )

    async def book_train(self, train_book_in: TrainBookIn) -> dict[str, Any]:
        """
        Бронирование Поезда
        """
        # Используем jsonable_encoder, чтобы поля типа date превратились в строку
        data = jsonable_encoder(train_book_in, exclude_none=True)
        return await self._make_request(
            method='post',
            path='train/book',
            json=data,
        )

    async def book_draft(self, billing_number: int) -> dict[str, Any]:
        """
        Забронировать черновик
        """
        return await self._make_request(
            method='post',
            path=f'/draft/{billing_number}/book',
        )

    async def issue_order(self, billing_number: int) -> dict[str, Any]:
        """
        Оформить заказ
        """
        return await self._make_request(
            method='post',
            path=f'/order/{billing_number}/issue',
            # В будущем оба параметра можно будет удалить - сделают необязательными
            params={
                # Ответ АЦ по этому полю:
                # в нашу бухгалтерскую систему генерируется специальный файл,
                # это внутренняя логика для агентства
                'no_sofi': 0,
                # по рекомендации АЦ это поле тоже всегда 0
                'use_private_person_invoice': 0,
            },
        )

    async def get_order(self, billing_number: int) -> dict[str, Any]:
        """
        Детализация заказа
        """
        return await self._make_request(
            method='get',
            path=f'/order/{billing_number}',
        )

    async def cancel_order(self, billing_number: int) -> dict[str, Any]:
        """
        Отмена черновика/брони
        """
        return await self._make_request(
            method='post',
            path=f'/order/{billing_number}/cancel',
        )

    async def request_refund(self, billing_number: int) -> dict[str, Any]:
        """
        Ручной возврат заказа. Будет обработан менеджером.
        Стоит вызывать только в случае, если не получилось отменить автоматикой.
        """
        return await self._make_request(
            method='post',
            path=f'/order/{billing_number}/refund-request',
        )

    async def refund_avia(self, billing_number: int) -> dict[str, Any]:
        """
        Возврат заказа
        """
        return await self._make_request(
            method='post',
            path=f'/avia/{billing_number}/refund',
        )

    async def refund_train(
        self,
        billing_number: int,
        passenger_ids: list[int],
    ) -> dict[str, Any]:
        """
        Возврат заказа
        """
        return await self._make_request(
            method='post',
            path=f'/train/{billing_number}/refund',
            params={'passenger_ids': passenger_ids},
        )

    async def refund_hotel(self, billing_number: int) -> dict[str, Any]:
        """
        Возврат заказа
        """
        return await self._make_request(
            method='post',
            path=f'/hotel/{billing_number}/refund',
        )

    async def get_stations(self, query: str, lang: str = None) -> dict[str, Any]:
        params = {
            'prefix': query,
        }
        if lang is not None:
            params['lang'] = lang
        return await self._make_request(
            method='get',
            path='/train/stations',
            params=params,
        )

    async def create_train_search(
            self,
            search_request: TrainSearchRequest,
            lang: str = None,
    ) -> dict[str, Any]:
        """
        Создать поиск по ж/д услугам
        """
        params = search_request.dict()
        if lang is not None:
            params['lang'] = lang
        return await self._make_request(
            method='get',
            path='/train/search-id',
            params=params,
        )

    async def get_train_search(
            self,
            company_id: int,
            search_id: str,
            travel_policy_id: int = None,
            lang: str = None,
            page: int = 1,
            limit: int = 20,
            search_filter: TrainSearchFilter = None,
            sort: list[TrainSort] = None,
    ) -> dict[str, Any]:
        """
        Получение результатов поиска ж/д по search_id
        """
        params = {
            'company_id': company_id,
            'search_id': search_id,
            'limit': limit,
            'offset': limit * (page - 1),
        }
        if lang is not None:
            params['lang'] = lang
        if travel_policy_id:
            params['travel_policy_id'] = travel_policy_id
        if search_filter:
            if search_filter.sort and sort is None:
                sort = search_filter.sort
                search_filter.sort = None
            params['post_filter'] = jsonable_encoder(search_filter, exclude_none=True)
        if sort:
            params['sort'] = jsonable_encoder(sort, exclude_none=True)
        return await self._make_request(
            method='get',
            path='/train/search',
            params=params,
        )

    async def get_train_info(
            self,
            company_id: int,
            search_id: str,
            train_number: str,
            departure_time: str,
            travel_policy_id: str = None,
    ) -> dict[str, Any]:
        """
        Детализация результата поиска ж/д
        """
        params = {
            'company_id': company_id,
            'search_id': search_id,
            'train_number': train_number,
            'departure_time': departure_time,
        }
        if travel_policy_id:
            params['travel_policy_id'] = travel_policy_id
        return await self._make_request(
            method='get',
            path='/train/info',
            params=params,
        )

    async def create_group_order(
        self,
        company_id: int,
    ) -> dict[str, Any]:
        """
        Создание группового заказа (корзины)
        """
        data = {
            'company_id': company_id,
            'type': 'mission',
        }
        return await self._make_request(
            method='post',
            path='/order-group/add',
            json=data,
        )

    async def add_group_order_service(
        self,
        group_order_id: int,
        service_order_id: int,
    ) -> dict[str, Any]:
        """
        Добавление услуги в групповой заказ
        """
        data = {
            'group_billing_number': group_order_id,
            'billing_number': service_order_id,
        }
        return await self._make_request(
            method='post',
            path=f'/order-group/{group_order_id}/add',
            json=data,
        )

    async def delete_group_order_service(
        self,
        group_order_id: int,
        service_order_id: int,
    ) -> dict[str, Any]:
        """
        Удаление услуги из группового заказа
        """
        data = {
            'group_billing_number': group_order_id,
            'billing_number': service_order_id,
        }
        return await self._make_request(
            method='post',
            path=f'/order-group/{group_order_id}/exclude',
            json=data,
        )

    async def close_group_order(self, group_order_id: int) -> dict[str, Any]:
        """
        Закрытие группового заказа. Закрытие переводит заказ в статус Closed и запрещает операции с услугами.
        """
        data = {
            'group_billing_number': group_order_id,
        }
        return await self._make_request(
            method='post',
            path=f'/order-group/{group_order_id}/close',
            json=data,
        )

    async def search_group_order(
        self,
        company_id: int,
        search_params: dict[str, Any],
        limit: int = 20,
        page: int = 1,
    ) -> dict[str, Any]:
        """
        Поиск группового заказа
        """
        params = {
            'company_id': company_id,
            'limit': limit,
            'offset': limit * (page - 1),
            **search_params,
        }
        return await self._make_request(
            method='get',
            path='/order-groups',
            params=params,
        )


aviacenter = AviaCenter(
    host=settings.AVIACENTER_API_URL,
    timeout=settings.AVIACENTER_TIMEOUT,
)
