from datetime import date
from pydantic import BaseModel, conlist

from intranet.trip.src.lib.aviacenter import enums
from intranet.trip.src.lib.aviacenter.fields import boolint


class TrainSegment(BaseModel):
    international_service_class: str = None  # Международный класс обслуживания
    set_electronic_registration: boolint = 1  # Требуется ли электронная регистрация
    bedding: boolint = 0  # Требуется ли постельное белье
    # Количество выкупаемых мест для вагонов, которые нужно выкупать полностью
    # (купе-переговорная, люкс)
    place_count: int = None
    place_requirements: str = None  # Дополнительные требования к выбранным местам
    search_id: str
    # У АЦ есть несколько типов сегмента ЖД:
    # - Railway - В одну сторону
    # - RailwayThere - Туда-Обратно, сегмент 'Туда'
    # - RailwayBack - Туда-Обратно, сегмент 'Обратно'
    # Учитывая, что с АК у нас сейчас ЖД только в одну сторону, есть смысл пока
    # явно проставлять тип сегмента Railway
    segment_type: str = 'Railway'
    train_number: str  # Номер поезда
    departure_time: str  # Дата отправления в формате YYYY-MM-DD hh:mm
    car_number: str  # Номер вагона
    car_type: str  # Тип вагона
    service_class: str = None  # Класс обслуживания (обязательный вместе с passengers.seats)


class TrainBonusCard(BaseModel):
    segment_type: str = 'Railway'
    type: str
    number: str


class TrainTariff(BaseModel):
    segment_type: str = 'Railway'
    code: int  # Код тарифа
    is_non_refundable: boolint = 0


class TrainPassenger(BaseModel):
    first_name: str = None  # Имя (как в документе). Обязательное если citizenship=ru
    last_name: str = None  # Фамилия (как в документе). Обязательное если citizenship=ru
    middle_name: str = None  # Отчество (как в документе)
    first_name_en: str = None
    last_name_en: str = None
    middle_name_en: str = None
    birthday: date  # День рождение пассажира в формате YYYY-MM-DD
    document_type_code: enums.DocumentTypeCode
    document_number: str  # Номер документа, как указан в документе
    document_expiration: str = None  # Срок действия документа в формате YYYY-MM-DD
    email: str = None
    phone: str = None  # Телефон пассажира в формате +7xxxxxxxxxx
    citizenship: str  # Гражданство пассажира в двухбуквенном формате (ru)
    gender: enums.PassengerGender
    type: int = 1  # Тип пассажира. 1 - Взрослый, 2 - Ребенок с местом, 3 - Ребенок без места
    send_cyrillic_to_provider: boolint = 1  # Отправлять ФИО на кириллице поставщику
    bonus_cards: list[TrainBonusCard] = None
    save_person: boolint = 0
    is_private_person: boolint = 0
    send_email: boolint = 0
    tariffs: conlist(TrainTariff, min_items=1)
    # Максимум два элемента верхенго уровня (в нашем случае всегда один)
    # первый - места туда
    # второй - места обратно
    seats: list[list[int]] = None


class TrainBookIn(BaseModel):
    contact_type: str = 'passengers'  # Об изменениях информировать пассажира
    company_id: int
    travel_policy_id: int = None
    billing_number: int = None
    group_billing_number: int = None
    # это поле нужно для статистики, потому что реальная цена может измениться
    search_price: int = None
    segments: conlist(TrainSegment, min_items=1, max_items=2)
    passengers: list[TrainPassenger]
    is_draft: boolint
