from async_clients.clients.base import BaseClient
from async_clients.auth_types import TVM2

from intranet.trip.src.lib.drive.enums import DriveWalletAction


# corporate - b2c тарифы (обязательный тег)
# rus - если хочется выбрать все города сразу (обязательный тег)
# fix_offer_regular - тариф Фикс
# Подробнее: https://wiki.yandex-team.ru/yandex.drive/carsharing/razrabotka/drivecorp/#8.5tarify
OFFERS_FILTER = 'corporate*rus*fix_offer_regular'


class DriveClient(BaseClient):

    AUTH_TYPES = {TVM2}

    async def get_user(self, phone: str):
        """
        Получение данных о пользователе по номеру телефона.

        :param phone: Номер телефона. Принимаемый формат: +79001234567
        """
        return await self._make_request(
            method='get',
            path='user/remap',
            params={
                'phones': phone,
            },
        )

    async def activate(self, org_id: str, user_id: str, tracker_issue: str):
        """
        Привязка пользователя к кошельку.

        :param org_id: Уникальный идентификатор организации в системе Драйва
        :param user_id: Id пользователя в системе Драйва
        """
        return await self._do_action(org_id, user_id, tracker_issue, DriveWalletAction.link)

    async def deactivate(self, org_id: str, user_id: str, tracker_issue: str):
        """
        Отвязка пользователя от кошелька.

        :param org_id: Уникальный идентификатор организации в системе Драйва
        :param user_id: Id пользователя в системе Драйва
        """
        return await self._do_action(org_id, user_id, tracker_issue, DriveWalletAction.unlink)

    async def _do_action(
            self,
            org_id: str,
            user_id: str,
            tracker_issue: str,
            action: DriveWalletAction,
    ):
        return await self._make_request(
            method='post',
            path='accounts/link',
            json={
                'users': [user_id],
                'name': org_id,
                'action': action,
                'active_flag': True,
                'external_id': tracker_issue,
                'account_meta': {
                    'offers_filter': OFFERS_FILTER,
                },
            }
        )

    async def get_accounts(self, org_id: str, user_id: str = None):
        """
        Получение кошельков организации (с возможностью фильтрации по пользователю).

        :param org_id: Уникальный идентификатор организации в системе Драйва
        :param user_id: Id пользователя в системе Драйва
        """
        params = {
            'name': org_id,
        }
        if user_id is not None:
            params['user_id'] = user_id
        return await self._make_request(
            method='get',
            path='accounts/list',
            params=params,
        )
