import logging

from typing import Optional

from async_clients.exceptions.base import BadResponseStatus

from intranet.trip.src.api.auth import get_tvm_service_ticket
from intranet.trip.src.config import settings
from intranet.trip.src.lib.drive.api import DriveClient


logger = logging.getLogger(__name__)


class DriveGateway:
    def __init__(self, service_ticket: str):
        self.api = DriveClient(
            host=settings.drive_api_url,
            service_ticket=service_ticket,
        )

    @classmethod
    async def init(cls):
        return cls(
            service_ticket=await get_tvm_service_ticket('drive'),
        )

    async def activate(
            self,
            org_id: str,
            user_id: str,
            tracker_issue: str,
    ) -> Optional[dict[str, list[int]]]:
        if await self.is_activated(org_id, user_id):
            return None
        try:
            return await self.api.activate(org_id, user_id, tracker_issue)
        except BadResponseStatus:
            logger.info('Cannot activate drive for org_id=%s, user_id=%s', org_id, user_id)
            return None

    async def deactivate(
            self,
            org_id: str,
            user_id: str,
            tracker_issue: str,
    ) -> Optional[dict[str, list[int]]]:
        if not await self.is_activated(org_id, user_id):
            return None
        return await self.api.deactivate(org_id, user_id, tracker_issue)

    async def get_user(self, phone: str) -> Optional[str]:
        data = await self.api.get_user(phone)
        for user in data['users']:
            if user['phone'] == phone:
                return user['id']
        return None

    async def is_activated(self, org_id: str, user_id: str) -> bool:
        accounts_list = await self.api.get_accounts(org_id, user_id)
        return len(accounts_list['accounts']) > 0
