from datetime import date
from pydantic import BaseModel, validator

from intranet.trip.src.config import settings
from intranet.trip.src.enums import ServiceType
from intranet.trip.src.models import Document


class HubBonusCard(BaseModel):
    number: str
    service_provider_type: ServiceType
    service_provider_code: str


class HubProfile(BaseModel):
    person_id: int  # это либо person_id либо ext_person_id
    login: str
    external_uid: str
    company_uid: str
    first_name_ru: str
    first_name_en: str = None
    last_name_ru: str
    last_name_en: str = None
    middle_name_ru: str = None
    middle_name_en: str = None
    gender: str
    date_of_birth: date = None
    email: str
    phone_number: int
    passports: list[Document] = None
    bonus_cards: list[HubBonusCard] = None

    @validator('gender', pre=True)
    def get_gender(cls, v):
        return v.capitalize() if v else ''

    @validator('phone_number', pre=True)
    def get_phone_number(cls, v):
        """в файле синхронизации номера принимаются числами"""
        return int(''.join([n for n in v if n.isdigit()])) if v else 0

    @validator('passports', always=True)
    def get_passports(cls, v):
        # Если список паспортов пустой, то Hub будет просто игнорировать синк документов
        if settings.ENABLE_AEROCLUB_API_DOCUMENTS_SYNC:
            return []
        return [p for p in v if p.issued_on is not None] if v else []

    @validator('bonus_cards', always=True)
    def get_bonus_cards(cls, v):
        return v or []
