from uuid import uuid4

from async_clients.clients.base import BaseClient
from async_clients.auth_types import TVM2

from intranet.trip.src.config import settings


class MessengerClient(BaseClient):

    AUTH_TYPES = {TVM2}

    def get_headers(self):
        request_id = str(uuid4())
        return {'X-REQUEST-ID': request_id}

    async def get_user(self, uid: str) -> dict:
        return await self._make_request(
            method='post',
            path='',
            json={
                'method': 'request_user',
                'params': {
                    'uid': uid,
                },
            },
        )

    async def create_chat(self, params: dict) -> dict:
        return await self._make_request(
            method='post',
            path='',
            json={
                'method': 'create_chat',
                'params': params,
            },
        )

    async def get_chat_members(self, chat_id: str) -> dict:
        return await self._make_request(
            method='post',
            path='',
            json={
                'method': 'get_chat_members',
                'params': {
                    'chat_id': chat_id,
                },
            },
        )

    async def add_member(self, uid: str, chat_id: str) -> dict:
        return await self._make_request(
            method='post',
            path='',
            json={
                'method': 'update_members',
                'params': {
                    'chat_id': chat_id,
                    'org_id': settings.MESSENGER_CHAT_ORG_ID,
                    'users': {
                        'add': [uid],
                        'remove': [],
                    },
                },
            },
        )


class BotMessengerClient(BaseClient):

    def get_auth_headers(self, *args, **kwargs) -> dict:
        prefix = 'OAuthTeam' if settings.IS_YA_TEAM else 'OAuth'
        return {
            'Authorization': f'{prefix} {settings.MESSENGER_ROBOT_OAUTH}',
        }

    def get_headers(self):
        request_id = str(uuid4())
        return {
            'X-REQUEST-ID': request_id,
            'Content-Type': 'application/json'
        }

    async def register_bot(self):
        return await self._make_request(
            method='get',
            path='team/register/',
        )

    async def update_bot(self, data: dict):
        return await self._make_request(
            method='post',
            path='team/update/',
            json=data,
        )

    async def get_info(self):
        return await self._make_request(
            method='get',
            path='team/me/',
        )

    async def send_message(self, chat_id: str, text: str, author_name: str = None):
        message_data = {
            'chat_id': chat_id,
            'text': text,
        }
        if author_name is not None:
            message_data['custom_from'] = {
                'guid': settings.MESSENGER_GUID_ROBOT_TRIP,
                'display_name': author_name,
                'is_robot': False,
            }
        return await self._make_request(
            method='post',
            path='bot/sendMessage/',
            json=message_data,
        )

    async def get_webhook_info(self) -> dict:
        return await self._make_request(
            method='get',
            path='bot/getWebhookInfo/',
        )

    async def get_file(self, file_id: str):
        return await self._make_request(
            method='get',
            path=f'bot/file/{file_id}',
            response_type='read',
        )

    async def send_file(self, chat_id, file_data):
        return await self._make_request(
            method='post',
            path='bot/sendPhoto/',
            data={
                'chat_id': chat_id,
                'photo': file_data,
            },
        )


bot_messenger = BotMessengerClient(
    host=settings.bot_messenger_url,
)
