import logging

from intranet.trip.src.api.auth import get_tvm_service_ticket
from intranet.trip.src.config import settings
from intranet.trip.src.lib.messenger.api import MessengerClient, bot_messenger


logger = logging.getLogger(__name__)


class MessengerGateway:

    def __init__(self, service_ticket: str, cache):
        self.api = MessengerClient(
            host=settings.messenger_url,
            service_ticket=service_ticket,
        )
        self.bot_api = bot_messenger
        self.cache = cache

    @classmethod
    async def init(cls, cache):
        return cls(
            service_ticket=await get_tvm_service_ticket('messenger'),
            cache=cache,
        )

    async def create_chat(self, uids: list[str], chat_name: str) -> str:
        guids = [await self.get_guid(uid) for uid in uids]
        guid_robot_trip = settings.MESSENGER_GUID_ROBOT_TRIP
        params = {
            'name': chat_name,
            'roles': {'admin': [guid_robot_trip]},
            'members': [guid_robot_trip] + guids,
            'org_id': settings.MESSENGER_CHAT_ORG_ID,
        }
        data = await self.api.create_chat(params)
        return data['data']['chat_id']

    async def stop_chat(self, chat_id: str):
        pass

    async def get_guid(self, uid: str) -> str:
        cache_key = f'messenger_guid:{uid}'
        guid = await self.cache.get(cache_key)
        if guid is not None:
            return guid
        data = await self.api.get_user(uid)
        guid = data['data']['user']['guid']
        await self.cache.set(
            key=cache_key,
            data=guid,
            expire_after=settings.MESSENGER_GUID_CACHE_EXPIRE_AFTER,
        )
        return guid

    async def send_message(self, chat_id: str, text: str):
        data = {
            'chat_id': chat_id,
            'text': text,
        }
        await self.bot_api.send_message(data)

    async def get_member_uids(self, chat_id: str) -> list[str]:
        data = await self.api.get_chat_members(chat_id)
        return [user['guid'] for user in data['data']['users']]

    async def add_member_to_chat(self, uid: str, chat_id: str):
        user_guid = await self.get_guid(uid)
        if user_guid in await self.get_member_uids(chat_id):
            return
        await self.api.add_member(user_guid, chat_id)
