import logging
from jinja2 import Template
from library.python import resource

from intranet.trip.src.api.auth import get_tvm_service_ticket
from intranet.trip.src.lib.messenger.api import bot_messenger
from intranet.trip.src.lib.messenger.gateway import MessengerGateway
from intranet.trip.src.cache import Cache
from intranet.trip.src.unit_of_work import UnitOfWork


logger = logging.getLogger(__name__)


class MessengerSync:
    """Класс, в котором описана логика создания и отмены чатов"""
    def __init__(self, uow: UnitOfWork, cache: Cache, messenger_service_ticket):
        self.uow = uow
        self.messenger_gateway = MessengerGateway(
            service_ticket=messenger_service_ticket,
            cache=cache,
        )

        path = 'intranet/trip/src/lib/messenger/start_message.jinja2'
        self.template_start_message = resource.resfs_read(path).decode('utf-8')

    @classmethod
    async def init(cls, uow: UnitOfWork, cache: Cache):
        return cls(
            uow=uow,
            cache=cache,
            messenger_service_ticket=await get_tvm_service_ticket('messenger'),
        )

    async def create_or_update_chats(self) -> None:
        person_trips = await self.uow.person_trips.get_person_trips_without_chat_id()
        logger.info('%d chats to create', len(person_trips))
        for person_trip in person_trips:
            try:
                await self._create_chat(
                    trip_id=person_trip.trip_id,
                    person_id=person_trip.person_id,
                    person_uid=person_trip.person.uid,
                )
            except Exception as err:
                logger.error(
                    'Failed to create chat for trip_id=%s, person_id=%s: %s',
                    person_trip.trip_id,
                    person_trip.person_id,
                    err
                )

    async def _create_chat(self, trip_id: int, person_id: int, person_uid: str = None) -> None:
        person_trip = await self.uow.person_trips.get_detailed_person_trip(trip_id, person_id)
        if person_uid is None:
            if person_trip.chat_id:
                logger.error(
                    'Failed to create chat for person_trip: trip_id=%d person_id=%d, has chat_id',
                    trip_id, person_id,
                )
                return
            person_uid = person_trip.person.uid

        chat_id = await self.messenger_gateway.create_chat([person_uid], 'Trip')
        await self.uow.person_trips.update(
            trip_id=trip_id,
            person_id=person_id,
            chat_id=chat_id,
        )
        trip = await self.uow.trips.get_trip(trip_id=trip_id)
        text = Template(self.template_start_message).render(trip=trip, person_trip=person_trip)
        await bot_messenger.send_message(
            chat_id=chat_id,
            text=text,
        )

    async def create_chats(self, trip_id: int, person_ids: list[int]) -> None:
        for person_id in person_ids:
            try:
                await self._create_chat(trip_id, person_id)
            except Exception as err:
                logger.error(
                    'Failed to create chat for trip_id=%s, person_id=%s: %s',
                    trip_id,
                    person_id,
                    err
                )
