import os
import logging

from asyncssh.constants import FILEXFER_TYPE_REGULAR
from yt.wrapper.ypath import ypath_join

from intranet.trip.src.lib.sftp_client.api import sftp_client
from intranet.trip.src.lib.sftp_client.exceptions import PermissionDenied, SFTPNoSuchFile
from intranet.trip.src.lib.xml_parser import RegistryParser, XmlParserError
from intranet.trip.src.lib.yt import YtRegistry, RegistryRow, YtError


logger = logging.getLogger(__name__)


async def registry_batch_upload(date: str, host: str, login: str, password: str, yt_prefix: str):
    sync_obj = RegistrySync(date, host, login, password, yt_prefix)
    try:
        await sync_obj.batch_upload()
    except PermissionDenied:
        logger.error('Permission denied: %s@%s/%s', login, host, date)
    except (FileNotFoundError, SFTPNoSuchFile):
        logger.error('File not found: %s', date)
    except YtError as e:
        logger.error(e.message)
    except XmlParserError as e:
        logger.error('Parse error %s', e)


class RegistrySync:
    """ Класс для выполнения синхронизации реестров по SFTP """
    def __init__(
        self,
        date: str,
        host: str,
        login: str,
        password: str,
        yt_prefix: str,
    ):
        self._date = date
        self.sftp_client = sftp_client(host, login, password)
        self._yt_registry = YtRegistry(prefix=yt_prefix)

    async def _parse_and_upload(self, table: str, contents: list):
        xml_parsers = [await RegistryParser(raw_xml=content).parser() for content in contents]
        rows = [
            row
            async for row in RegistryParser.rows_iterator(
                xml_parsers=xml_parsers,
                type_row=RegistryRow,
            )
        ]
        await self._yt_registry.write_rows(table=table, rows=rows)

    async def upload(self):
        """ Upload xml-registry-files from subdir """
        async with self.sftp_client as sftp:
            for file in await sftp.readdir(self._date):
                if file.attrs.type != FILEXFER_TYPE_REGULAR or not file.filename.endswith('.xml'):
                    continue
                xml_file = await sftp.open(path=os.path.join(self._date, file.filename))
                # one file - one table
                table = ypath_join(
                    self._date,
                    file.filename.replace(' ', '_'),
                )
                try:
                    await self._parse_and_upload(
                        table=table,
                        contents=[await xml_file.read()],
                    )
                except XmlParserError as e:
                    logger.error('Parse error %s', e)
                    continue

    async def batch_upload(self):
        """ Upload all registry in one table """
        contents = []
        async with self.sftp_client as sftp:
            for file in await sftp.readdir(self._date):
                if file.attrs.type != FILEXFER_TYPE_REGULAR or not file.filename.endswith('.xml'):
                    continue
                xml_file = await sftp.open(path=os.path.join(self._date, file.filename))
                contents.append(await xml_file.read())
        # upload all rows in one table
        await self._parse_and_upload(
            table=self._date,
            contents=contents,
        )

    async def get_uploaded(self) -> list:
        """ Получить список уже загруженных реестров """
        raise NotImplementedError
