import logging
from typing import Optional

from tenacity import (
    retry,
    retry_if_result,
    wait_exponential,
    stop_after_attempt,
    RetryCallState,
)

from async_clients.exceptions.base import BaseClientsException

from intranet.trip.src.config import settings
from intranet.trip.src.api.auth import get_tvm_service_ticket
from intranet.trip.src.api.schemas import TripStaffFilter
from intranet.trip.src.models import Person, Trip, TripFull
from intranet.trip.src.lib.utils import safe_getitem
from intranet.trip.src.lib.staffapi.gateway import StaffApiGateway
from intranet.trip.src.lib.staff.converters import StaffToTripConverter, TripToStaffConverter
from intranet.trip.src.lib.staff.api import StaffClient


logger = logging.getLogger(__name__)


def has_fallback_assignment(assignment_response: dict) -> bool:
    """
    Проверяет ответ стаффа про assignments на наличие fallback значения -999.
    :return: true, если такое значение найдено, иначе false
    """
    if not assignment_response:
        logger.warning('Something happened when getting assignment from staff')
        return True
    for login, assignment_data in assignment_response.items():
        if safe_getitem(assignment_data, ['choices', 0, 'value'], -999) == -999:
            logger.warning(
                'Something happened when getting assignment from staff for login %s',
                login,
            )
            return True
    return False


def return_last_attempt_result(retry_state: RetryCallState) -> dict[str, dict]:
    """
    Логгирует ошибку в случае падения всех ретраев и возвращает результат последней попытки
    """
    logger.error(
        'Cannot get assignments after %s attempts, returning result from last attempt',
        retry_state.attempt_number,
    )
    return retry_state.outcome.result()


class StaffGateway:
    """
    Класс, который инкапсулирует в себе всю работу с api Стаффа
    """
    def __init__(self, service_ticket=None, user_ticket=None):
        self.api = StaffClient(
            host=settings.staff_url,
            service_ticket=service_ticket,
            user_ticket=user_ticket,
        )

    @classmethod
    async def init(cls, user_ticket=None):
        return cls(
            service_ticket=await get_tvm_service_ticket('staff'),
            user_ticket=user_ticket,
        )

    async def _get_persons_map(self, data: list[dict]) -> dict[int, Person]:
        staff_ids = set()
        for item in data:
            staff_ids.add(item['author'])
            for employee in item['employee_list']:
                staff_ids.add(employee['employee'])

        staffapi_gateway = await StaffApiGateway.init()
        persons_map: dict[int, Person] = {}
        if staff_ids:
            persons_map = await staffapi_gateway.get_persons(list(staff_ids))
        return persons_map

    def _get_trip_filter_params(self, fltr: TripStaffFilter) -> dict:
        params = {}
        if fltr.date_from:
            params['date_from'] = fltr.date_from.strftime('%Y-%m-%d')
        if fltr.date_to:
            params['date_to'] = fltr.date_to.strftime('%Y-%m-%d')
        if fltr.event_type:
            params['event_type'] = fltr.event_type
        if fltr.participants:
            params['participants'] = fltr.participants
        if fltr.page:
            params['_page'] = fltr.page
        if fltr.limit:
            params['_limit'] = fltr.limit
        return params

    async def get_list_and_count(self, fltr: TripStaffFilter) -> tuple[list[Trip], int]:
        params = self._get_trip_filter_params(fltr)
        try:
            response_data = await self.api.get_trips(params=params)
        except BaseClientsException:
            return [], 0

        persons_map = await self._get_persons_map(response_data['result'])
        trips = StaffToTripConverter().convert(
            staff_trips=response_data['result'],
            persons_map=persons_map,
        )
        count = response_data['total']

        return trips, count

    async def create_or_update_trip(self, trip: TripFull) -> Optional[str]:
        data = TripToStaffConverter(trip).as_create_json()
        params = {
            'login': trip.author.login,
            'event_type': trip.event_type.value,
        }
        if trip.staff_trip_uuid:
            params['trip_uuid'] = trip.staff_trip_uuid
        try:
            response_data = await self.api.save_trip(data, params)
        except BaseClientsException:
            return None
        if response_data.get('errors'):
            logger.error('response_data: %s, trip_id=%s', response_data, trip.trip_id)
        return response_data['content'].get('trip_uuid')

    async def get_assignment(self, login: str) -> dict:
        data = await self.get_assignments([login])
        return safe_getitem(data, [login, 'choices', 0], -999)

    @retry(
        retry=retry_if_result(has_fallback_assignment),
        wait=wait_exponential(settings.STAFF_GET_ASSIGNMENT_WAIT_MULTIPLIER),
        stop=stop_after_attempt(settings.STAFF_GET_ASSIGNMENT_RETRIES),
        retry_error_callback=return_last_attempt_result,
    )
    async def get_assignments(self, logins: list[str]) -> dict[str, dict]:
        params = {'logins': ','.join(logins)}

        try:
            response_data = await self.api.get_assignments(params=params)
        except BaseClientsException:
            return {}

        return response_data

    async def get_trips_by_uuids(self, staff_trip_uuids: list[str]) -> list[dict]:
        data = {'trip_uuids': staff_trip_uuids}

        try:
            response_data = await self.api.get_trips_by_uuids(data=data)
        except BaseClientsException:
            return []

        return response_data
