import logging
from typing import Optional
from dataclasses import dataclass
from datetime import timedelta, date

from intranet.trip.src.api.auth import get_tvm_service_ticket
from intranet.trip.src.lib.utils import safe_getitem
from intranet.trip.src.lib.staffapi.gateway import StaffApiGateway
from intranet.trip.src.lib.staff.gateway import StaffGateway
from intranet.trip.src.models import TripFull, Trip
from intranet.trip.src.unit_of_work import UnitOfWork
from intranet.trip.src.enums import Provider


logger = logging.getLogger(__name__)


@dataclass
class StaffTrip:
    staff_trip_uuid: str
    conf_issue: Optional[str]
    travel_issue: Optional[str]

    @classmethod
    def from_staff_raw(cls, raw: dict):
        return cls(
            staff_trip_uuid=raw['uuid'],
            travel_issue=safe_getitem(raw, ['trip_issue', 'key']),
            conf_issue=safe_getitem(raw, ['conf_issue', 'key']),
        )


@dataclass
class DBTrip(StaffTrip):
    trip_id: int


@dataclass
class StaffPersonTrip:
    staff_trip_uuid: str
    login: str
    conf_issue: Optional[str]
    travel_issue: Optional[str]
    is_approved: bool

    @classmethod
    def from_staff_raw(cls, uuid: str, raw: dict):
        login = (
            safe_getitem(raw, ['conf_issue', 'employee', 'id'])
            or safe_getitem(raw, ['trip_issue', 'employee', 'id'])
        )
        is_approved = 'Согласовано' in (
            safe_getitem(raw, ['trip_issue', 'approvementStatus']),
            safe_getitem(raw, ['conf_issue', 'approvementStatus']),
        )
        return cls(
            staff_trip_uuid=uuid,
            login=login,
            travel_issue=safe_getitem(raw, ['trip_issue', 'key']),
            conf_issue=safe_getitem(raw, ['conf_issue', 'key']),
            is_approved=is_approved,
        )


@dataclass
class DBPersonTrip(StaffPersonTrip):
    trip_id: int
    person_id: int
    is_offline: bool
    provider: Provider
    gap_date_from: date
    gap_date_to: date
    group_travel_issue: Optional[str]
    group_conf_issue: Optional[str]


class StaffTripPull:
    """Класс, в котором описана логика синхронизации данных командировок из Staff в Trip"""
    def __init__(self, uow: UnitOfWork, staff_gateway: StaffGateway):
        self.uow = uow
        self.staff_gateway = staff_gateway

    @classmethod
    async def init(cls, uow: UnitOfWork):
        staff_gateway = StaffGateway(service_ticket=await get_tvm_service_ticket('staff'))
        return cls(uow=uow, staff_gateway=staff_gateway)

    async def pull_staff_trips(self) -> None:
        """
        Подтягивание тикетов и признака согласованности из Стаффа в Командировки
        """
        raw_person_trips = await self.uow.person_trips.get_person_trips_for_staff_pull()
        db_person_trips = [DBPersonTrip(**item) for item in raw_person_trips]
        logger.info('[StaffTripPull] %s person_trips to sync in staff', len(db_person_trips))
        if not db_person_trips:
            return

        staff_trip_uuids = list({item.staff_trip_uuid for item in db_person_trips})
        staff_data = await self.staff_gateway.get_trips_by_uuids(staff_trip_uuids)
        if not staff_data:
            return

        db_trips_by_uuid = {}
        db_person_trips_by_uuid_login = {}
        for pt in db_person_trips:
            uuid = pt.staff_trip_uuid
            db_person_trips_by_uuid_login[(uuid, pt.login)] = pt
            if uuid not in db_trips_by_uuid:
                db_trips_by_uuid[uuid] = DBTrip(
                    trip_id=pt.trip_id,
                    staff_trip_uuid=pt.staff_trip_uuid,
                    travel_issue=pt.group_travel_issue,
                    conf_issue=pt.group_conf_issue,
                )

        for staff_item in staff_data:
            staff_trip = StaffTrip.from_staff_raw(staff_item)
            uuid = staff_trip.staff_trip_uuid

            db_trip = db_trips_by_uuid.get(uuid)
            if not db_trip:
                return

            try:
                await self.update_trip_in_db(db_trip, staff_trip)
            except Exception:
                logger.exception('Error pull trip from staff, trip_id: %s', db_trip.trip_id)

            for employee in safe_getitem(staff_item['employee_list'], []):
                staff_pt = StaffPersonTrip.from_staff_raw(uuid=uuid, raw=employee)
                db_pt = db_person_trips_by_uuid_login.get((uuid, staff_pt.login))
                if not db_pt:
                    continue
                try:
                    await self.update_person_trip_in_db(db_pt, staff_pt)
                except Exception:
                    logger.exception(
                        'Error pull person trip from staff, trip_id: %s, person_id: %s',
                        db_pt.trip_id,
                        db_pt.person_id,
                    )
                    continue

    async def update_trip_in_db(self, db_trip: DBTrip, staff_trip: StaffTrip):
        if db_trip.travel_issue is None and staff_trip.travel_issue is not None:
            await self.uow.trips.update(
                trip_id=db_trip.trip_id,
                issue_travel=staff_trip.travel_issue,
            )
            logger.info(
                'Travel issue %s was pulled from staff for trip %d',
                staff_trip.travel_issue,
                db_trip.trip_id,
            )

        if db_trip.conf_issue is None and staff_trip.conf_issue is not None:
            await self.uow.trips.update_conf_details(
                trip_id=db_trip.trip_id,
                tracker_issue=staff_trip.conf_issue,
            )
            logger.info(
                'Conf issue %s was pulled from staff for trip %d',
                staff_trip.conf_issue,
                db_trip.trip_id,
            )

    async def update_person_trip_in_db(self, db_pt: DBPersonTrip, staff_pt: StaffPersonTrip):
        trip_id = db_pt.trip_id
        person_id = db_pt.person_id
        is_travel_issue_updated = False

        if db_pt.travel_issue is None and staff_pt.travel_issue is not None:
            await self.uow.person_trips.update_travel_details(
                trip_id=trip_id,
                person_id=person_id,
                tracker_issue=staff_pt.travel_issue,
                taxi_date_from=db_pt.gap_date_from - timedelta(days=1),
                taxi_date_to=db_pt.gap_date_to + timedelta(days=1),
                city_id=1,  # временный костыль
                is_created_on_provider=False,  # временный костыль
            )
            is_travel_issue_updated = True
            logger.info(
                'Travel issue %s was pulled from staff for person_trip (%d, %d)',
                staff_pt.travel_issue,
                db_pt.trip_id,
                db_pt.person_id,
            )

        if db_pt.conf_issue is None and staff_pt.conf_issue is not None:
            await self.uow.person_trips.update_conf_details(
                trip_id=trip_id,
                person_id=person_id,
                tracker_issue=staff_pt.conf_issue,
            )
            logger.info(
                'Conf issue %s was pulled from staff for person_trip (%d, %d)',
                staff_pt.conf_issue,
                db_pt.trip_id,
                db_pt.person_id,
            )

        if not db_pt.is_approved and staff_pt.is_approved:
            await self.uow.person_trips.update(
                trip_id=trip_id,
                person_id=person_id,
                is_approved=True,
            )
            logger.info(
                'Approvement status was pulled from staff for person_trip (%d, %d)',
                db_pt.trip_id,
                db_pt.person_id,
            )

        if is_travel_issue_updated and db_pt.provider == Provider.aeroclub:
            if db_pt.is_offline:
                await self.uow.run_job(
                    'notify_offline_person_trip_created_task',
                    trip_id=db_pt.trip_id,
                    person_id=db_pt.person_id,
                )
            else:
                await self.uow.run_job(
                    'update_trip_custom_properties',
                    trip_id=trip_id,
                    person_id=person_id,
                )


class StaffTripPush:
    """Класс, в котором описана логика пуша данных командировок из Trip в Staff"""
    def __init__(self, uow: UnitOfWork, staff_service_ticket, staffapi_service_ticket):
        self.uow = uow
        self.staff_gateway = StaffGateway(service_ticket=staff_service_ticket)
        self.staffapi_gateway = StaffApiGateway(service_ticket=staffapi_service_ticket)

    @classmethod
    async def init(cls, uow: UnitOfWork):
        return cls(
            uow=uow,
            staff_service_ticket=await get_tvm_service_ticket('staff'),
            staffapi_service_ticket=await get_tvm_service_ticket('staff-api'),
        )

    async def _enrich_person_trips(self, person_trips_data: list[dict]):
        """
        Обогощаем person_trips из базы данными о oebs-назначениях, staff_id и logins
        """
        uids: list[str] = [pt['person']['uid'] for pt in person_trips_data]
        # TODO: Тут можно одним запросов в staffapi обойтись
        uid_to_login = await self.staffapi_gateway.get_logins_by_uids(uids)
        uid_to_id = await self.staffapi_gateway.get_ids_by_uids(uids)

        assignments_data = await self.staff_gateway.get_assignments(logins=uid_to_login.values())

        missed_logins = set(uid_to_login.values()) - set(assignments_data.keys())
        if missed_logins:
            raise ValueError(f'Error getting assignments for {missed_logins}')

        uid_to_assignment = {
            uid: assignments_data.get(uid_to_login[uid])
            for uid in uids
        }
        for person_trip_data in person_trips_data:
            uid = person_trip_data['person']['uid']
            person_trip_data['person']['staff_id'] = uid_to_id[uid]

            assignment = uid_to_assignment.get(uid, '')
            if assignment:
                assignment = safe_getitem(assignment, ['choices', 0, 'value'], -999)
            person_trip_data['employee_assignment'] = assignment

    async def enrich_trip(self, trip: Trip) -> TripFull:
        """
        Собирает объект TripFull на основе данных:
          - из базы
          - id сотрудников из Стаффа
          - oebs-назначения через прокси-ручку на Стаффе
        """
        trip = trip.dict()
        if trip['person_trips']:
            await self._enrich_person_trips(trip['person_trips'])
        return TripFull(**trip)

    async def _create_or_update_staff_trip(self, trip: Trip):
        trip_full = await self.enrich_trip(trip)
        trip_uid = await self.staff_gateway.create_or_update_trip(trip_full)
        if not trip.staff_trip_uuid:
            await self.uow.trips.update(trip_id=trip.trip_id, staff_trip_uuid=trip_uid)

    async def create_or_update_staff_trip(self, trip_id: int) -> None:
        async with self.uow:
            trip = await self.uow.trips.get_trip_for_staff_push(trip_id=trip_id)
            if trip is not None:
                await self._create_or_update_staff_trip(trip)

    async def create_staff_trips(self) -> None:
        """
        Создаем командировки на Стаффе
        """
        trip_ids = await self.uow.trips.get_trip_ids_for_staff_push()
        logger.info('[StaffTripPush] %d trips to create in staff', len(trip_ids))

        for trip_id in trip_ids:
            try:
                await self.create_or_update_staff_trip(trip_id)
            except Exception:
                logger.exception(
                    'Error during create staff trip for trip_id=%s',
                    trip_id,
                )
                continue
