import logging
from collections.abc import AsyncIterable
from typing import Any, Awaitable, Callable

from intranet.trip.src.api.auth import get_tvm_service_ticket
from intranet.trip.src.models import Person
from intranet.trip.src.enums import Gender
from intranet.trip.src.config import settings
from intranet.trip.src.lib.staffapi.api import StaffApiClient

logger = logging.getLogger(__name__)


class StaffApiGateway:

    # маппинг полов стаффапи на наши
    gender_map = {
        'male': Gender.male,
        'female': Gender.female,
    }

    # запрашиваемые поля коллекции persons в staff-api
    staffapi_person_fields = [
        'id',
        'login',
        'uid',
        'name',

        'environment.timezone',
        'language.ui',
        'personal.gender',
        'phones',
        'official.organization',
        'official.is_dismissed',
        'official.affiliation',
        'work_email',

        'chiefs.id',
        'chiefs.uid',
        'chiefs.login',
    ]

    staffapi_organization_fields = [
        'id',
        'name',
    ]

    def __init__(self, service_ticket: str, user_ticket: str = None):
        self.api = StaffApiClient(
            host=settings.staff_api_url,
            service_ticket=service_ticket,
            user_ticket=user_ticket,
        )

    @classmethod
    async def init(cls, user_ticket: str = None):
        return cls(
            service_ticket=await get_tvm_service_ticket('staff-api'),
            user_ticket=user_ticket,
        )

    async def get_person(self, staff_id: int) -> Person:
        persons = await self.get_persons([staff_id])
        return persons[staff_id]

    async def get_persons_by_uids(self, uids: list[str]) -> dict[str, Person]:
        return await self._get_persons('uid', uids)

    async def get_person_by_uid(self, uid: str) -> Person:
        persons = await self.get_persons_by_uids([uid])
        return persons[uid]

    async def get_persons(self, ids: list[int]) -> dict[int, Person]:
        return await self._get_persons('id', ids)

    async def _get_chiefs(self, field_name: str, field_value) -> list[Person]:
        params = {
            field_name: field_value,
            '_limit': 1,
            '_fields': ','.join([
                'id',
                'login',
                'uid',
                *[f'chiefs.{field}' for field in self.staffapi_person_fields],
            ])
        }
        data = await self.api.get_person(params)
        return [
            Person(
                uid=chief_dict['uid'],
                login=chief_dict['login'],
                first_name=chief_dict['name']['first']['ru'],
                last_name=chief_dict['name']['last']['ru'],
                first_name_en=chief_dict['name']['first']['en'],
                last_name_en=chief_dict['name']['last']['en'],
                staff_id=chief_dict['id'],
            )
            for chief_dict in data['chiefs']
        ]

    async def get_chiefs(self, uid: str) -> list[Person]:
        return await self._get_chiefs('uid', uid)

    async def _get_persons_data(self, field_name: str, values: list) -> list[dict]:
        params = {
            field_name: ','.join(map(str, values)),
            '_limit': 10000,
            '_fields': ','.join(self.staffapi_person_fields)
        }
        data = await self.api.get_persons(params)
        persons_data = data['result']
        return persons_data

    async def _paginate_staff_data(
            self,
            staff_data_getter_coroutine: Callable[[dict], Awaitable[dict]],
            page_size: int,
            fields: list[str],
            params: dict[str, Any] = None,
    ):
        if params is None:
            params = {}

        query_params = {
            '_limit': page_size,
            '_fields': ','.join(fields),
            '_nopage': 1,
            '_sort': 'id',
            **params,
        }
        while True:
            chunk_data = await staff_data_getter_coroutine(query_params)
            chunk_data = chunk_data['result']
            if not chunk_data:
                break
            yield chunk_data

            last_id = chunk_data[-1]['id']
            query_params['_query'] = f'id>{last_id}'

    async def get_persons_by_chunks(
            self,
            page_size: int = 1000,
            fields: list[str] = None,
    ) -> AsyncIterable[list[dict]]:
        """Чанками yield'ит данные по ВСЕМ НЕУВОЛЕННЫМ сотрудникам из staff-api"""
        fields = fields or self.staffapi_person_fields
        params = {
            'official.is_dismissed': 'false'
        }
        person_chunks_coro = self._paginate_staff_data(
            staff_data_getter_coroutine=self.api.get_persons,
            page_size=page_size,
            fields=fields,
            params=params,
        )
        async for person_chunk in person_chunks_coro:
            yield person_chunk

    async def _get_persons(self, field_name: str, values: list) -> dict[Any, Person]:
        persons_data = await self._get_persons_data(field_name, values)
        result = {}
        for person_record in persons_data:
            phone_number = self.extract_phone_number(person_record['phones'])

            first_name = person_record['name']['first']
            last_name = person_record['name']['last']
            middle_name = person_record['name']['middle']
            language = person_record['language']['ui']
            gender = person_record['personal']['gender']
            organization_id = person_record['official']['organization']['id']

            result[person_record[field_name]] = Person(
                uid=person_record['uid'],
                login=person_record['login'],
                staff_id=person_record['id'],
                gender=self.gender_map[gender].name,
                timezone=person_record['environment']['timezone'],
                language=language,
                phone_number=phone_number,
                company_id=organization_id,
                first_name=first_name['ru'],
                last_name=last_name['ru'],
                middle_name=middle_name,
                first_name_en=first_name['en'],
                last_name_en=last_name['en'],
                middle_name_en=None,
                email=person_record['work_email'],
                is_dismissed=person_record['official']['is_dismissed'],
                is_limited_access=False,
            )
        return result

    @staticmethod
    def extract_phone_number(numbers: list[dict]) -> str:
        """
        Выбирает основной номер телефона.
        Исходим из того, что в staff-api всегда один номер помечен основным
        """
        for number_data in numbers:
            if number_data['is_main']:
                return number_data['number']

    async def get_fields_by_uids(self, uids: list[str], field_name: str) -> dict[str, Any]:
        params = {
            'uid': ','.join(uids),
            '_limit': 10000,
            '_fields': ','.join([
                'uid',
                field_name,
            ]),
        }
        response_data = await self.api.get_persons(params)
        return {
            rd['uid']: rd[field_name]
            for rd in response_data['result']
        }

    async def get_logins_by_uids(self, uids: list[str]) -> dict[str, str]:
        return await self.get_fields_by_uids(uids, field_name='login')

    async def get_ids_by_uids(self, uids: list[str]) -> dict[str, int]:
        return await self.get_fields_by_uids(uids, field_name='id')

    async def get_organizations_by_chunks(self, page_size: int = 10000):
        async for organization_chunk in self._paginate_staff_data(
            self.api.get_organizations,
            page_size,
            self.staffapi_organization_fields
        ):
            yield organization_chunk
