import base64
import json
import logging

from collections.abc import Iterable
from typing import Sized, AnyStr, Union

logger = logging.getLogger(__name__)


def dict_to_b64json(data: dict) -> str:
    return base64.b64encode(json.dumps(data).encode()).decode()


def b64json_to_dict(data: str) -> dict:
    return json.loads(base64.b64decode(data))


def safe_getitem(data, path: list[Union[str, int]], default=None):
    try:
        for key in path:
            data = data[key]
    except (KeyError, IndexError, TypeError):
        return default
    return data


def paginate(iterable: Sized, by: int = 10) -> Iterable[Iterable]:
    index = 0
    count = len(iterable)
    while index < count:
        yield iterable[index:index + by]
        index += by


def is_hashable(value):
    if isinstance(value, (list, dict)):
        return True
    try:
        hash(value)
    except TypeError:
        return False
    return True


def freeze(value):
    if not is_hashable(value):
        return None
    if isinstance(value, dict):
        freeze_nested = {
            k: freeze(v)
            for k, v in value.items()
        }
        return frozenset(freeze_nested.items())
    if isinstance(value, (list, tuple, set)):
        return tuple([freeze(item) for item in value])
    return value


def make_hash(value):
    return str(hash(freeze(value)))


def mask_value(
    value: AnyStr,
    placeholder: str = '*',
    visible_prefix_len: int = 0,
    visible_suffix_len: int = 0,
) -> AnyStr:
    """
    Маскирует часть строки, заменяя ее определенными символами.

    :param value: строка, часть которой необходимо скрыть
    :param placeholder: символ, на который нужно заменить скрытые символы
    :param visible_prefix_len: количество символов в начале строки,
                               которые не скрываются
    :param visible_suffix_len: количество символов в конце строки,
                               которые не скрываются
    """
    if value is None:
        return ''
    if visible_prefix_len < 0 or visible_suffix_len < 0:
        raise ValueError('prefix or suffix length is less than 0')
    if visible_prefix_len >= len(value):
        visible_prefix_len = len(value) // 2
    if visible_suffix_len >= len(value):
        visible_suffix_len = len(value) // 2
    masked_chars_count = len(value) - visible_prefix_len - visible_suffix_len
    if masked_chars_count <= 0:
        return value
    prefix = value[:visible_prefix_len]
    suffix = value[-visible_suffix_len:] if visible_suffix_len else ''
    masked = placeholder * masked_chars_count
    return prefix + masked + suffix


DEFAULT_LANG = 'en'


def get_by_lang(data: dict[str, str], lang: str):
    """
    Get data from structure like:
    {
        'en': 'Tomatoes',
        'ru': 'Помидоры',
    }
    """
    if not data:
        return ''
    res = data.get(lang) or data.get(DEFAULT_LANG)
    if not res:
        for v in data.values():
            if v:
                return v
    return res or ''


def unify_values(d: dict, keys: list[str]) -> dict:
    """
    Warning: изменяет входной словарь!
    Выбирает одно из значений keys, которое не является None и проставляет его во все keys
    Используется для того, чтобы переименовать поле в апи с обратной совместимостью
    >>> unify_values({'a':1}, ['b', 'a'])
    {'a': 1, 'b': 1}
    >>> unify_values({'a':1, 'b': 2}, ['b', 'a'])
    {'a': 2, 'b': 2}
    >>> unify_values({'a':1, 'b': None}, ['b', 'a'])
    {'a': 1, 'b': 1}
    >>> unify_values({}, ['b', 'a'])
    {'a': None, 'b': None}
    >>> unify_values({'a': 1, 'c': 3}, ['b', 'a'])
    {'a': 1, 'b': 1, 'c': 3}
    """
    not_none_value = next(
        (val for k in keys if (val := d.get(k)) is not None),
        None,
    )
    for k in keys:
        d[k] = not_none_value
    return d
