import logging

from asgiref.sync import sync_to_async
from typing import Any
from yt.wrapper import YtClient
from yt.wrapper.mappings import VerifiedDict

from intranet.trip.src.lib.yt.exceptions import YtError


logger = logging.getLogger(__name__)


def sync_connect(config: VerifiedDict) -> YtClient:
    try:
        client = YtClient(config=config)
        logger.info('Yt successfully connection %s', config['proxy']['url'])
    except Exception as e:
        logger.error('Yt connection failed %s', config['proxy']['url'])
        raise YtError(message=str(e))
    return client


def sync_write_rows(client: YtClient, row_type: Any, table: str, rows: list):
    try:
        client.write_table_structured(
            table=table,
            row_type=row_type,
            input_stream=rows,
        )
        logger.info('Yt wrote %d rows to %s', len(rows), table)
    except Exception as e:
        logger.error('Yt writing to %s failed', table)
        raise YtError(message=str(e))


def sync_read_rows(client: YtClient, row_type: Any, table: str):
    try:
        it = client.read_table_structured(
            table=table,
            row_type=row_type,
        )
        rows = list(it)
        logger.info('Yt read %d rows from %s', len(rows), table)
    except Exception as e:
        logger.error('Yt reading from %s failed', table)
        raise YtError(message=str(e))
    return rows


def sync_insert_rows(client: YtClient, table: str, rows: list):
    try:
        client.insert_rows(
            table=table,
            input_stream=rows,
        )
        logger.info('Yt inserted %d rows to %s', len(rows), table)
    except Exception as e:
        logger.error('Yt add rows to %s failed', table)
        raise YtError(message=str(e))


@sync_to_async(thread_sensitive=True)
def write_rows(row_type: Any, table: str, rows: list, config: VerifiedDict):
    """
    YT не является потокобезопасной, поэтому все действия нужно проводить в одном потоке
    """
    client = sync_connect(config)
    sync_write_rows(client, row_type, table, rows)
