import logging

from intranet.trip.src.models import PersonTrip
from intranet.trip.src.lib.aeroclub.models.custom_properties import CustomProperty

logger = logging.getLogger(__name__)


class CustomPropertyFetcher:
    """
    Извлекает из CustomProperty значение на основе PersonTrip, которое нужно прокинуть в Аэроклуб
    """
    fallback_value = None

    def __init__(self, person_trip: PersonTrip, custom_property: CustomProperty):
        self.person_trip = person_trip
        self.custom_property = custom_property

    def get_fallback_value(self):
        if self.custom_property.is_list:
            return self.custom_property.values[0].id
        return self.fallback_value

    def fetch(self):
        """Получить значение custom_property (id или текст) на основе person_trip"""
        raise NotImplementedError

    def find_value_id(self, name):
        """
        Получить id конкретного варианта из списка значений custom_property
        за счет матчинга по его названию
        """
        for value in self.custom_property.values:
            if value.name.ru == name:
                return value.id
        logger.error(
            'Did not find value %s for custom property %s (trip_id=%s, person_id=%s)',
            name,
            self.custom_property.name.en,
            self.person_trip.trip_id,
            self.person_trip.person_id,
        )


class GradeFetcher(CustomPropertyFetcher):

    def fetch(self):
        purposes = self.person_trip.purposes or []
        grades = [p.aeroclub_grade for p in purposes if p.aeroclub_grade]
        grade = str(max(grades)) if grades else '1'
        return self.find_value_id(grade) or self.get_fallback_value()


class PurposeFetcher(CustomPropertyFetcher):

    def fix_purpose_name(self, name):
        if name == 'Внешняя конференция':
            return 'Внешняя конференция. Слушатель'
        return name


class FirstPurposeFetcher(PurposeFetcher):

    def fetch(self):
        if not self.person_trip.purposes:
            logger.error(
                'PersonTrip (trip_id=%d, person_id=%d) does not have purposes',
                self.person_trip.trip_id,
                self.person_trip.person_id,
            )
            return self.get_fallback_value()
        purpose_name = self.person_trip.purposes[0].name
        purpose_name = self.fix_purpose_name(purpose_name)
        return self.find_value_id(purpose_name) or self.get_fallback_value()


class SecondPurposeFetcher(PurposeFetcher):
    fallback_name = 'Нет'

    def fetch(self):
        purposes = self.person_trip.purposes or []
        if len(purposes) > 1:
            purpose_name = purposes[1].name
            purpose_name = self.fix_purpose_name(purpose_name)
            return self.find_value_id(purpose_name) or self.find_value_id(self.fallback_name)
        else:
            return self.find_value_id(self.fallback_name)


class AssignmentFetcher(CustomPropertyFetcher):
    fallback_value = '-999'

    def fetch(self):
        return self.person_trip.assignment or self.get_fallback_value()


class IssueFetcher(CustomPropertyFetcher):
    fallback_value = 'TRAVEL-123456'

    def fetch(self):
        travel_details = self.person_trip.travel_details
        travel_issue = travel_details and travel_details.tracker_issue
        if travel_issue:
            return travel_issue
        logger.error(
            'PersonTrip (trip_id=%d, person_id=%d) does not have issue, fallback to %s',
            self.person_trip.trip_id,
            self.person_trip.person_id,
            self.fallback_value,
        )
        return self.get_fallback_value()


def fetch_custom_properties(person_trip: PersonTrip, aeroclub_data: dict):
    """
    Занимается формированием дополнительных данных для отправки в Аэроклуб.
    Каждое доп. данное имеет id.
    Есть два типа доп. данных: текстовые и списковые.
    Для текстовых нужно передать текст.
    Для списковых нужно передать id значения.

    Все возможные доп. данные мы получаем из Аэроклуба. Он содержатся в aeroclub_data
    id доп. данных и их значений зависят от юр.лица + могут меняться со временем.
    Поэтому нам на основе данных person_trip и Аэроклуба необходимо формировать доп. данные на лету.
    """
    custom_properties = [
        CustomProperty(**item) for item
        in aeroclub_data['data']['business_trip_custom_properties'][0]['custom_properties']
    ]
    result = {
        'id_properties': {},
        'text_properties': {},
    }
    fetcher_to_names = {
        GradeFetcher: ('Level', 'Grade'),
        FirstPurposeFetcher: ('PURPOSE OF TRIP 1',),
        SecondPurposeFetcher: ('PURPOSE OF TRIP 2', 'PURPOSE OF TPIP 2'),
        AssignmentFetcher: ('PURPOSE',),
        IssueFetcher: ('TRAVEL TICKET',),
    }
    name_to_fetcher = {}

    for fetcher_class, names in fetcher_to_names.items():
        for name in names:
            name_to_fetcher[name] = fetcher_class

    for cp in custom_properties:
        name = cp.name.en

        key = 'id_properties' if cp.is_list else 'text_properties'
        if name in name_to_fetcher:
            fetcher = name_to_fetcher[name](person_trip, cp)
            result[key][cp.id] = fetcher.fetch()
        else:
            logger.error(
                'Unknown custom property %s for trip_id=%s, person_id=%s',
                name,
                person_trip.trip_id,
                person_trip.person_id,
            )
            result[key][cp.id] = cp.values[0].id if cp.is_list else '-999'
    return result
