import logging

from intranet.trip.src.config import settings
from intranet.trip.src.lib.aeroclub.api import aeroclub
from intranet.trip.src.lib.aeroclub.enums import AuthorizationStatus
from intranet.trip.src.lib.aeroclub.exceptions import AeroclubWorkflowError
from intranet.trip.src.lib.aeroclub.models import JourneyIn, TravellerIn
from intranet.trip.src.lib.staff.gateway import StaffGateway
from intranet.trip.src.logic.aeroclub.custom_properties import fetch_custom_properties
from intranet.trip.src.models import PersonTrip
from intranet.trip.src.unit_of_work import UnitOfWork

logger = logging.getLogger(__name__)


class AeroclubTripController:
    """
    Класс, который производит различные действия с командировками Аэроклуба. В частности:
      - создают командирвоки
      - авторизовывает командировки
    """
    def __init__(self, uow: UnitOfWork, staff_gateway: StaffGateway = None):
        self.uow = uow
        self.staff_gateway = staff_gateway

    @classmethod
    async def init(cls, uow: UnitOfWork):
        staff_gateway = await StaffGateway.init()
        return cls(uow, staff_gateway)

    @staticmethod
    def _get_journey_in(person_trip: PersonTrip) -> JourneyIn:
        return JourneyIn(
            starts_on=person_trip.gap_date_from,
            ends_on=person_trip.gap_date_to,
            to_city_id=int(person_trip.provider_city_to_id),
            travellers=[
                TravellerIn(
                    company_id=person_trip.person.company.aeroclub_company_id,
                    from_city_id=int(person_trip.provider_city_from_id),
                    payment_type_id=settings.AEROCLUB_PAYMENT_TYPE_ID,
                    profile_id=person_trip.person.provider_profile_id,
                ),
            ]
        )

    async def create_aeroclub_trips(self, trip_id=None, person_ids: list[int] = None):
        if trip_id is None or person_ids is None:
            person_trip_ids = await self.uow.person_trips.get_person_trip_ids_for_aeroclub_create(
                trip_id=trip_id,
                person_ids=person_ids,
            )
        else:
            person_trip_ids = [(trip_id, person_id) for person_id in person_ids]
        logger.info('%d trips to create in Aeroclub', len(person_trip_ids))
        for trip_id, person_id in person_trip_ids:
            try:
                async with self.uow:
                    person_trip = await self.uow.person_trips.get_person_trip_for_aeroclub_create(
                        trip_id,
                        person_id,
                    )
                    if person_trip is not None:
                        await self._create_aeroclub_trip(person_trip)
            except Exception:
                logger.exception(
                    'Failed to create aeroclub trip for trip_id %d, person_id %d',
                    trip_id,
                    person_id,
                )

    async def _create_aeroclub_trip(self, person_trip: PersonTrip):
        trip_id = person_trip.trip_id
        person_id = person_trip.person_id
        logger.info('Try to create Aeroclub trip for trip_id %d, person_id %d', trip_id, person_id)

        if person_trip.person.provider_profile_id is None:
            logger.info(
                'Cannot create Aeroclub trip because of missing profile_id for person: %d',
                person_trip.person_id,
            )
            await self.uow.run_job(
                'create_profiles_to_ihub_and_aeroclub_trips_task',
                trip_id=trip_id,
                person_ids=[person_trip.person_id],
            )
            return

        journey_in = self._get_journey_in(person_trip)
        response = await aeroclub.create_journey(data=journey_in.dict())

        aeroclub_journey_id = response['data']['journey_number']
        aeroclub_trip_id = response['data']['business_trip_numbers'][0]

        await self.uow.person_trips.update(
            trip_id=person_trip.trip_id,
            person_id=person_trip.person_id,
            aeroclub_journey_id=aeroclub_journey_id,
            aeroclub_trip_id=aeroclub_trip_id,
        )
        logger.info(
            'Aeroclub trip created with journey_id %d, trip_id %d',
            aeroclub_journey_id,
            aeroclub_trip_id,
        )

        await self.uow.run_job(
            'authorize_aeroclub_trips_task',
            trip_id=person_trip.trip_id,
            person_id=person_trip.person_id
        )

    async def authorize_aeroclub_trips(self, trip_id: int = None, person_id: int = None):
        person_trips = await self.uow.person_trips.get_not_authorized_person_trips(
            trip_id=trip_id,
            person_id=person_id,
        )
        logger.info('Try to authorize %d trips', len(person_trips))
        for person_trip in person_trips:
            try:
                await self._authorize_aeroclub_trip(person_trip)
            except Exception:
                logger.exception(
                    'Error during authorize aeroclub trip with trip_id %d, person_id %d',
                    person_trip.trip_id,
                    person_trip.person_id,
                )

    async def add_custom_properties_for_person_trip(self, trip_id: int, person_id: int):
        person_trip = await self.uow.person_trips.get_detailed_person_trip(trip_id, person_id)
        if person_trip.aeroclub_trip_id and person_trip.aeroclub_journey_id:
            await self._add_custom_properties(person_trip)
        else:
            logger.info(
                "Unable to add custom properties to aeroclub: "
                f"Missing aeroclub ids (trip_id: {trip_id}, person_id: {person_id})"
            )
        await self.uow.run_job(
            'notify_by_tracker_comment_task',
            issue=person_trip.main_tracker_issue,
            template_name='person_trip_created.jinja2',
            context={
                'person_trip_url': person_trip.url,
            },
        )

    async def _add_custom_properties(self, person_trip: PersonTrip):
        assignment = await self.staff_gateway.get_assignment(person_trip.person.login)
        person_trip.assignment = assignment['value']

        data = await aeroclub.get_trip_custom_properties(
            journey_id=person_trip.aeroclub_journey_id,
            trip_id=person_trip.aeroclub_trip_id,
        )
        await aeroclub.add_trip_custom_properties(
            journey_id=person_trip.aeroclub_journey_id,
            trip_id=person_trip.aeroclub_trip_id,
            **fetch_custom_properties(person_trip, data),
        )

    async def _authorize_aeroclub_trip(self, person_trip: PersonTrip):
        trip_id = person_trip.trip_id
        person_id = person_trip.person_id

        logger.info(
            'Try to add custom properties and authorize Aeroclub trip for trip_id %d, person_id %d',
            trip_id, person_id,
        )

        if not person_trip.aeroclub_trip_id:
            logger.error(
                'PersonTrip with trip_id %s, person_id %d does not have aeroclub_trip_id',
                trip_id, person_id,
            )
            return

        await self._add_custom_properties(person_trip)

        data = await aeroclub.get_trip_authorization_state(
            journey_id=person_trip.aeroclub_journey_id,
            trip_id=person_trip.aeroclub_trip_id,
        )
        authorization_status = data['data']['authorization_status']
        if authorization_status != AuthorizationStatus.authorized:
            logger.error('Wrong authorization status: %s', authorization_status)
            raise AeroclubWorkflowError

        await self.uow.person_trips.update(trip_id, person_id, is_authorized=True)
