from decimal import Decimal
import logging
import uuid

from intranet.trip.src.config import settings
from intranet.trip.src.enums import TransactionStatus
from intranet.trip.src.exceptions import WorkflowError, PermissionDenied
from intranet.trip.src.logic.base import Action
from intranet.trip.src.api.auth import (
    LazyManagersMapByTrip,
    is_yandex_coordinator,
)
from intranet.trip.src.models import User, billing
from intranet.trip.src.unit_of_work import UnitOfWork
from intranet.trip.src.api.schemas import TransactionCreate, TransactionPartialUpdate


logger = logging.getLogger(__name__)


def _split_fee_for_yandex_and_provider(transaction_data: dict):
    """
    BTRIP-3163: Если транзакция – штраф, то сервисный сбор Яндекса за неё – 0,
    иначе всегда settings.YANDEX_FEE
    """
    yandex_fee = (
        Decimal('0')
        if transaction_data['is_penalty']
        else Decimal(settings.YANDEX_FEE)
    )
    provider_fee = transaction_data['fee'] - yandex_fee

    transaction_data['provider_fee'] = provider_fee
    transaction_data['yandex_fee'] = yandex_fee
    del transaction_data['fee']


class TransactionAction(Action):

    def __init__(
        self,
        uow: UnitOfWork,
        user: User,
        transaction: billing.Transaction,
        managers_map: LazyManagersMapByTrip = None,
    ):
        self.uow = uow
        self.user = user
        self.transaction = transaction
        self.managers_map = (
            managers_map
            or LazyManagersMapByTrip(uow, trip_ids=[transaction.trip_id])
        )

    @classmethod
    async def init(
        cls,
        uow: UnitOfWork,
        user: User,
        transaction_id: uuid.UUID,
        managers_map: LazyManagersMapByTrip = None,
    ):
        transaction = await cls.get_transaction(uow, transaction_id)
        return cls(uow, user, transaction, managers_map)

    @classmethod
    async def get_transaction(
        cls,
        uow: UnitOfWork,
        transaction_id: uuid.UUID,
    ) -> billing.Transaction:
        return await uow.billing_transactions.get_transaction(transaction_id)


class CreateTransactionAction(Action):

    action_name = 'create'

    def __init__(
        self,
        uow: UnitOfWork,
        user: User,
        transaction_create: TransactionCreate,
        managers_map: LazyManagersMapByTrip = None,
    ):
        self.uow = uow
        self.user = user
        self.transaction = transaction_create
        self.managers_map = (
            managers_map
            or LazyManagersMapByTrip(uow, trip_ids=[transaction_create.trip_id])
        )

    async def check_permissions(self) -> None:
        if not (
            is_yandex_coordinator(self.user)
            or await self.managers_map.is_manager(self.user.person_id, self.transaction.trip_id)
        ):
            raise PermissionDenied()

        if self.transaction.status in (TransactionStatus.verified, TransactionStatus.completed):
            raise WorkflowError('Too late to update transaction')

    async def execute(self) -> uuid.UUID:
        transaction_data = self.transaction.dict()
        _split_fee_for_yandex_and_provider(transaction_data)

        return await self.uow.billing_transactions.create(
            author_id=self.user.person_id,
            **transaction_data,
        )


class UpdateTransactionAction(TransactionAction):

    action_name = 'update'

    async def check_permissions(self) -> None:
        if not (
            is_yandex_coordinator(self.user)
            or await self.managers_map.is_manager(self.user.person_id, self.transaction.trip_id)
        ):
            raise PermissionDenied()

        if self.transaction.status in (TransactionStatus.verified, TransactionStatus.completed):
            raise WorkflowError('Too late to update transaction')

    async def execute(self, transaction_update: TransactionPartialUpdate) -> uuid.UUID:
        transaction_data = transaction_update.dict(exclude_unset=True)

        if transaction_data.get('is_penalty') is None:
            transaction_data['is_penalty'] = self.transaction.is_penalty

        if transaction_data.get('fee') is None:
            transaction_data['fee'] = self.transaction.fee

        _split_fee_for_yandex_and_provider(transaction_data)

        async with self.uow:
            return await self.uow.billing_transactions.update(
                transaction_id=self.transaction.transaction_id,
                **transaction_data,
            )


class DeleteTransactionAction(TransactionAction):

    action_name = 'delete'

    async def check_permissions(self) -> None:
        if not (
            is_yandex_coordinator(self.user)
            or await self.managers_map.is_manager(self.user.person_id, self.transaction.trip_id)
        ):
            raise PermissionDenied()

        if self.transaction.status in (TransactionStatus.verified, TransactionStatus.completed):
            raise WorkflowError('Too late to delete transaction')

    async def execute(self):
        await self.uow.billing_transactions.delete(self.transaction.transaction_id)


async def get_transaction_actions(
    uow: UnitOfWork,
    user: User,
    transaction: billing.Transaction,
):
    data = {}
    action_classes = [
        UpdateTransactionAction,
        DeleteTransactionAction,
    ]
    for action_class in action_classes:
        action = action_class(
            uow=uow,
            user=user,
            transaction=transaction,
        )
        data[action.action_name] = await action.is_available()
    return billing.TransactionActions(**data)


async def get_transaction_editable_fields(
    transaction: billing.Transaction,
    user: User,
    managers_map: LazyManagersMapByTrip,
) -> set[str]:
    # TODO: BTRIP-3006
    return {'trip_id', 'service_id', 'general_service_id', 'price'}
