from abc import ABC, abstractmethod
from typing import Any, Iterator

from intranet.trip.src.lib.messenger.api import bot_messenger


class BaseMessageProcessor(ABC):
    TASK_NAME = None

    def __init__(self, uow, person, chat_id):
        self.uow = uow
        self.person = person
        self.chat_id = chat_id

    async def get_profile_id(self) -> int:
        return await self.uow.persons.get_provider_profile_id_by_uid(str(self.person.get('uid')))

    @abstractmethod
    async def get_job_kwargs(self, data: Any) -> dict[str, Any]:
        raise NotImplementedError

    async def process(self, data: dict):
        job_kwargs = await self.get_job_kwargs(data)
        async with self.uow:
            profile_id = await self.get_profile_id()
            self.uow.add_job(
                self.TASK_NAME,
                unique=False,
                chat_id=self.chat_id,
                profile_id=profile_id,
                **job_kwargs,
            )


class TextMessageProcessor(BaseMessageProcessor):
    TASK_NAME = 'send_message_to_aeroclub_task'

    async def get_job_kwargs(self, data: str) -> dict[str, Any]:
        return {
            'text': data,
        }


class FileMessageProcessor(BaseMessageProcessor):
    TASK_NAME = 'send_file_to_aeroclub_task'

    async def get_job_kwargs(self, data: dict) -> dict[str, Any]:
        file_data = await bot_messenger.get_file(data['file_id'])
        return {
            'file_name': data['file_name'],
            'file_data': file_data,
        }


class PhotoMessageProcessor(BaseMessageProcessor):
    TASK_NAME = 'send_file_to_aeroclub_task'

    async def get_job_kwargs(self, data: list[dict]) -> dict[str, Any]:
        file_id = data[0]['file_id'].split('?')[0]
        file_data = await bot_messenger.get_file(file_id)
        return {
            'file_name': 'image.png',
            'file_data': file_data,
        }


class MultiplePhotoMessageProcessor(PhotoMessageProcessor):
    async def process(self, data: list[list[dict]]):
        for photo in data:
            await super().process(photo)


def get_message_processors(message: dict) -> Iterator[tuple[type[BaseMessageProcessor], dict]]:
    key_to_processor = {
        'text': TextMessageProcessor,
        'document': FileMessageProcessor,
        'photo': PhotoMessageProcessor,
        'photos': MultiplePhotoMessageProcessor,
    }
    for key, processor in key_to_processor.items():
        if key in message:
            yield processor, message[key]
