import logging
import uuid

from intranet.trip.src.api.schemas import ExtPersonCreate, ExtPersonUpdate
from intranet.trip.src.unit_of_work import UnitOfWork
from intranet.trip.src.enums import ExtPersonStatus
from intranet.trip.src.db.gateways.base import RecordNotFound


logger = logging.getLogger(__name__)


class ExtPersonException(Exception):
    pass


async def create_ext_person(
        uow: UnitOfWork,
        person_id: int,
        ext_person_create: ExtPersonCreate,
) -> dict:
    secret = str(uuid.uuid4())
    external_uid = str(uuid.uuid4())
    async with uow:
        ext_person_id = await uow.ext_persons.create(
            person_id=person_id,
            secret=secret,
            external_uid=external_uid,
            status=ExtPersonStatus.pending,
            **ext_person_create.dict(),
        )
    return {
        'ext_person_id': ext_person_id,
        'secret': secret,
    }


async def regenerate_secret(
        uow: UnitOfWork,
        person_id: int,
        ext_person_id: int,
) -> dict:
    secret = str(uuid.uuid4())
    async with uow:
        ext_person = await uow.ext_persons.get_by_id(ext_person_id)
        if ext_person.person_id != person_id:
            raise ExtPersonException('person_id %d does not correspond ext_person_id')
        ext_person_id = await uow.ext_persons.update(
            ext_person_id=ext_person_id,
            secret=secret,
            status=ExtPersonStatus.pending,
        )
    return {
        'ext_person_id': ext_person_id,
        'secret': secret,
    }


async def update_ext_person(
        uow: UnitOfWork,
        ext_person_id: int,
        ext_person_update: ExtPersonUpdate,
) -> dict:
    try:
        ext_person = await uow.ext_persons.get_by_id(ext_person_id)
    except RecordNotFound:
        raise ExtPersonException('Person is not found')

    if ext_person_update.secret != ext_person.secret:
        raise ExtPersonException('Wrong secret')

    ext_person_data = ext_person_update.dict()
    secret = str(uuid.uuid4())
    ext_person_data['secret'] = secret
    ext_person_data['status'] = ExtPersonStatus.completed

    documents = ext_person_data.pop('documents')
    bonus_cards = ext_person_data.pop('bonus_cards')

    async with uow:
        await uow.ext_persons.update(ext_person_id=ext_person_id, **ext_person_data)
        await uow.ext_persons.delete_documents(ext_person_id=ext_person_id)
        await uow.ext_persons.delete_bonus_cards(ext_person_id=ext_person_id)
        for document in documents:
            await uow.documents.create(ext_person_id=ext_person_id, **document)
        for bonus_card in bonus_cards:
            await uow.bonus_cards.create(ext_person_id=ext_person_id, **bonus_card)

        uow.add_job(
            job_name='sync_profiles_to_ihub_task',
            ext_person_ids=[ext_person_id],
            unique=False,
        )
    return {
        'ext_person_id': ext_person_id,
        'secret': secret,
    }
