import logging

from intranet.trip.src.config import settings

from intranet.trip.src.lib.aeroclub.api import aeroclub
from intranet.trip.src.lib.messenger.api import bot_messenger
from intranet.trip.src.lib.utils import safe_getitem
from intranet.trip.src.models import PersonTrip
from intranet.trip.src.unit_of_work import UnitOfWork


logger = logging.getLogger(__name__)


def _is_aeroclub_message(message: dict):
    return safe_getitem(message, ['posted_by', 'company_info', 'holding_name', 'ru']) == 'Аэроклуб'


async def send_attachments(
    uow: UnitOfWork,
    person_trip: PersonTrip,
    author: str,
    attachments: list[dict],
):
    text = '\n'.join(
        f'[{attachment["name"]}]({attachment["url"]})'
        for attachment in attachments
        if attachment.get('url') and attachment.get('name')
    )
    await bot_messenger.send_message(
        chat_id=person_trip.chat_id,
        text=text,
        author_name=author,
    )
    uow.add_job(
        job_name='send_attachments_to_tracker_task',
        trip_id=person_trip.trip_id,
        person_id=person_trip.person_id,
        attachments=attachments,
    )


async def create_new_messages_from_aeroclub(
        uow: UnitOfWork,
        aeroclub_journey_id: int,
        aeroclub_trip_id: int,
) -> None:
    person_trip = await uow.person_trips.get_person_trip_by_aeroclub_id(
        aeroclub_journey_id=aeroclub_journey_id,
        aeroclub_trip_id=aeroclub_trip_id,
    )
    trip_id = person_trip.trip_id
    person_id = person_trip.person_id
    is_first_aeroclub_message = person_trip.aeroclub_last_message_id is None
    has_aeroclub_message = False
    aeroclub_last_message_id = person_trip.aeroclub_last_message_id or 0

    all_messages = await get_all_message_from_aeroclub(
        aeroclub_journey_id=aeroclub_journey_id,
        aeroclub_trip_id=aeroclub_trip_id,
    )

    if not all_messages or all_messages[-1]['id'] <= aeroclub_last_message_id:
        logger.warning(f'Not found new messages for trip_id={trip_id}, person_id={person_id}')
        return

    for message in all_messages:
        if message['id'] <= aeroclub_last_message_id:
            continue

        if not _is_aeroclub_message(message=message):
            continue

        async with uow:
            await uow.person_trips.update(
                trip_id=trip_id,
                person_id=person_id,
                aeroclub_last_message_id=message['id'],
            )

            has_aeroclub_message = True
            text = message['message']
            author = message['posted_by']['first_name']['ru']
            if text:
                await bot_messenger.send_message(
                    chat_id=person_trip.chat_id,
                    text=text,
                    author_name=author,
                )
            attachments = message.get('attachments')
            if attachments:
                await send_attachments(
                    uow=uow,
                    person_trip=person_trip,
                    author=author,
                    attachments=attachments,
                )

    if is_first_aeroclub_message and has_aeroclub_message:
        async with uow:
            uow.add_job(
                'notify_by_tracker_comment_task',
                issue=person_trip.travel_tracker_issue,
                template_name='notification_messenger.jinja2',
                context={'chat_url': person_trip.chat_url or ''},
                summonees=[person_trip.person.login],
            )


async def get_all_message_from_aeroclub(aeroclub_journey_id: int, aeroclub_trip_id: int):
    messages = await aeroclub.get_messages(
        journey_id=aeroclub_journey_id,
        trip_id=aeroclub_trip_id,
    )
    all_messages = messages
    while len(messages) == settings.AEROCLUB_LIMIT_MESSAGES:
        messages = await aeroclub.get_messages(
            journey_id=aeroclub_journey_id,
            trip_id=aeroclub_trip_id,
            last_message_id=messages[-1]['id'],
        )
        all_messages += messages

    all_messages = all_messages[::-1]
    return all_messages
