from dataclasses import asdict
from intranet.trip.src.api.schemas.provider.suggest import SuggestResponse
from intranet.trip.src.api.schemas import (
    AviaDetailResponse,
    AviaSearchResult,
    AviaSearchFilter,
    FilterItem,
    HotelDetailResponse,
    HotelSearchFilter,
    HotelSearchInfo,
    HotelSearchResult,
    ProviderSearchResultCount,
    SearchAviaRequestIn,
    SearchId,
    AviaSearchInfo,
    SearchHotelRequestIn,
    SearchRailRequestIn,
    RailDetailResponse,
    RailSearchFilter,
    RailSearchResult,
    RailSearchInfo,
)
from intranet.trip.src.enums import ServiceType
from intranet.trip.src.lib.aeroclub.api import aeroclub
from intranet.trip.src.lib.aeroclub.enums import ReferenceType
from intranet.trip.src.logic.providers import ProviderGateway
from intranet.trip.src.logic.providers.aeroclub import converters


class AeroclubGateway(ProviderGateway):

    async def city_suggest(
            self,
            query: str,
            service_type: ServiceType,
    ) -> list[SuggestResponse]:
        result = await aeroclub.get_references(
            query=query,
            type=ReferenceType.city.value,
        )
        converter_out = converters.AeroclubReferencesConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def create_avia_search(
            self,
            search_request_in: SearchAviaRequestIn,
    ) -> SearchId:
        converter = converters.AeroclubAviaSearchRequestConverterIn(lang=self.user.language)
        search_request = converter.convert(search_request_in)
        data = await aeroclub.create_search_request(search_request.dict())
        return SearchId(search_id=data.get('data'))

    async def create_hotel_search(
            self,
            search_request_in: SearchHotelRequestIn,
    ) -> SearchId:
        converter = converters.AeroclubHotelSearchRequestConverterIn(lang=self.user.language)
        search_request = converter.convert(search_request_in)
        data = await aeroclub.create_search_request(search_request.dict())
        return SearchId(search_id=data['data'])

    async def get_hotel_search_results(
            self,
            search_id: str,
            limit: int = 20,
            page: int = 1,
            search_filter: HotelSearchFilter = None,
    ) -> HotelSearchResult:
        params = {}
        if search_filter:
            converter_in = converters.AeroclubHotelSearchFiltersConverterIn(lang=self.user.language)
            params = asdict(converter_in.convert(search_filter))
        params['maxCount'] = limit
        params['pageNumber'] = page - 1
        result = await aeroclub.get_search_results(request_id=search_id, params=params)
        converter_out = converters.AeroclubHotelSearchConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def create_rail_search(
            self,
            search_request_in: SearchRailRequestIn,
    ) -> SearchId:
        converter = converters.AeroclubRailSearchRequestConverterIn(lang=self.user.language)
        search_request = converter.convert(search_request_in)
        data = await aeroclub.create_search_request(search_request.dict())
        return SearchId(search_id=data['data'])

    async def get_rail_search_results_count(
        self,
        search_id: str,
        search_filter: RailSearchFilter = None,
    ) -> ProviderSearchResultCount:
        converter_in = converters.AeroclubRailSearchFiltersConverterIn(lang=self.user.language)
        params = asdict(converter_in.convert(search_filter))
        result = await aeroclub.get_search_results_count(
            request_id=search_id,
            params=params,
        )
        converter_out = converters.AeroclubRailSearchCountConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_rail_search_info(self, search_id: str) -> RailSearchInfo:
        result = await aeroclub.get_search_request(request_id=search_id)
        converter_out = converters.AeroclubRailSearchInfoConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_avia_search_results(
            self,
            search_id: str,
            limit: int = 20,
            page: int = 1,
            search_filter: AviaSearchFilter = None,
    ) -> AviaSearchResult:
        converter_in = converters.AeroclubAviaSearchFiltersConverterIn(lang=self.user.language)
        params = asdict(converter_in.convert(search_filter))
        params['maxCount'] = limit
        params['pageNumber'] = page - 1
        result = await aeroclub.get_search_results(request_id=search_id, params=params)
        converter_out = converters.AeroclubAviaSearchConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_search_results_count(
            self,
            search_id: str,
            search_filter: AviaSearchFilter = None,
    ) -> ProviderSearchResultCount:
        converter_in = converters.AeroclubAviaSearchFiltersConverterIn(lang=self.user.language)
        params = asdict(converter_in.convert(search_filter))
        result = await aeroclub.get_search_results_count(
            request_id=search_id,
            params=params,
        )
        converter_out = converters.AeroclubAviaSearchCountConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_avia_search_info(self, search_id: str) -> AviaSearchInfo:
        result = await aeroclub.get_search_request(request_id=search_id)
        converter_out = converters.AeroclubAviaSearchInfoConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_avia_search_details(self, search_id: str, key: str) -> AviaDetailResponse:
        key_converter_in = converters.AeroclubKeyConverterIn(lang=self.user.language)
        result = await aeroclub.get_search_option_details(
            request_id=search_id,
            **key_converter_in.convert(key),
        )
        converter_out = converters.AeroClubAviaDetailConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_hotel_search_filters(self, search_id: str) -> list[FilterItem]:
        result = await aeroclub.get_search_filters(request_id=search_id)
        converter_out = converters.AeroclubHotelSearchFiltersConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_avia_search_filters(self, search_id: str) -> list[FilterItem]:
        result = await aeroclub.get_search_filters(request_id=search_id)
        converter_out = converters.AeroclubAviaSearchFiltersConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_rail_search_results(
        self,
        search_id: str,
        limit: int = 20,
        page: int = 1,
        search_filter: RailSearchFilter = None,
    ) -> RailSearchResult:
        converter_in = converters.AeroclubRailSearchFiltersConverterIn(lang=self.user.language)
        params = asdict(converter_in.convert(search_filter))
        params['maxCount'] = limit
        params['pageNumber'] = page - 1
        result = await aeroclub.get_search_results(request_id=search_id, params=params)
        converter_out = converters.AeroclubRailSearchConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_hotel_search_details(
            self,
            search_id: str,
            key: str,
    ) -> HotelDetailResponse:
        key_converter_in = converters.AeroclubKeyConverterIn(lang=self.user.language)
        result = await aeroclub.get_search_option_details(
            request_id=search_id,
            **key_converter_in.convert(key),
        )
        converter_out = converters.AeroclubHotelDetailConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_hotel_search_results_count(
        self,
        search_id: str,
        search_filter: HotelSearchFilter = None,
    ) -> ProviderSearchResultCount:
        converter_in = converters.AeroclubHotelSearchFiltersConverterIn(lang=self.user.language)
        params = asdict(converter_in.convert(search_filter))
        result = await aeroclub.get_search_results_count(
            request_id=search_id,
            params=params,
        )
        converter_out = converters.AeroclubHotelSearchCountConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_hotel_search_info(self, search_id: str) -> HotelSearchInfo:
        result = await aeroclub.get_search_request(request_id=search_id)
        converter_out = converters.AeroclubHotelSearchInfoConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_rail_search_filters(self, search_id: str) -> list[FilterItem]:
        result = await aeroclub.get_search_filters(request_id=search_id)
        converter_out = converters.AeroclubRailSearchFiltersConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_rail_search_train_details(
            self,
            search_id: str,
            key: str,
    ) -> RailDetailResponse:
        key_converter_in = converters.AeroclubKeyConverterIn(lang=self.user.language)
        result = await aeroclub.get_search_option_details(
            request_id=search_id,
            **key_converter_in.convert(key),
        )
        converter_out = converters.AeroclubRailDetailConverterOut(lang=self.user.language)
        return converter_out.convert(result)
