from datetime import datetime as dt, timedelta
from decimal import Decimal
from enum import Enum
import re
from typing import Iterable, Union

from intranet.trip.src.api.schemas import (
    AviaSearchFilter,
    HotelSearchFilter,
    RailSearchFilter,
)
from intranet.trip.src.api.schemas.provider.search import (
    AviaDetailResponse,
    AviaLocation,
    AviaSearchInfo,
    AviaSearchResult,
    BaseFlight,
    BaseInfo,
    FilterItem,
    FilterSelectValue,
    Flight,
    FlightBaggage,
    GeoPosition,
    Hotel,
    HotelDetail,
    HotelDetailResponse,
    HotelLocation,
    HotelSearchInfo,
    HotelSearchResult,
    Loc,
    ProviderLeg,
    ProviderSearchResultCount,
    ProviderSegment,
    RailDetailResponse,
    RailLocation,
    RailSearchInfo,
    RailSearchResult,
    Room,
    SearchAviaRequestIn,
    SearchHotelRequestIn,
    SearchRailRequestIn,
    Train,
    TrainCarriage,
    TrainCarriageDetail,
    TrainCarriagePlace,
    TransportBaggage,
)
from intranet.trip.src.config import settings
from intranet.trip.src.enums import (
    FilterValueType,
    ObjectType,
    SearchOrdering,
    SearchStatus,
    ServiceType,
)
from intranet.trip.src.lib import aviacenter
from intranet.trip.src.lib.aviacenter.enums import (
    AviaSortField,
    CabinGender,
    HotelSortField,
    SortDirection,
    TrainCarService,
    TrainCarType,
    TrainCarTypeDisplay,
    TrainCategory,
    TrainPlaceType,
    TrainSortField,
)
from intranet.trip.src.lib.aviacenter.models import (
    AviaSearchRequest,
    AviaSearchSegment,
    AviaSort,
    HotelSearchGuest,
    HotelSearchParams,
    HotelSearchRequest,
    HotelSearchRoom,
    HotelSort,
    TrainSearchRequest,
    TrainSort,
)
from intranet.trip.src.lib.aviacenter.models.search_request import (
    AviaSearchFilterSegment,
    TimeInterval,
)
from intranet.trip.src.lib.utils import b64json_to_dict, dict_to_b64json, safe_getitem

from intranet.trip.src.logic.providers import BaseConverter

PLACE_WITH_GENDER_RE = re.compile(r'^(?P<number>\d+)(?P<gender>\D)?$')


class BaseAviacenterSearchConverterOut(BaseConverter):

    @staticmethod
    def convert_suggest(data: dict, type_: ObjectType) -> BaseInfo:
        return BaseInfo(
            type=type_,
            id=data.get('code') or data.get('iata'),
            name=data.get('title') or data.get('name') or '',
        )

    @staticmethod
    def convert_status(data: dict) -> SearchStatus:
        if data['is_completed']:
            return SearchStatus.completed
        return SearchStatus.in_progress

    @staticmethod
    def convert_date_time(date_time: str) -> dt:
        return dt.strptime(date_time, '%d.%m.%Y %H:%M')

    @staticmethod
    def convert_date_time_with_seconds(date_time: str) -> dt:
        return dt.strptime(date_time, '%d.%m.%Y %H:%M:%S')

    @staticmethod
    def convert_date(date: str) -> dt.date:
        return dt.strptime(date.replace('.', '-'), '%d-%m-%Y').date()

    def convert_datetime_and_utc(self, ts_local: str, ts_moscow: str):
        datetime_local = self.convert_date_time(ts_local)
        datetime_moscow = self.convert_date_time(ts_moscow)
        utc_offset = int((datetime_local - datetime_moscow).total_seconds() / 3600) + 3
        return datetime_local, datetime_local - timedelta(hours=utc_offset)

    @staticmethod
    def convert_page(data: dict) -> int:
        return data['offset'] // data['limit'] + 1

    def convert(self, *args, **kwargs):
        raise NotImplementedError


class BaseAviacenterSearchCountConverterOut(BaseConverter):

    service_type = None

    def convert(self, data: dict) -> ProviderSearchResultCount:
        assert self.service_type is not None
        return ProviderSearchResultCount(
            service_type=self.service_type,
            count=data['total'],
        )


class BaseAviacenterKeyConverterIn(BaseConverter):

    fields: Iterable[str] = None

    def convert(self, key_data: str) -> dict:
        key_decoded = b64json_to_dict(key_data)
        return {field: key_decoded[field] for field in self.fields}


class AviacenterAviaKeyConverterIn(BaseAviacenterKeyConverterIn):

    fields = ('tid',)


class AviacenterAviaSearchFiltersConverterIn(BaseConverter):  # TODO: неподходящее название

    @staticmethod
    def get_segments(uf: AviaSearchFilter) -> list[AviaSearchFilterSegment]:
        departure_time_intervals = None
        if uf.departure_time_from:
            departure_time_intervals = [
                TimeInterval(
                    from_time=uf.departure_time_from,
                    to_time=uf.departure_time_to,
                )
            ]
        arrival_time_intervals = None
        if uf.arrival_time_from:
            arrival_time_intervals = [
                TimeInterval(
                    from_time=uf.arrival_time_from,
                    to_time=uf.arrival_time_to,
                )
            ]
        return [
            AviaSearchFilterSegment(
                # TODO: from_id=,
                # TODO: to_id=,
                departure_time_intervals=departure_time_intervals,
                arrival_time_intervals=arrival_time_intervals,
            ),
        ]

    @staticmethod
    def get_sort_field(order_by: SearchOrdering) -> AviaSortField | None:
        map_order_by = {
            SearchOrdering.price: AviaSortField.price,
            SearchOrdering.duration: AviaSortField.duration,
            SearchOrdering.departure_time: AviaSortField.departure_time,
            SearchOrdering.arrival_time: AviaSortField.arrival_time,
        }
        return map_order_by.get(order_by)

    def convert(self, uf: AviaSearchFilter) -> aviacenter.AviaSearchFilter:
        """
        Convert universal filter object to aviacenter SearchFilter

        :param uf: AviaSearchFilter - universal filter object
        :return: SearchFilter - aviacenter filter object
        """
        sort = None
        if uf.order_by:
            sort = [
                AviaSort(
                    field=self.get_sort_field(uf.order_by),
                    direction=SortDirection.desc if uf.is_descending else SortDirection.asc,
                ),
            ]
        return aviacenter.AviaSearchFilter(
            is_refundable=uf.is_refundable,
            is_exchangeable=uf.is_changeable,
            is_restricted_by_travel_policy=uf.is_restricted_by_travel_policy,
            has_baggage=uf.has_baggage,
            maximum_transfers_count=uf.maximum_transfers_count,
            carriers=uf.air_companies,
            segments=self.get_segments(uf),
            sort=sort,
        )


class BaseAviacenterTransportSearchFiltersConverterOut(BaseConverter):

    def get_order_by_values(self) -> list[FilterSelectValue]:
        return [
            FilterSelectValue(
                target_id=SearchOrdering.price,
                caption={
                    'ru': 'По цене',
                    'en': 'By price',
                }[self.lang],
            ),
            FilterSelectValue(
                target_id=SearchOrdering.duration,
                caption={
                    'ru': 'По длительности',
                    'en': 'By duration',
                }[self.lang],
            ),
            FilterSelectValue(
                target_id=SearchOrdering.departure_time,
                caption={
                    'ru': 'По времени отправления',
                    'en': 'By duration time',
                }[self.lang],
            ),
            FilterSelectValue(
                target_id=SearchOrdering.arrival_time,
                caption={
                    'ru': 'По времени прибывания',
                    'en': 'By arrival time',
                }[self.lang],
            ),
        ]


class AviacenterAviaSearchFiltersConverterOut(BaseAviacenterTransportSearchFiltersConverterOut):

    def convert(self, data: dict = None) -> list[FilterItem]:
        order_by_values = self.get_order_by_values()
        return [
            FilterItem(name='order_by', type=FilterValueType.select, values=order_by_values),
            FilterItem(name='is_descending', type=FilterValueType.boolean),
            FilterItem(name='is_refundable', type=FilterValueType.boolean),
            FilterItem(name='is_changeable', type=FilterValueType.boolean),
            FilterItem(name='is_restricted_by_travel_policy', type=FilterValueType.boolean),
            FilterItem(name='has_baggage', type=FilterValueType.boolean),
            FilterItem(name='maximum_transfers_count', type=FilterValueType.integer),
            FilterItem(name='departure_time_from', type=FilterValueType.time),
            FilterItem(name='departure_time_to', type=FilterValueType.time),
            FilterItem(name='arrival_time_from', type=FilterValueType.time),
            FilterItem(name='arrival_time_to', type=FilterValueType.time),
        ]


class AviacenterAviaSearchRequestConverterIn(BaseConverter):

    def convert(self, search_request_in: SearchAviaRequestIn) -> AviaSearchRequest:
        segments = [
            AviaSearchSegment(
                from_id=search_request_in.from_id,
                to_id=search_request_in.to_id,
                date=search_request_in.departure_on,
            ),
        ]
        if search_request_in.departure_back_on:
            segments.append(
                AviaSearchSegment(
                    from_id=search_request_in.to_id,
                    to_id=search_request_in.from_id,
                    date=search_request_in.departure_back_on,
                ),
            )
        return AviaSearchRequest(
            company_id=settings.AVIACENTER_COMPANY_ID,  # TODO: надо брать из user
            segments=segments,
            is_async=1,  # Необходимо создавать асинхронный поиск что бы получать статус поиска
        )


class AviacenterAviaSearchCountConverterOut(BaseAviacenterSearchCountConverterOut):

    service_type = ServiceType.avia


class AviacenterAviaSearchConverterOut(BaseAviacenterSearchConverterOut):
    """
    aviacenter /avia/search-result response -> trip /search response
    """
    @staticmethod
    def convert_search_location(location_data: dict) -> AviaLocation:
        # У АЦ будем пока искать только по городам
        return AviaLocation(
            country=Loc(type=ObjectType.country, name=location_data['country']['name']),
            city=Loc(type=ObjectType.city, name=location_data['city']['name']),
        )

    @staticmethod
    def convert_trip_location(trip_location_data: dict) -> AviaLocation:
        return AviaLocation(
            terminal=Loc(type=ObjectType.terminal, name=trip_location_data['terminal']),
            airport=Loc(type=ObjectType.airport, name=trip_location_data['airport']['title']),
            city=Loc(type=ObjectType.city, name=trip_location_data['city']['title']),
            country=Loc(type=ObjectType.country, name=trip_location_data['country']['title']),
        )

    @staticmethod
    def convert_class(class_data: dict) -> str:
        return class_data['name']

    @staticmethod
    def convert_baggage(data: dict) -> FlightBaggage:
        baggage = TransportBaggage(
            quantity=data['baggage']['piece'],
            weight=data['baggage']['weight'],
        ) if data['baggage']['piece'] else None

        hand_baggage = TransportBaggage(
            quantity=data['cbaggage']['piece'],
            weight=data['cbaggage']['weight'],
        ) if data['cbaggage']['piece'] else None

        return FlightBaggage(
            baggage=baggage,
            hand_baggage=hand_baggage,
        )

    def convert_segment(self, segment_data: dict) -> ProviderSegment:
        arrival_at = self.convert_date_time_with_seconds(segment_data['arrival']['datetime'])
        arrival_at_timezone_offset = segment_data['arrival']['timezone_offset']
        arrival_at_utc = arrival_at
        if arrival_at_timezone_offset:
            arrival_at_utc -= timedelta(hours=int(arrival_at_timezone_offset))

        departure_at = self.convert_date_time_with_seconds(segment_data['departure']['datetime'])
        departure_at_timezone_offset = segment_data['departure']['timezone_offset']
        departure_at_utc = departure_at
        if departure_at_timezone_offset:
            departure_at_utc -= timedelta(hours=int(departure_at_timezone_offset))

        return ProviderSegment(
            arrival=self.convert_trip_location(segment_data['arrival']),
            arrival_at=arrival_at,
            arrival_at_utc=arrival_at_utc,
            arrival_at_timezone_offset=arrival_at_timezone_offset,
            departure=self.convert_trip_location(segment_data['departure']),
            departure_at=departure_at,
            departure_at_utc=departure_at_utc,
            departure_at_timezone_offset=departure_at_timezone_offset,
            seats=segment_data['seats'],
            flight_number=segment_data['flight_number'],
            flight_duration=safe_getitem(segment_data, ['duration', 'flight', 'common'], 0),
            transfer_duration=safe_getitem(segment_data, ['duration', 'transfer', 'common'], 0),
            route_duration=segment_data['route_duration'],
            comment=segment_data['comment'],
            baggage=self.convert_baggage(segment_data),
            flight_class=self.convert_class(segment_data['class']),
            carrier=self.convert_suggest(segment_data['carrier'], ObjectType.carrier),
            fare_code=segment_data['seats'],
            aircraft=self.convert_suggest(segment_data['aircraft'], ObjectType.aircraft),
        )

    def convert_legs(self, segments_data: list[dict]) -> list[ProviderLeg]:
        """
        пока не вижу способа получить сложный маршрут с пересадками
        """
        directions = [[], []]
        for segment in segments_data:
            # segment['direction'] == 0 если перелет туда, а если 1 то это перелет обратно
            directions[int(segment['direction'])].append(segment)
        if not directions[1]:
            del directions[1]
        legs = []
        for direction in directions:
            legs.append(
                ProviderLeg(
                    segments=[self.convert_segment(segment) for segment in direction],
                    segments_count=len(direction),
                    route_duration=direction[0]['route_duration'],
                )
            )
        return legs

    @staticmethod
    def convert_flight_id(flight_data: dict) -> str:
        return dict_to_b64json({'tid': flight_data['id']})

    def convert_flight(self, flight_data: dict, travel_policy: dict) -> Flight:
        is_travel_policy_compliant = (
            travel_policy is None
            or not travel_policy[flight_data['id']]['is_violated']
        )
        return Flight(
            id=self.convert_flight_id(flight_data),
            price=flight_data['price']['RUB']['amount'],
            is_refundable=flight_data['is_refund'],
            is_changeable=flight_data['segments'][0]['is_change'],
            is_travel_policy_compliant=is_travel_policy_compliant,
            legs=self.convert_legs(flight_data['segments']),
        )

    def convert_flights(self, data: dict) -> list[Flight]:
        return [
            self.convert_flight(flight_data, data['travel_policy'])
            for flight_data in data['flights']
        ]

    def convert(self, data: dict) -> AviaSearchResult:
        limit = data['limit']
        return AviaSearchResult(
            total=data['total'],
            page=self.convert_page(data),
            limit=limit,
            data=self.convert_flights(data),
        )


class AviacenterAviaSearchInfoConverterOut(AviacenterAviaSearchConverterOut):

    def convert_search_info(self, data: dict) -> AviaSearchInfo:
        search_segment = safe_getitem(data, ['search', 'segments', 0], {})
        departure_back_on = safe_getitem(data, ['search', 'segments', 1, 'date'])
        if departure_back_on:
            departure_back_on = self.convert_date(departure_back_on)

        return AviaSearchInfo(
            location_from=self.convert_search_location(search_segment['from']),
            location_to=self.convert_search_location(search_segment['to']),
            departure_on=self.convert_date(search_segment['date']),
            departure_back_on=departure_back_on,
            status=self.convert_status(data),
        )

    def convert(self, data: dict) -> AviaSearchInfo:
        return self.convert_search_info(data)


class AviacenterAviaDetailConverterOut(AviacenterAviaSearchConverterOut):

    def convert_flight(self, flight_data: dict, travel_policy: dict) -> BaseFlight:
        is_travel_policy_compliant = (
            travel_policy is None
            or not travel_policy.get(flight_data['id'], {}).get('is_violated', False)
        )
        return BaseFlight(
            price=flight_data['price']['RUB']['amount'],
            is_refundable=flight_data['is_refund'],
            is_changeable=flight_data['segments'][0]['is_change'],
            is_travel_policy_compliant=is_travel_policy_compliant,
        )

    def convert(self, data: dict) -> AviaDetailResponse:
        return AviaDetailResponse(
            flight=self.convert_flight(data['flight'], data['travel_policy']),
            data=self.convert_legs(data['flight']['segments']),
        )


class AviacenterHotelKeyConverterIn(BaseAviacenterKeyConverterIn):

    fields = ('search_id', )


class AviacenterHotelSearchFiltersConverterIn(BaseConverter):  # TODO: неподходящее название

    @staticmethod
    def get_sort_field(order_by: SearchOrdering) -> HotelSortField | None:
        map_order_by = {
            SearchOrdering.price: HotelSortField.price,
            SearchOrdering.contract: HotelSortField.contract,
            SearchOrdering.favorite: HotelSortField.favorite,
        }
        return map_order_by.get(order_by)

    def convert(self, uf: HotelSearchFilter) -> aviacenter.HotelSearchFilter:
        sort = None
        if uf.order_by:
            sort = [
                HotelSort(
                    field=self.get_sort_field(uf.order_by),
                    direction=SortDirection.desc if uf.is_descending else SortDirection.asc,
                ),
            ]
        return aviacenter.HotelSearchFilter(
            is_restricted_by_travel_policy=uf.is_restricted_by_travel_policy,
            min_price=uf.price_from,
            max_price=uf.price_to,
            stars=uf.stars,
            hotel_categories=uf.hotel_types,
            sort=sort,
        )


class AviacenterHotelSearchFiltersConverterOut(BaseConverter):

    def get_order_by_values(self) -> list[FilterSelectValue]:
        return [
            FilterSelectValue(
                target_id=SearchOrdering.price,
                caption={
                    'ru': 'По цене',
                    'en': 'By price',
                }[self.lang],
            ),
        ]

    def convert(self, data: dict) -> list[FilterItem]:
        star_values = [
            FilterSelectValue(
                target_id="1",
                caption="5",
            ),
            FilterSelectValue(
                target_id="2",
                caption="3",
            ),
        ]
        hotel_values = [
            FilterSelectValue(
                target_id="1",
                caption="Hotel",
            ),
            FilterSelectValue(
                target_id="2",
                caption="CapsuleHotel",
            ),
            FilterSelectValue(
                target_id="3",
                caption="Villa",
            ),
        ]
        return [
            FilterItem(
                name='order_by',
                type=FilterValueType.select,
                values=self.get_order_by_values(),
            ),
            FilterItem(name='is_descending', type=FilterValueType.boolean),
            FilterItem(name='is_restricted_by_travel_policy', type=FilterValueType.boolean),
            FilterItem(name='price_from', type=FilterValueType.integer),
            FilterItem(name='price_to', type=FilterValueType.integer),
            FilterItem(name='stars', type=FilterValueType.multiselect, values=star_values),
            FilterItem(name='hotel_types', type=FilterValueType.multiselect, values=hotel_values),
        ]


class AviacenterHotelSearchRequestConverterIn(BaseConverter):

    def convert(self, search_request_in: SearchHotelRequestIn) -> HotelSearchRequest:
        search_params = HotelSearchParams(
            city=search_request_in.target_id,
            check_in=search_request_in.check_in_on,
            check_out=search_request_in.check_out_on,
            rooms=[  # Пока создаем поиск только на 1 человека
                HotelSearchRoom(
                    guests=[
                        HotelSearchGuest(
                            citizenship='ru',  # TODO: надо брать для конкретного user
                        ),
                    ],
                ),
            ],
        )
        return HotelSearchRequest(
            company_id=settings.AVIACENTER_COMPANY_ID,  # TODO: надо брать из user
            search=search_params,
        )


class AviacenterHotelSearchCountConverterOut(BaseAviacenterSearchCountConverterOut):

    service_type = ServiceType.hotel


class AviacenterHotelSearchConverterOut(BaseAviacenterSearchConverterOut):

    @staticmethod
    def convert_hotel_id(hotel_data: dict) -> str:
        return dict_to_b64json({'search_id': hotel_data['search_id']})

    @staticmethod
    def get_price_info(prices: dict[str, float]) -> tuple[str, Decimal]:
        if not prices:
            return 'RUB', Decimal(0)
        currencies = ['RUB', 'USD', 'EUR']
        for c in currencies:
            if c in prices:
                return c, prices[c]
        return next(iter(prices.items()))

    @staticmethod
    def convert_hotel_location(data: dict) -> HotelLocation:
        return HotelLocation(
            city=Loc(type=ObjectType.city, name=data['city']),
            country=Loc(type=ObjectType.country, name=data['country']),
        )

    def convert_hotel(self, data: dict) -> Hotel:
        geo = data['location']
        currency, price = self.get_price_info(data['min_prices'])
        return Hotel(
            id=self.convert_hotel_id(data),
            hotel_name=data['name'],
            description=data['description'],
            stars=data['stars'],
            image_url=data['photos'][0]['url'],
            currency=currency,
            min_price_per_night=price,
            address=data['address'],
            geo_position=GeoPosition(
                latitude=geo['latitude'],
                longitude=geo['longitude'],
            ),
            location=self.convert_hotel_location(data),
        )

    def convert(self, data: dict) -> HotelSearchResult:
        return HotelSearchResult(
            total=data['total'],
            page=self.convert_page(data),
            limit=data['limit'],
            data=[
                self.convert_hotel(hotel)
                for hotel in data['hotels']
            ],
        )


class AviacenterHotelSearchInfoConverterOut(AviacenterHotelSearchConverterOut):

    def convert(self, data: dict) -> HotelSearchInfo:
        search_data = safe_getitem(data, ['search'], {})
        hotel = safe_getitem(data, ['hotels', 0], {})
        return HotelSearchInfo(
            check_in=self.convert_date(search_data['check_in']),
            check_out=self.convert_date(search_data['check_out']),
            status=SearchStatus.completed if data.get('is_completed') else SearchStatus.in_progress,
            location=self.convert_hotel_location(hotel),
        )


class AviacenterHotelSearchDetailsConverterOut(AviacenterHotelSearchConverterOut):

    @staticmethod
    def get_price_info(prices: dict[str, float]) -> tuple[str, Decimal]:
        if not prices:
            return 'RUB', Decimal(0)
        currencies = ['RUB', 'USD', 'EUR']
        for c in currencies:
            if c in prices:
                return c, prices[c]
        return next(iter(prices.items()))

    def convert_rooms(self, data: dict, num_of_nights: int) -> list[Room]:
        return [
            self.convert_room(room_data, num_of_nights, i)
            for i, room_data in enumerate(data)
        ]

    def convert_room(self, data: dict, num_of_nights: int, index: int) -> Room:
        without_meal_str = 'Без питания'
        meal_name = data['meal_type']
        currency, price = self.get_price_info(data['prices'])
        images = (data['data'] or {}).get('images') or []
        return Room(
            index=index,
            description=data['description'],
            name=data['name'],
            is_meal_included=meal_name != without_meal_str,
            meal_names=[meal_name] if meal_name != without_meal_str else [],
            currency=currency,
            is_travel_policy_compliant=bool(data['travel_policy']),
            images=[it['src'] for it in images],
            price_total=price,
            price_per_night=price / num_of_nights,
        )

    def convert_hotel(self, data: dict) -> HotelDetail:
        search_obj = data['search']
        check_in = self.convert_date(search_obj['check_in'])
        check_out = self.convert_date(search_obj['check_out'])
        hotel_data = data['hotel']
        geo = hotel_data['location']
        return HotelDetail(
            hotel_name=hotel_data['name'],
            description=hotel_data['description'],
            stars=hotel_data['stars'],
            images=[it['url'] for it in hotel_data['photos']],
            address=hotel_data['address'],
            geo_position=GeoPosition(
                latitude=geo['latitude'],
                longitude=geo['longitude'],
            ),
            location=self.convert_hotel_location(hotel_data),
            check_in=check_in,
            check_out=check_out,
            num_of_nights=(check_out - check_in).days,
        )

    def convert(self, data: dict) -> HotelDetailResponse:
        hotel = self.convert_hotel(data)
        rooms = self.convert_rooms(data['hotel']['rooms'], hotel.num_of_nights)
        return HotelDetailResponse(hotel=hotel, data=rooms)


class AviacenterRailKeyConverterIn(BaseAviacenterKeyConverterIn):

    fields = ('train_number', 'departure_time')


class AviacenterRailSearchFiltersConverterIn(BaseConverter):  # TODO: неподходящее название

    @staticmethod
    def convert_carriage_types(uf: RailSearchFilter) -> Union[list[TrainCarType], None]:
        return (
            uf.carriage_types
            and [TrainCarType[carriage_type] for carriage_type in uf.carriage_types]
        )

    @staticmethod
    def convert_train_categories(uf: RailSearchFilter) -> list[TrainCategory]:
        return (
            uf.train_categories
            and [TrainCategory[train_category] for train_category in uf.train_categories]
        )

    @staticmethod
    def get_sort_field(order_by: SearchOrdering) -> TrainSortField | None:
        map_order_by = {
            SearchOrdering.price: TrainSortField.price,
            SearchOrdering.duration: TrainSortField.duration,
            SearchOrdering.departure_time: TrainSortField.departure_time,
            SearchOrdering.arrival_time: TrainSortField.arrival_time,
        }
        return map_order_by.get(order_by)

    def convert(self, uf: RailSearchFilter) -> aviacenter.TrainSearchFilter:
        """
        Convert universal filter object to aviacenter SearchFilter

        :param uf: RailSearchFilter - universal filter object
        :return: TrainSearchFilter - aviacenter filter object
        """
        departure_time_intervals = None
        if uf.departure_time_from:
            departure_time_intervals = [
                TimeInterval(
                    from_time=uf.departure_time_from,
                    to_time=uf.departure_time_to,
                )
            ]
        arrival_time_intervals = None
        if uf.arrival_time_from:
            arrival_time_intervals = [
                TimeInterval(
                    from_time=uf.arrival_time_from,
                    to_time=uf.arrival_time_to,
                )
            ]

        sort = None
        if uf.order_by:
            sort = [
                TrainSort(
                    field=self.get_sort_field(uf.order_by),
                    direction=SortDirection.desc if uf.is_descending else SortDirection.asc,
                ),
            ]

        return aviacenter.TrainSearchFilter(
            departure_time_intervals=departure_time_intervals,
            arrival_time_intervals=arrival_time_intervals,
            train_names=uf.train_names,
            carriers=uf.carriage_owners,
            car_groups=self.convert_carriage_types(uf),
            train_categories=self.convert_train_categories(uf),
            is_restricted_by_travel_policy=uf.is_restricted_by_travel_policy,
            from_stations=uf.from_stations,
            to_stations=uf.to_stations,
            sort=sort,
        )


class AviacenterRailSearchFiltersConverterOut(BaseAviacenterTransportSearchFiltersConverterOut):

    @staticmethod
    def convert_enum(enum: Enum) -> list[FilterSelectValue]:
        return [
            FilterSelectValue(
                target_id=value.name,
                caption=value.value,
            )
            for value in enum
        ]

    def convert(self, data: dict = None) -> list[FilterItem]:
        return [
            FilterItem(
                name='order_by',
                type=FilterValueType.select,
                values=self.get_order_by_values(),
            ),
            FilterItem(name='is_descending', type=FilterValueType.boolean),
            FilterItem(name='is_restricted_by_travel_policy', type=FilterValueType.boolean),
            FilterItem(name='departure_time_from', type=FilterValueType.time),
            FilterItem(name='departure_time_to', type=FilterValueType.time),
            FilterItem(name='arrival_time_from', type=FilterValueType.time),
            FilterItem(name='arrival_time_to', type=FilterValueType.time),
            FilterItem(
                name='train_categories',
                type=FilterValueType.multiselect,
                values=self.convert_enum(TrainCategory),
            ),
            FilterItem(
                name='carriage_types',
                type=FilterValueType.multiselect,
                values=self.convert_enum(TrainCarTypeDisplay),
            ),
            FilterItem(name='train_names', type=FilterValueType.multiselect),
        ]


class AviacenterRailSearchRequestConverterIn(BaseConverter):

    def convert(self, search_request_in: SearchRailRequestIn) -> TrainSearchRequest:
        return TrainSearchRequest(
            company_id=settings.AVIACENTER_COMPANY_ID,  # TODO: надо брать из user
            from_id=search_request_in.from_id,
            to_id=search_request_in.to_id,
            date=search_request_in.departure_on,
        )


class AviacenterRailSearchCountConverterOut(BaseAviacenterSearchCountConverterOut):

    service_type = ServiceType.rail


class AviacenterRailSearchConverterOut(BaseAviacenterSearchConverterOut):
    """
    aviacenter /train/search-result response -> trip /search response
    """
    @staticmethod
    def convert_location_data(search_data: dict) -> dict:
        return {
            'date': search_data['date'],
            'from': {
                'train_station': search_data.get('passenger_departure_station'),
                'country': search_data['from_country'],
                'city': search_data['from'],
            },
            'to': {
                'train_station': search_data.get('passenger_arrival_station'),
                'country': search_data['to_country'],
                'city': search_data['to'],
            },
        }

    @staticmethod
    def convert_search_location(location_data: dict) -> RailLocation:
        return RailLocation(
            city=Loc(type=ObjectType.city, name=location_data.get('city')),
            country=Loc(type=ObjectType.country, name=location_data.get('country')),
        )

    @staticmethod
    def convert_trip_location(location_data: dict) -> RailLocation:
        train_station = Loc(
            type=ObjectType.train_station,
            name=location_data.get('train_station'),
        )
        return RailLocation(
            city=Loc(type=ObjectType.city, name=location_data.get('city')),
            country=Loc(type=ObjectType.country, name=location_data.get('country')),
            train_station=train_station,
        )

    @staticmethod
    def convert_carriage(carriage_data: dict) -> TrainCarriage:
        return TrainCarriage(
            min_price=carriage_data['min_ticket_price'],
            max_price=carriage_data['max_ticket_price'],
            carriage_type=TrainCarType(carriage_data['car_type_name']).name,
            carriage_owner=carriage_data['carriers'][0],
            is_travel_policy_compliant=not carriage_data['travel_policy']['is_violated'],
            travel_policy_violations=carriage_data['travel_policy']['notices'],
            place_count=carriage_data['total_place_quantity'],
        )

    def convert_train_id(self, train_data: dict) -> str:
        departure_date_time = self.convert_date_time(train_data['departure_date_time'])
        train_id_data = {
            'train_number': train_data['train_number'],
            'departure_time': departure_date_time.strftime('%Y-%m-%d %H:%M'),
        }
        return dict_to_b64json(train_id_data)

    def convert_train(self, train_data: dict, search_data: dict) -> Train:
        trip_location_data = self.convert_location_data(train_data | search_data)
        departure_local, departure_utc = self.convert_datetime_and_utc(
            ts_local=train_data['local_departure_date_time'],
            ts_moscow=train_data['departure_date_time'],
        )
        # NOTE: тут возвращается список из-за возможной "перецепки вагонов"
        arrival_local, arrival_utc = self.convert_datetime_and_utc(
            ts_local=train_data['local_arrival_date_times'][0],
            ts_moscow=train_data['arrival_date_times'][0],
        )
        return Train(
            id=self.convert_train_id(train_data),
            train_name=train_data['train_name'],
            train_number=train_data['train_number'],
            train_category=TrainCategory(train_data['train_category']).name,
            ride_duration=int(train_data['trip_duration']),
            departure=self.convert_trip_location(trip_location_data['from']),
            departure_at=departure_local,
            departure_at_utc=departure_utc,
            arrival=self.convert_trip_location(trip_location_data['to']),
            arrival_at=arrival_local,
            arrival_at_utc=arrival_utc,
            carriages=[
                self.convert_carriage(carriage_data)
                for carriage_data in train_data['car_groups']
            ],
        )

    def convert(self, data: dict) -> RailSearchResult:
        return RailSearchResult(
            total=data['total'],
            page=self.convert_page(data),
            limit=data['limit'],
            data=[self.convert_train(train_data, data['search']) for train_data in data['items']],
        )


class AviacenterRailSearchInfoConverterOut(AviacenterRailSearchConverterOut):

    def convert_search_info(self, data: dict) -> RailSearchInfo:
        search_data = safe_getitem(data, ['search'], {})
        search_location_data = self.convert_location_data(search_data)
        return RailSearchInfo(
            location_from=self.convert_search_location(search_location_data['from']),
            location_to=self.convert_search_location(search_location_data['to']),
            departure_on=self.convert_date(search_location_data['date']),
            status=SearchStatus.completed,
        )

    def convert(self, data: dict) -> RailSearchInfo:
        return self.convert_search_info(data)


class AviacenterRailDetailConverterOut(AviacenterRailSearchConverterOut):
    # TODO: неподходящее название класса

    @staticmethod
    def convert_travel_policy(data: dict) -> dict:
        return {
            'is_violated': data['travel_policy'].get('is_violated', False),
            'violations': data['travel_policy'].get('notices', []),
        }

    @staticmethod
    def convert_place_type(place_data: dict) -> str:
        try:
            return TrainPlaceType(place_data['type']).name
        except ValueError:
            return TrainPlaceType.undefined.name

    @staticmethod
    def convert_place_number(place_data: dict):
        # Половой признак места возвращается внутри номера: 18Ж -- место 18, женское
        matches = PLACE_WITH_GENDER_RE.match(place_data['place'])
        gender = matches.group('gender') and CabinGender(matches.group('gender')).name
        return int(matches.group('number')), gender

    def convert_place(self, place_data: dict) -> TrainCarriagePlace:
        place_number, place_gender = self.convert_place_number(place_data)
        return TrainCarriagePlace(
            place_number=place_number,
            min_price=place_data['min_price'],
            max_price=place_data['max_price'],
            compartment_number=int(place_data['compartment_number']),
            compartment_gender=place_gender,
            place_type=self.convert_place_type(place_data),
            place_type_description=place_data['type_description'],
        )

    @staticmethod
    def convert_services(services_data: dict) -> list[str]:
        services = []
        # Здесь АЦ возвращает либо словарь услуг, либо пустой список (:
        for service in (services_data or {}).keys():
            try:
                services.append(TrainCarService(service).name)
            except ValueError:
                pass
        return services

    def convert_carriage(self, carriage_data: dict) -> TrainCarriageDetail:
        travel_policy_data = self.convert_travel_policy(carriage_data)
        return TrainCarriageDetail(
            has_electronic_registration=carriage_data['has_electronic_registration'],
            min_price=carriage_data['min_ticket_price'],
            max_price=carriage_data['max_ticket_price'],
            carriage_type=TrainCarType(carriage_data['car_type_name']).name,
            carriage_owner=carriage_data['carrier'],
            place_count=len(carriage_data['places']),
            is_travel_policy_compliant=not travel_policy_data['is_violated'],
            travel_policy_violations=travel_policy_data['violations'],
            carriage_number=int(carriage_data['car_number']),
            service_class_code=carriage_data['service_class'],
            service_class_description=carriage_data['service_class_description'],
            services=self.convert_services(carriage_data['services']),
            places=[self.convert_place(place) for place in carriage_data['places']],
        )

    def convert(self, data: dict) -> RailDetailResponse:
        return RailDetailResponse(
            data=[self.convert_carriage(carriage_data) for carriage_data in data['items']],
        )
