from intranet.trip.src.config import settings
from intranet.trip.src.enums import ServiceType
from intranet.trip.src.lib.aviacenter.api import aviacenter
from intranet.trip.src.api.schemas import SuggestResponse
from intranet.trip.src.logic.providers import ProviderGateway
from intranet.trip.src.logic.providers.aviacenter import converters
from intranet.trip.src.api.schemas import (
    AviaDetailResponse,
    AviaSearchResult,
    AviaSearchFilter,
    FilterItem,
    HotelDetailResponse,
    HotelSearchFilter,
    HotelSearchResult,
    HotelSearchInfo,
    ProviderSearchResultCount,
    SearchAviaRequestIn,
    SearchId,
    AviaSearchInfo,
    SearchHotelRequestIn,
    SearchRailRequestIn,
    RailDetailResponse,
    RailSearchResult,
    RailSearchFilter,
    RailSearchInfo,
)


class AviacenterGateway(ProviderGateway):

    async def city_suggest(
            self,
            query: str,
            service_type: ServiceType,
    ) -> list[SuggestResponse]:
        lang = self.user.language
        if service_type == ServiceType.avia:
            result = await aviacenter.get_airports(query=query, lang=lang)
            converter_out_class = converters.AviacenterAirportsConverterOut
        elif service_type == ServiceType.rail:
            result = await aviacenter.get_stations(query=query, lang=lang)
            converter_out_class = converters.AviacenterStationsConverterOut
        elif service_type == ServiceType.hotel:
            result = await aviacenter.get_cities(query=query)
            converter_out_class = converters.AviacenterCitiesConverterOut
        else:
            raise TypeError(f'Unknown service_type: {service_type}')
        converter_out = converter_out_class(lang=lang)
        return converter_out.convert(result)

    async def create_avia_search(
            self,
            search_request_in: SearchAviaRequestIn,
    ) -> SearchId:
        converter = converters.AviacenterAviaSearchRequestConverterIn(lang=self.user.language)
        search_request = converter.convert(search_request_in)
        result = await aviacenter.create_avia_search(
            search_request=search_request,
            lang=self.user.language,
        )
        return SearchId(search_id=result['sro'])

    async def get_avia_search_results(
            self,
            search_id: str,
            limit: int = 20,
            page: int = 1,
            search_filter: AviaSearchFilter = None,
    ) -> AviaSearchResult:
        converter_in = converters.AviacenterAviaSearchFiltersConverterIn(lang=self.user.language)
        result = await aviacenter.get_avia_search_results(
            company_id=settings.AVIACENTER_COMPANY_ID,  # TODO: надо брать из user
            sro=search_id,
            language=self.user.language,
            travel_policy_id=settings.AVIACENTER_TRAVEL_POLICY_ID,
            page=page,
            limit=limit,
            search_filter=converter_in.convert(search_filter),
        )
        converter_out = converters.AviacenterAviaSearchConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_search_results_count(
            self,
            search_id: str,
            search_filter: AviaSearchFilter = None,
    ) -> ProviderSearchResultCount:
        converter_in = converters.AviacenterAviaSearchFiltersConverterIn(lang=self.user.language)
        result = await aviacenter.get_avia_search_results(
            company_id=settings.AVIACENTER_COMPANY_ID,  # TODO: надо брать из user
            sro=search_id,
            language=self.user.language,
            limit=1,  # для информации достаточно одного результата
            page=1,
            is_async=1,
            search_filter=converter_in.convert(search_filter),
        )
        converter_out = converters.AviacenterAviaSearchCountConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_avia_search_info(self, search_id: str) -> AviaSearchInfo:
        result = await aviacenter.get_avia_search_results(
            company_id=settings.AVIACENTER_COMPANY_ID,  # TODO: надо брать из user
            sro=search_id,
            language=self.user.language,
            limit=1,  # для информации достаточно одного результата
            page=1,
            is_async=1,
        )
        converter_out = converters.AviacenterAviaSearchInfoConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_avia_search_details(self, search_id: str, key: str) -> AviaDetailResponse:
        key_converter_in = converters.AviacenterAviaKeyConverterIn(lang=self.user.language)
        result = await aviacenter.get_flight_info(
            company_id=settings.AVIACENTER_COMPANY_ID,  # TODO: надо брать из user
            travel_policy_id=settings.AVIACENTER_TRAVEL_POLICY_ID,
            **key_converter_in.convert(key),
        )
        converter_out = converters.AviacenterAviaDetailConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_hotel_search_filters(self, search_id: str) -> list[FilterItem]:
        # TODO: дергать ручку АЦ, в которой они будут возвращать допустимые значения мультиселектов
        converter_out = converters.AviacenterHotelSearchFiltersConverterOut(lang=self.user.language)
        return converter_out.convert(None)

    async def get_avia_search_filters(self, search_id: str) -> list[FilterItem]:
        # TODO: дергать ручку АЦ, в которой они будут возвращать допустимые значения мультиселектов
        converter_out = converters.AviacenterAviaSearchFiltersConverterOut(lang=self.user.language)
        return converter_out.convert()

    async def create_hotel_search(
            self,
            search_request_in: SearchHotelRequestIn,
    ) -> SearchId:
        converter = converters.AviacenterHotelSearchRequestConverterIn(lang=self.user.language)
        search_request = converter.convert(search_request_in)
        result = await aviacenter.create_hotel_search(
            search_request=search_request,
            lang=self.user.language,
        )
        return SearchId(search_id=result['search_id'])

    async def get_hotel_search_results(
            self,
            search_id: str,
            limit: int = 20,
            page: int = 1,
            search_filter: HotelSearchFilter = None,
    ) -> HotelSearchResult:
        converter_in = converters.AviacenterHotelSearchFiltersConverterIn(lang=self.user.language)
        result = await aviacenter.get_hotel_search_results(
            company_id=settings.AVIACENTER_COMPANY_ID,  # TODO: надо брать из user
            search_id=search_id,
            language=self.user.language,
            travel_policy_id=settings.AVIACENTER_TRAVEL_POLICY_ID,
            page=page,
            limit=limit,
            search_filter=converter_in.convert(search_filter),
        )
        converter_out = converters.AviacenterHotelSearchConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_hotel_search_details(
            self,
            search_id: str,
            key: str,
    ) -> HotelDetailResponse:
        key_converter_in = converters.AviacenterHotelKeyConverterIn(lang=self.user.language)
        result = await aviacenter.get_hotel_search_view(
            company_id=settings.AVIACENTER_COMPANY_ID,  # TODO: надо брать из user
            search_id=key_converter_in.convert(key)['search_id'],
        )
        converter_out = converters.AviacenterHotelSearchDetailsConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_hotel_search_results_count(
            self,
            search_id: str,
            search_filter: HotelSearchFilter = None,
    ) -> ProviderSearchResultCount:
        converter_in = converters.AviacenterHotelSearchFiltersConverterIn(lang=self.user.language)
        result = await aviacenter.get_hotel_search_results(
            company_id=settings.AVIACENTER_COMPANY_ID,  # TODO: надо брать из user
            search_id=search_id,
            language=self.user.language,
            limit=1,
            page=1,
            search_filter=converter_in.convert(search_filter),
        )
        converter_out = converters.AviacenterHotelSearchCountConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_hotel_search_info(self, search_id: str) -> HotelSearchInfo:
        result = await aviacenter.get_hotel_search_results(
            company_id=settings.AVIACENTER_COMPANY_ID,  # TODO: надо брать из user
            search_id=search_id,
            language=self.user.language,
            limit=1,
            page=1,
        )
        converter_out = converters.AviacenterHotelSearchInfoConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def create_rail_search(
            self,
            search_request_in: SearchRailRequestIn,
    ) -> SearchId:
        converter = converters.AviacenterRailSearchRequestConverterIn(lang=self.user.language)
        search_request = converter.convert(search_request_in)
        result = await aviacenter.create_train_search(
            search_request=search_request,
            lang=self.user.language,
        )
        return SearchId(search_id=result['search_id'])

    async def get_rail_search_results(
            self,
            search_id: str,
            limit: int = 20,
            page: int = 1,
            search_filter: RailSearchFilter = None,
    ) -> RailSearchResult:
        converter_in = converters.AviacenterRailSearchFiltersConverterIn(lang=self.user.language)
        result = await aviacenter.get_train_search(
            company_id=settings.AVIACENTER_COMPANY_ID,  # TODO: надо брать из user
            search_id=search_id,
            lang=self.user.language,
            travel_policy_id=settings.AVIACENTER_TRAVEL_POLICY_ID,
            page=page,
            limit=limit,
            search_filter=converter_in.convert(search_filter),
        )
        converter_out = converters.AviacenterRailSearchConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_rail_search_results_count(
            self,
            search_id: str,
            search_filter: RailSearchFilter = None,
    ) -> ProviderSearchResultCount:
        converter_in = converters.AviacenterRailSearchFiltersConverterIn(lang=self.user.language)
        result = await aviacenter.get_train_search(
            company_id=settings.AVIACENTER_COMPANY_ID,  # TODO: надо брать из user
            search_id=search_id,
            lang=self.user.language,
            travel_policy_id=settings.AVIACENTER_TRAVEL_POLICY_ID,
            limit=1,  # для информации достаточно одного результата
            page=1,
            search_filter=converter_in.convert(search_filter),
        )
        converter_out = converters.AviacenterRailSearchCountConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_rail_search_info(self, search_id: str) -> RailSearchInfo:
        result = await aviacenter.get_train_search(
            company_id=settings.AVIACENTER_COMPANY_ID,  # TODO: надо брать из user
            search_id=search_id,
            lang=self.user.language,
            travel_policy_id=settings.AVIACENTER_TRAVEL_POLICY_ID,
            limit=1,  # для информации достаточно одного результата
            page=1,
        )
        converter_out = converters.AviacenterRailSearchInfoConverterOut(lang=self.user.language)
        return converter_out.convert(result)

    async def get_rail_search_filters(self, search_id: str) -> list[FilterItem]:
        # TODO: дергать ручку АЦ, в которой они будут возвращать допустимые значения мультиселектов
        converter_out = converters.AviacenterRailSearchFiltersConverterOut(lang=self.user.language)
        return converter_out.convert()

    async def get_rail_search_train_details(
            self,
            search_id: str,
            key: str,
    ) -> RailDetailResponse:
        key_converter_in = converters.AviacenterRailKeyConverterIn(lang=self.user.language)
        result = await aviacenter.get_train_info(
            company_id=settings.AVIACENTER_COMPANY_ID,  # TODO: надо брать из user
            search_id=search_id,
            travel_policy_id=settings.AVIACENTER_TRAVEL_POLICY_ID,
            **key_converter_in.convert(key),
        )
        converter_out = converters.AviacenterRailDetailConverterOut(lang=self.user.language)
        return converter_out.convert(result)
