from abc import ABC, abstractmethod

from intranet.trip.src.api.schemas import SuggestResponse
from intranet.trip.src.api.schemas import (
    AviaSearchResult,
    AviaDetailResponse,
    AviaSearchFilter,
    FilterItem,
    HotelSearchFilter,
    HotelSearchInfo,
    HotelSearchResult,
    ProviderSearchResultCount,
    SearchAviaRequestIn,
    SearchId,
    AviaSearchInfo,
    SearchHotelRequestIn,
    SearchRailRequestIn,
    RailSearchInfo,
)
from intranet.trip.src.enums import ServiceType
from intranet.trip.src.models.person import User
from intranet.trip.src.api.schemas.provider.search import (
    HotelDetailResponse,
    RailDetailResponse,
    RailSearchResult,
)
from intranet.trip.src.api.schemas.provider.search_filter import RailSearchFilter


class ProviderGateway(ABC):
    """
    Интерфейс для классов, которые реализуют походы в разных поставщиков (Аэроклуб, Авицентр)
    """
    def __init__(self, user: User):
        self.user = user

    @abstractmethod
    async def city_suggest(
            self,
            query: str,
            service_type: ServiceType,
    ) -> list[SuggestResponse]:
        raise NotImplementedError

    @abstractmethod
    async def create_avia_search(
            self,
            search_request_in: SearchAviaRequestIn,
    ) -> SearchId:
        raise NotImplementedError

    @abstractmethod
    async def create_hotel_search(
            self,
            search_request_in: SearchHotelRequestIn,
    ) -> SearchId:
        raise NotImplementedError

    @abstractmethod
    async def get_hotel_search_results(
            self,
            search_id: str,
            limit: int = 20,
            page: int = 1,
            search_filter: HotelSearchFilter = None,
    ) -> HotelSearchResult:
        raise NotImplementedError

    @abstractmethod
    async def get_hotel_search_details(self, search_id: str, key: str) -> HotelDetailResponse:
        raise NotImplementedError

    @abstractmethod
    async def get_hotel_search_results_count(
        self,
        search_id: str,
        search_filter: HotelSearchFilter = None,
    ) -> ProviderSearchResultCount:
        raise NotImplementedError

    @abstractmethod
    async def get_hotel_search_info(self, search_id: str) -> HotelSearchInfo:
        raise NotImplementedError

    @abstractmethod
    async def create_rail_search(
            self,
            search_request_in: SearchRailRequestIn,
    ) -> SearchId:
        raise NotImplementedError

    @abstractmethod
    async def get_avia_search_results(
            self,
            search_id: str,
            limit: int = 20,
            page: int = 1,
            search_filter: AviaSearchFilter = None,
    ) -> AviaSearchResult:
        raise NotImplementedError

    @abstractmethod
    async def get_search_results_count(
            self,
            search_id: str,
            search_filter: AviaSearchFilter = None,
    ) -> ProviderSearchResultCount:
        raise NotImplementedError

    @abstractmethod
    async def get_avia_search_info(self, search_id: str) -> AviaSearchInfo:
        raise NotImplementedError

    async def get_hotel_search_filters(self, search_id: str) -> list[FilterItem]:
        raise NotImplementedError

    @abstractmethod
    async def get_avia_search_filters(self, search_id: str) -> list[FilterItem]:
        raise NotImplementedError

    @abstractmethod
    async def get_rail_search_results(
            self,
            search_id: str,
            limit: int = 20,
            page: int = 1,
            search_filter: RailSearchFilter = None,
    ) -> RailSearchResult:
        raise NotImplementedError

    @abstractmethod
    async def get_rail_search_results_count(
        self,
        search_id: str,
        search_filter: RailSearchFilter = None,
    ) -> ProviderSearchResultCount:
        raise NotImplementedError

    @abstractmethod
    async def get_rail_search_info(self, search_id: str) -> RailSearchInfo:
        raise NotImplementedError

    @abstractmethod
    async def get_rail_search_filters(self, search_id: str) -> list[FilterItem]:
        raise NotImplementedError

    @abstractmethod
    async def get_rail_search_train_details(self, search_id: str, key: str) -> RailDetailResponse:
        raise NotImplementedError

    @abstractmethod
    async def get_avia_search_details(
        self,
        search_id: str,
        key: str,
    ) -> AviaDetailResponse:
        raise NotImplementedError
