import logging

from abc import ABC, abstractmethod

from ylog.context import log_context


logger = logging.getLogger(__name__)


class BaseTrigger(ABC):
    """
    Базовый класс подключения/отключения определенной фичи для командировки
    Общий флоу:
        - проверяем текущее состояние (is_active)
        - проверяем ожидаемое состояние (should_be_active)
        - если они не совпадают - выполняем подключение (activate) либо отключение (deactivate)
    """

    def __init__(self, uow, person_trip):
        self.uow = uow
        self.person_trip = person_trip

    @abstractmethod
    async def activate(self):
        """
        Метод, выполняющий логику подключения фичи
        """
        raise NotImplementedError

    @abstractmethod
    async def deactivate(self):
        """
        Метод, выполняющий логику отключения фичи
        """
        raise NotImplementedError

    @abstractmethod
    def is_active(self) -> bool:
        """
        Проверка текущего состояния фичи
        """
        raise NotImplementedError

    @abstractmethod
    def should_be_active(self) -> bool:
        """
        Проверка, должна ли фича быть активна
        """
        raise NotImplementedError

    def get_context(self):
        return {
            'name': self.__class__.__name__,
            'trip_id': self.person_trip.trip_id,
            'person_id': self.person_trip.person_id,
        }

    async def execute(self):
        with log_context(trigger=self.get_context()):
            is_active = self.is_active()
            should_be_active = self.should_be_active()
            if is_active and not should_be_active:
                logger.info('trigger %s - deactivate', self.__class__.__name__)
                await self.deactivate()
            elif not is_active and should_be_active:
                logger.info('trigger %s - activate', self.__class__.__name__)
                await self.activate()
            else:
                state = 'active' if is_active else 'inactive'
                logger.info(
                    'trigger %s should not change state, current: %s',
                    self.__class__.__name__,
                    state,
                )
