import logging
from itertools import product
from typing import Union, Optional
from pydantic import ValidationError

from intranet.trip.src.api.schemas import TripCreate, TripUpdate
from intranet.trip.src.api.auth import is_holding_coordinator, has_person_perm
from intranet.trip.src.api.schemas.validators import db_validate_by_ids
from intranet.trip.src.config import settings
from intranet.trip.src.enums import (
    TripStatus,
    Provider,
    PTStatus,
    ConferenceParticiationType,
)
from intranet.trip.src.exceptions import PermissionDenied, WorkflowError
from intranet.trip.src.logic.base import Action
from intranet.trip.src.models import User, Trip
from intranet.trip.src.unit_of_work import UnitOfWork


logger = logging.getLogger(__name__)


async def get_holding_id(holding_ids: list[int]) -> Optional[int]:
    trip_holding_id = None
    for holding_id in holding_ids:
        if trip_holding_id and trip_holding_id != holding_id:
            raise WorkflowError('Different holdings in one trip')
        trip_holding_id = holding_id
    return trip_holding_id


class TripAction(Action):

    def __init__(self, uow: UnitOfWork, user: User, trip: Trip = None):
        self.uow = uow
        self.user = user
        self.trip = trip

    @classmethod
    async def init(cls, uow: UnitOfWork, user: User, trip_id: int = None):
        trip = await cls.get_trip(uow, trip_id)
        self = cls(uow, user, trip)
        if not trip:
            self.trip_id = trip_id
        return self

    @classmethod
    async def get_trip(cls, uow: UnitOfWork, trip_id: int) -> Trip:
        if trip_id:
            return await uow.trips.get_detailed_trip(trip_id)


class TripCreateUpdateAction(TripAction):

    def __init__(self, uow: UnitOfWork, user: User, trip: Trip = None):
        super().__init__(uow, user, trip)
        if trip:
            self.trip_id = trip.trip_id

    async def check_permissions(self) -> None:
        if self.trip:
            if not has_person_perm(self.user, self.trip.author):
                raise PermissionDenied(
                    log_message=(
                        f'{self.__class__.__name__}: User has not read '
                        f'permission for person {self.trip.author}'
                    ),
                )

    async def validate(self, trip: TripCreate) -> None:
        """
        pydantic синхронный, он не рассчитан на то, чтобы делать какое-то IO в валидаторах
        поэтому валидацию через базу данных нужно делать отдельно
        В качестве улучшения можно на уровне модели добавить хотя бы какую-то мету,
        чтобы сильнее связать между собой саму модель и post-валидацию
        """
        errors = []
        errors.append(await db_validate_by_ids(self.uow.purposes, trip, 'purposes'))
        errors = [e for e in errors if e]
        if errors:
            raise ValidationError(errors, model=TripCreate)

    async def _update(self, trip: TripUpdate):
        await self.uow.trips.clean_purposes(self.trip_id)
        if trip.purposes:
            await self.uow.trips.add_purposes(trip_id=self.trip_id, purpose_ids=trip.purposes)

        trip_id = await self.uow.trips.update(self.trip_id, **self._get_trip_fields(trip))
        if trip.conf_details:
            await self.uow.trips.update_conf_details(
                trip_id=trip_id,
                **trip.conf_details.dict(),
            )
        await self.uow.person_trips.bulk_update(
            trip_id=trip_id,
            **self._get_common_person_trip_fields(trip),
        )

    async def _create(self, trip: TripCreate) -> int:
        # BTRIP-2601 На самом деле тут person_id, а не person_uid
        persons = await self.uow.persons.get_persons(
            person_ids=[int(pt.person_uid) for pt in trip.person_trips]
        )
        person_by_person_id = {person.person_id: person for person in persons}
        person_ids = list(person_by_person_id.keys())

        holding_ids = [person.company.holding_id for person in persons]
        holding_id = await get_holding_id(holding_ids)
        if (
            self.user.company.holding_id != holding_id
            and not is_holding_coordinator(self.user, holding_id)
        ):
            raise WorkflowError('Forbidden')

        trip_id = await self.uow.trips.create(status=TripStatus.new, **self._get_trip_fields(trip))

        person_trips_list = []
        person_conf_details_list = []
        for item in trip.person_trips:
            # Данные персональных командировок формируются из данных групповой командировки
            # А поле person_trips должно работать по принципу
            # "что передали - перезатерли, что не передали - не трогаем"
            # Поэтому используем exclude_unset
            person_trip_data = item.dict(exclude_unset=True)
            person_id = int(person_trip_data.pop('person_uid'))
            person = person_by_person_id.get(person_id)
            if not person:
                continue
            person_trip_data['person_id'] = person_id
            person_trip_data['is_offline'] = person.is_offline_trip
            person_trip_data['provider'] = person.company.provider
            person_trip_data['status'] = PTStatus.new
            person_conf_details = person_trip_data.pop('conf_details', {})
            person_trips_list.append(person_trip_data)
            person_conf_details_list.append({
                'trip_id': trip_id,
                'person_id': person_id,
                'role': ConferenceParticiationType.listener.value,
                'comment': '',
                'is_another_city': (
                    trip.conf_details.is_another_city if trip.conf_details else False
                ),
                **person_conf_details,
            })

        await self.uow.person_trips.bulk_create(
            trip_id=trip_id,
            values=[
                self._get_common_person_trip_fields(trip) | person_trip
                for person_trip in person_trips_list
            ],
        )

        if trip.conf_details:
            await self.uow.trips.create_conf_details(trip_id=trip_id, **trip.conf_details.dict())
            for person_conf_details in person_conf_details_list:
                await self.uow.person_trips.update_conf_details(**person_conf_details)

        if trip.route:
            await self._create_route_info(trip_id, person_ids, trip)

        if trip.purposes:
            await self._add_purposes(trip_id, person_ids, purpose_ids=trip.purposes)

        self.uow.add_job(
            'create_aeroclub_trips_task',
            trip_id=trip_id,
            person_ids=[
                person.person_id for person in persons
                if person.company.provider == Provider.aeroclub
            ],
            unique=False,
        )
        self.uow.add_job('create_trip_chats_task', trip_id=trip_id, person_ids=person_ids)
        return trip_id

    async def _create_route_info(self, trip_id: int, person_ids: list[int], trip: TripCreate):
        route_info = [point.dict(exclude={'aeroclub_city_id'}) for point in trip.route]
        for index, point in enumerate(route_info):
            point['trip_id'] = trip_id
            point['is_start_city'] = index == 0
        await self.uow.trips.create_route_points(route_info)
        await self.uow.person_trips.create_route_points(
            route_points=[
                {'person_id': person_id, **route_point}
                for person_id, route_point in product(person_ids, route_info)
            ]
        )

    async def _add_purposes(self, trip_id: int, person_ids: list[int], purpose_ids: list[int]):
        await self.uow.trips.add_purposes(trip_id=trip_id, purpose_ids=purpose_ids)
        person_ids = person_ids or []
        for person_id in person_ids:
            await self.uow.person_trips.add_purposes(
                trip_id=trip_id,
                person_id=person_id,
                purpose_ids=purpose_ids,
            )

    def _get_trip_fields(self, trip: Union[TripCreate, TripUpdate]):
        return {
            'city_from': trip.city_from,
            'city_to': trip.city_to,
            'country_from': trip.country_from,
            'country_to': trip.country_to,
            'date_from': trip.date_from,
            'date_to': trip.date_to,
            'author_id': self.user.person_id,
            'comment': trip.comment,
            'provider_city_from_id': trip.provider_city_from_id,
            'provider_city_to_id': trip.provider_city_to_id,
        }

    def _get_common_person_trip_fields(self, trip: Union[TripCreate, TripUpdate]):
        return {
            'description': trip.description,
            'with_days_off': trip.with_days_off,
            'city_from': trip.city_from,
            'country_from': trip.country_from,
            'provider_city_from_id': trip.provider_city_from_id,
            'gap_date_from': trip.date_from,
            'gap_date_to': trip.date_to,
        }

    async def execute(self, trip: Union[TripCreate, TripUpdate]) -> int:
        await self.validate(trip)

        async with self.uow:
            if not self.trip_id:
                self.trip_id = await self._create(trip)
            else:
                await self._update(trip)

            if settings.IS_YA_TEAM:
                self.uow.add_job(
                    'create_or_update_staff_trip_task',
                    trip_id=self.trip_id,
                    unique=False,
                )

        return self.trip_id
