from datetime import date

from pydantic import BaseModel, Field, root_validator

from intranet.trip.src.lib.utils import unify_values


class LocalizedString(BaseModel):
    en: str = None
    ru: str = None


class City(BaseModel):
    name: LocalizedString


class Country(BaseModel):
    name: LocalizedString


class RoutePoint(BaseModel):
    """
    Точка в маршруте
    Считаем дату вылета и дату прибытия у смежных точек условно одинаковыми
    Пример для маршрута Москва - Питер - Сочи - Москва:
        будет созданы следующие поинты
        RoutePoint(
            city - Москва
            date - дата вылета из Москвы и дата прилета в Питер
        )
        RoutePoint(
            city - Питер
            date - дата вылета из Питер и дата прилета в Сочи
        )
        RoutePoint(
            city - Сочи
            date - дата вылета из Сочи и дата прилета в Москвы
        )
    """
    city: City
    country: Country
    date: date  # дата отправления из точки
    aeroclub_city_id: int = Field(None, deprecated=True)  # id города у поставшика
    provider_city_id: str = None  # id города у поставшика
    need_hotel: bool = True  # Если флаг = False ⇒ пользщователю не нужен отель в этой точке
    need_transfer: bool = True  # Если флаг = False ⇒ пользщователю не нужен трансфер из этой точки

    @root_validator(pre=True)
    def set_provider_city(cls, values):
        return unify_values(values, ['provider_city_id', 'aeroclub_city_id'])


class RouteMixin:
    @classmethod
    def get_city(cls, values, suffix, date_prefix):
        city = values.get(f'city_{suffix}')
        country = values.get(f'country_{suffix}')
        city_date = values.get(f'{date_prefix}_{suffix}')
        provider_city_id = values.get(f'provider_city_{suffix}_id')
        if city and country:
            return RoutePoint(
                city=City(name=LocalizedString(ru=city)),
                country=Country(name=LocalizedString(ru=country)),
                date=city_date,
                provider_city_id=provider_city_id,
            )
        return None

    @classmethod
    def get_city_list(cls, values, date_prefix='date'):
        city_list = []
        city_from = cls.get_city(values, 'from', date_prefix)
        city_to = cls.get_city(values, 'to', date_prefix)
        if city_from:
            city_list.append(city_from)
        if city_to:
            city_list.append(city_to)
        return city_list


def build_complex_route(route):
    if route is None:
        return route
    for point in route:
        point['city'] = {'name': {'ru': point['city']}}
        point['country'] = {'name': {'ru': point['country']}}
    return route
