from datetime import date, datetime

from pydantic import BaseModel, root_validator

from intranet.trip.src import enums
from intranet.trip.src.models.document import ExtPersonBonusCard, ExtPersonDocument


class Holding(BaseModel):
    holding_id: int
    name: str = None


class Company(BaseModel):
    company_id: int
    name: str = None
    aeroclub_company_id: int = None
    aeroclub_company_uid: str = None
    country: enums.Citizenship = None
    provider: enums.Provider = None
    default_active: bool = True
    holding_id: int = None


class UserUid(BaseModel):
    uid: str


class User(UserUid):
    person_id: int
    provider_profile_id: int = None
    user_ticket: str = None
    is_coordinator: bool = False
    is_limited_access: bool = False
    is_yandex_employee: bool = False

    login: str
    first_name: str
    last_name: str
    language: str = None
    timezone: str = None
    is_dismissed: bool = None
    is_active: bool = None
    email: str = None
    date_of_birth: date = None
    phone_number: str = None
    gender: enums.Gender = None
    display_covid_message: bool = False
    company_id: int = None
    company: Company = None
    is_offline_trip: bool = False
    email_confirmed_at: datetime = None
    rejected_at: datetime = None
    avatar_id: str = None

    @root_validator(pre=True)
    def fill_fields_depend_on_company(cls, values):
        company = values.get('company')
        if company is not None:
            values['display_covid_message'] = company['country'] == enums.Citizenship.KZ
            values['is_offline_trip'] = company['aeroclub_company_id'] is None
        return values


class PersonBase(BaseModel):
    """
    Общая часть для Person и ExtPerson
    """
    first_name: str
    first_name_en: str = None
    last_name: str
    last_name_en: str = None
    middle_name: str = None
    middle_name_en: str = None
    provider_profile_id: int = None
    external_uid: str = None
    email: str = None
    date_of_birth: date = None
    phone_number: str = None
    gender: enums.Gender = None


class Person(PersonBase):
    uid: str
    person_id: int = None

    login: str = None
    staff_id: int = None
    company_id: int = None
    company: Company = None
    timezone: str = None
    language: str = None

    first_name_ac: str = None
    first_name_ac_en: str = None
    last_name_ac: str = None
    last_name_ac_en: str = None
    middle_name_ac: str = None
    middle_name_ac_en: str = None

    is_dismissed: bool = None
    is_active: bool = None
    is_limited_access: bool = None
    is_offline_trip: bool = False

    support_id: int = None

    @property
    def is_belarusian(self):
        return self.company.country == enums.Citizenship.BY

    @root_validator(pre=True)
    def fill_fields_depend_on_company(cls, values):
        company = values.get('company')
        if company is not None:
            values['is_offline_trip'] = company['aeroclub_company_id'] is None
        return values


class ExtPerson(PersonBase):
    ext_person_id: int
    person_id: int
    name: str
    status: enums.ExtPersonStatus = enums.ExtPersonStatus.pending
    secret: str = None
    documents: list[ExtPersonDocument] = None
    bonus_cards: list[ExtPersonBonusCard] = None


class PersonRelationship(BaseModel):
    role: str
    owner_id: int
    dependant_id: int
    is_direct: bool
