from datetime import date, datetime
from pydantic import BaseModel, Field, validator, root_validator

from intranet.trip.src.enums import TripStatus
from intranet.trip.src.lib.utils import unify_values
from intranet.trip.src.models.common import RouteMixin, RoutePoint, build_complex_route
from intranet.trip.src.models.purpose import Purpose
from intranet.trip.src.models.person import Person
from intranet.trip.src.models.person_trip import PersonTrip


class ConfDetails(BaseModel):
    trip_id: int
    tracker_issue: str = None
    conf_date_from: date = None
    conf_date_to: date = None
    conference_name: str = None
    price: str = None
    ticket_type: str = None
    promo_code: str = None
    is_another_city: bool = False
    conference_url: str = None
    program_url: str = None
    participation_terms: str = None
    cancellation_terms: str = None


class Trip(BaseModel, RouteMixin):
    trip_id: int
    status: TripStatus

    created_at: datetime = None
    updated_at: datetime = None

    staff_trip_uuid: str = None
    issue_travel: str = None
    issue_conf: str = None
    date_from: date = None
    date_to: date = None
    city_from: str = None
    city_to: str = None
    country_from: str = None
    country_to: str = None
    comment: str = None

    purposes: list[Purpose] = None
    author: Person = None
    conf_details: ConfDetails = None
    person_trips: list[PersonTrip] = None

    aeroclub_city_from_id: int = Field(None, deprecated=True)
    provider_city_from_id: str = None

    aeroclub_city_to_id: int = Field(None, deprecated=True)
    provider_city_to_id: str = None

    chat_id: str = None

    route: list[RoutePoint] = None

    @validator('route', pre=True)
    def build_trip_complex_route(cls, v):
        return build_complex_route(v)

    @validator('route', always=True)
    def collect_route_if_empty(cls, v, values):
        if v is not None:
            return v
        city_list = cls.get_city_list(values)
        if len(city_list) == 1:
            raise ValueError('Either city_from or city_to is not filled')
        return city_list

    @property
    def route_as_string(self):
        if self.route is None:
            cities = [
                self.city_from,
                self.city_to,
                self.city_from,
            ]
        else:
            cities = [item.city.name.ru for item in self.route]
            cities.append(cities[0])
        return ' — '.join(cities)

    @root_validator(pre=True)
    def set_provider_cities(cls, values):
        values = unify_values(values, ['provider_city_from_id', 'aeroclub_city_from_id'])
        values = unify_values(values, ['provider_city_to_id', 'aeroclub_city_to_id'])
        return values
