import inspect
from functools import partial

from arq import cron

from intranet.trip.lib.arq_utils.settings import WorkerSettingsBase

from intranet.trip.src.config import settings
from intranet.trip.src.db import get_db_engine
from intranet.trip.src.redis import get_redis_settings
from intranet.trip.src.worker import tasks
from intranet.trip.src.worker import periodic_tasks


async def startup(ctx):
    ctx['db'] = await get_db_engine()


async def shutdown(ctx):
    if settings.ENABLE_PYSCOPG2:
        await ctx['db'].close()
    else:
        ctx['db'].close()
        await ctx['db'].wait_closed()


def is_job(is_cron: bool, task):
    if not inspect.isfunction(task):
        return False
    arq_settings = getattr(task, 'arq_config', None)
    if arq_settings is None:
        return False
    if not is_cron:
        return True
    return arq_settings['is_cron'] and arq_settings.get('predicate', True)


def get_jobs(is_cron: bool):
    validate_job = partial(is_job, is_cron)
    jobs_from_tasks = inspect.getmembers(tasks, validate_job)
    jobs_from_periodic_tasks = inspect.getmembers(periodic_tasks, validate_job)
    return [function for name, function in jobs_from_tasks + jobs_from_periodic_tasks]


def get_cron_jobs():
    jobs = get_jobs(is_cron=True)
    return [cron(job, **job.arq_config['cron_kwargs']) for job in jobs]


class WorkerSettings(WorkerSettingsBase):
    on_startup = startup
    on_shutdown = shutdown
    redis_settings = get_redis_settings()
    functions = get_jobs(is_cron=False)
    cron_jobs = get_cron_jobs()
    queue_name = settings.REDIS_QUEUE_NAME
