# coding: utf-8
from __future__ import unicode_literals

import attr
import codecs
from collections import defaultdict, OrderedDict
import importlib
import json
import logging
import os
import sys
import time

import tensorflow as tf
import yaml

from vins_core.utils.data import get_checksum, TarArchive, find_vinsfile
from vins_core.utils.app import compile_nlu, compile_custom_entities


APP_NAME = 'uhura'
MODULE_NAME = 'uhura.app'
CLASS_NAME = 'UhuraApp'


logger = logging.getLogger(__name__)


def set_logger_config():
    logging.basicConfig(level=logging.DEBUG, format='[%(asctime)s] [%(name)s] [%(levelname)s] %(message)s')

    tf.logging.set_verbosity(tf.logging.ERROR)

    logging.getLogger('requests.packages.urllib3.connectionpool').setLevel(logging.INFO)
    logging.getLogger('vins_core.utils.decorators').setLevel(logging.INFO)
    logging.getLogger('vins_core.ext.base').setLevel(logging.WARNING)


@attr.s
class SaveResult(object):
    app_name = attr.ib()
    checksum = attr.ib()
    archive = attr.ib()
    vinsfile = attr.ib()
    total_time = attr.ib()


def main():
    update_command_list(
        vins_file='uhura/Vinsfile.json',
        commandlist_file='uhura/vins_project/intents/utils/get_commands_list/get_commands_list.nlg'
    )
    erase_config()

    res = save_model(APP_NAME, classifiers=None, taggers=True, skip_train=False, feature_cache=None)

    update_config(res.checksum, res.archive, res.vinsfile)
    print "%s %.1fs: %s" % (res.app_name, res.total_time, res.checksum)

    return 0


def update_command_list(vins_file, commandlist_file):
    yamb_descriptions = defaultdict(list)
    tg_descriptions = defaultdict(list)

    with open(vins_file) as f:
        data = json.load(f, object_pairs_hook=OrderedDict)

    for include in data['project']['includes']:
        with open(include['path']) as f:
            project_data = yaml.load(f)

        for intent in project_data['intents']:
            if 'types' in intent:
                for type in intent['types']:
                    tg_descriptions[type].append((intent['description'], intent['examples']))
                    if 'yamb' not in intent:
                        yamb_descriptions[type].append((intent['description'], intent['examples']))

    with open(commandlist_file, 'w') as f:
        f.write('{% phrase what_do_you_know %}\n')
        f.write('Про что хочешь узнать подробнее?\n'.encode('utf-8'))
        f.write('{% endphrase %}\n\n')

        for type, descriptions in tg_descriptions.iteritems():
            f.write('{{% phrase tg_{} %}}\n'.format(type))
            for description in descriptions:
                if len(description) == 2:
                    f.write('{}\n\\n'.format(description[0]).encode('utf-8'))
                    for example in description[1]:
                        f.write('--- "{}"\n\\n'.format(example).encode('utf-8'))
                else:
                    f.write('{}\n\\n'.format(description).encode('utf-8'))
            f.write('\n{% endphrase %}\n\n')

        for type, descriptions in yamb_descriptions.iteritems():
            f.write('{{% phrase yamb_{} %}}\n'.format(type))
            for description in descriptions:
                if len(description) == 2:
                    f.write('{}\n\\n'.format(description[0]).encode('utf-8'))
                    for example in description[1]:
                        f.write('--- "{}"\n\\n'.format(example).encode('utf-8'))
                else:
                    f.write('{}\n\\n'.format(description).encode('utf-8'))
            f.write('\n{% endphrase %}\n\n')


def update_config(sha, archive, vins_file):
    with codecs.open(vins_file, encoding='utf8') as fin:
        data = json.load(fin, object_pairs_hook=OrderedDict)
    data['nlu']['compiled_model'] = {
        'sha256': sha,
        'path': os.path.basename(archive),
        'archive': 'TarArchive'
    }

    with codecs.open(vins_file, 'w', encoding='utf8') as f:
        json.dump(data, f, sort_keys=False, ensure_ascii=False, indent=2, separators=(',', ': '))
        f.write('\n')


def erase_config(vins_file='uhura/Vinsfile.json'):
    with codecs.open(vins_file, encoding='utf8') as fin:
        data = json.load(fin, object_pairs_hook=OrderedDict)
    data['nlu']['compiled_model'] = {}
    with codecs.open(vins_file, 'w', encoding='utf8') as f:
        json.dump(data, f, sort_keys=False, ensure_ascii=False, indent=2, separators=(',', ': '))
        f.write('\n')


def _create_app(vins_file, **kwargs):
    app_class = getattr(importlib.import_module(MODULE_NAME), CLASS_NAME)
    logger.info('%s imported from %s', MODULE_NAME, CLASS_NAME)
    app = app_class(vins_file=vins_file, load_data=True, load_custom_entities=False, **kwargs)
    if hasattr(app, 'pretrain_hook'):
        app.pretrain_hook()
    return app


def _compile_app(vins_file, archive, **kwargs):
    app = _create_app(vins_file, **kwargs)
    compile_nlu(app.nlu, archive, **kwargs)


def save_model(app_name, **kwargs):
    file_ = find_vinsfile(app_name)

    out = os.path.join(
        os.path.dirname(file_),
        app_name + '_model.tar.gz'
    )

    start = time.time()
    with TarArchive(out, mode='w:gz') as archive:
        _compile_app(file_, archive, **kwargs)

    tot_time = time.time() - start
    return SaveResult(app_name, get_checksum(out), out, file_, tot_time)


if __name__ == '__main__':
    sys.exit(main())
