# coding: utf-8

import codecs
import collections
import datetime
import functools
from hashlib import md5
import json
import os
import re

from gitchronicler import chronicler
from fabric.api import task, local

VINS_COMMIT_HASH_RE = r"COMMIT_HASH='(\w*)'"
conf = {
    'project': 'registry.yandex.net/tools/uhura',
}


def setup_dc():
    os.environ.setdefault('DC_BRIDGE_SUBNET', 'fc00::/8')
    os.environ.setdefault('MONGO_PORT', '27017')
    os.environ.setdefault('POSTGRES_PORT', '5432')
    os.environ.setdefault('UHURA_PORT', '80')


@task
def dc(cmd):
    setup_dc()
    local('docker-compose {0}'.format(cmd))


@task
def pull_model():
    conf['hash'] = _get_vins_project_hash()
    try:
        local('docker pull {project}-model:{hash}'.format(**conf))
    except SystemExit:
        print 'Model for current vins_project state not found'
        return False
    else:
        _download_model()
        return True


@task
def build_model(push=True, force=False):
    conf['hash'] = _get_vins_project_hash()
    if force or not pull_model():
        dc('build deps model')
        local('docker tag {project}-model:latest {project}-model:{hash}'.format(**conf))
        if push:
            local('docker push {project}-model:{hash}'.format(**conf))
        _download_model()


@task
def build_project(env='development', push=False, force_model=False, version=None):
    if not os.path.exists('uhura/uhura_model.tar.gz') or force_model:
        build_model()

    conf['version'] = get_version(env)
    if env == 'development' and version:
        conf['version'] = version

    dc('build deps app')
    if push:
        local('docker push {project}-deps:latest'.format(**conf))
        local('docker tag {project}:latest {project}:{version}'.format(**conf))
        local('docker push {project}:latest'.format(**conf))
        local('docker push {project}:{version}'.format(**conf))


@task
def build_tests(force_model=False):
    if not os.path.exists('uhura/uhura_model.tar.gz') or force_model:
        build_model()
    dc('build deps tests')


@task
def build(env='development', push=False, force_model=False, version=None):
    build_project(env, push, force_model, version)


@task
def run(cmd='', force_model=False):
    if not os.path.exists('uhura/uhura_model.tar.gz') or force_model:
        build_model()
    if not cmd:
        dc('run --service-ports -e POLLING=1 app')
    else:
        dc('run --service-ports app {}'.format(cmd))


@task
def yamb(force_model=False):
    if not os.path.exists('uhura/uhura_model.tar.gz') or force_model:
        build_model()
    dc('run -e POLLING=1 -e BOT_TYPE=yamb app')


@task
def test(test_path='', force_model=False, repl_mode=False):
    if not os.path.exists('uhura/uhura_model.tar.gz') or force_model:
        build_model()
    if repl_mode:
        test_path = 'bash'
    elif test_path:
        test_path = 'bash entrypoint-tests.sh "%s"' % test_path
    else:
        test_path = 'bash entrypoint-tests.sh'
    dc('run tests {}'.format(test_path))


@task
def deploy(env='production', qloud_env='testing', bot_type='all', only_deploy=False, is_qloud=False, version=None):
    if not only_deploy:
        build_project(env, push=True)
    conf['version'] = version or get_version(env)
    conf['qloud_env'] = qloud_env
    bot_type = bot_type.lower()
    assert bot_type in ['all', 'telegram', 'yamb'], "Invalid bot_type, not in  ['all', 'telegram', 'yamb']"
    if bot_type == 'all':
        bots = ['telegram', 'yamb']
    else:
        bots = [bot_type]

    for b in bots:
        if is_qloud:
            if b == 'telegram':
                conf['components'] = ','.join(['bot', 'celery', 'cron'])
            else:
                conf['components'] = 'bot'
            conf['bot_type'] = b
            local(
                'ya tool releaser deploy -v %(version)s -e %(bot_type)s-%(qloud_env)s -c %(components)s '
                '--deploy-comment-format "releasing version %(version)s"' % conf
            )
        else:
            if b == 'telegram':
                conf['components'] = ','.join(['bot:bot', 'celery:celery', 'cron:cron'])
            else:
                conf['components'] = 'bot:bot'
            conf['bot_type'] = b
            conf['box'] = 'bot'
            conf['stage'] = 'tools_uhura-{}_{}'.format(b, conf['qloud_env'])
            local(
                'ya tool releaser deploy -v %(version)s --deploy '
                '--stage %(stage)s --deploy-units %(components)s --box %(box)s '
                '--deploy-comment-format "releasing version %(version)s"' % conf
            )


@task
def release():
    # local('git checkout master')
    # local('git pull origin master')
    local('git fetch --tags')
    local('ya tool releaser changelog')
    local('ya tool releaser vcs-commit')
    local('ya tool releaser vcs-tag')
    local('ya tool releaser vcs-push')


def get_version(env):
    if env == 'development':
        version = 'latest'
    elif env == 'testing':
        version = _get_snapshot_version()
    elif env == 'production':
        version = chronicler.get_current_version()
    else:
        raise ValueError('Unknown env')
    return version


def _get_snapshot_version():
    version = chronicler.get_current_version()
    if version is None:
        raise ValueError
    snapshot = local(
        'git log --no-merges --format=%s {0}..HEAD | wc -l'.format(version),
        capture=True
    )
    return '{0}snapshot{1}'.format(version, snapshot)


@task
def wiki_talks_send_pull_request(talk_id='all'):
    branch = local('git symbolic-ref HEAD', capture=True).strip()
    if branch != 'refs/heads/master':
        raise ValueError('Your branch is "%s", please checkout on master')
    changes = local('git status -s | grep -v uhura/Vinsfile.json | wc -l', capture=True).strip()
    if changes != '0':
        raise ValueError('You have changes in your repo, please commit or stash them')
    now = datetime.datetime.utcnow().strftime('%Y-%m-%d-%H-%M-%S')
    new_branch_name = 'talks-patch-%s' % now
    local('git checkout -b %s' % new_branch_name)
    dc('run app uhura get_wiki_talks %s' % talk_id)
    commit_message = 'talks patch %s' % now
    local('git add uhura/ && git commit -m "%s"' % commit_message)
    local('git push origin %s' % new_branch_name)
    local('hub pull-request -m "%s"' % commit_message)


@task
def create_intent_template(intent_name):
    if not intent_name:
        raise ValueError('Specify intent name. Please run: "fab create_intent_template:awesome_intent_name"')
    subintent = False
    if len(intent_name.split('__')) > 1:
        subintent = True

    if subintent:
        long_dir_path = 'uhura/vins_project/intents/%s' % intent_name.split('__')[0]
        short_dir_path = 'intents/%s' % intent_name.split('__')[0]
        if not os.path.exists(long_dir_path):
            raise ValueError('Path of base intent "%s" is absent' % long_dir_path)
    else:
        long_dir_path = 'uhura/vins_project/intents/%s' % intent_name
        short_dir_path = 'intents/%s' % intent_name
        local('mkdir %s' % long_dir_path)
    long_path_template = long_dir_path + '/' + intent_name
    short_path_template = short_dir_path + '/' + intent_name
    local('touch %s.json' % long_path_template)
    if not subintent:
        local('touch %s.nlu' % long_path_template)
        local('touch %s.nlg' % long_path_template)

    vp_dm_sort_order = ['form', 'events', 'slots']
    vp_dm_intent_template = {
        'form': intent_name,
        'events': [{
            'event': 'submit',
            'handlers': [{
                'handler': 'callback',
                'name': intent_name,
            }],
        }],
        'slots': [],
    }
    _dump_ordered_dict(long_path_template + '.json', vp_dm_intent_template, vp_dm_sort_order, 4)

    vp_intent_template = {
        'intent': intent_name,
        'description': 'FILLME',
        'examples': ['FILLME'],
        'types': ['FILLME'],
        'reset_form': True,
        'fallback_threshold': 0.9,
        'trainable_classifiers': ['uhura'],
        'dm': {'path': short_path_template + '.json'},
        'nlu': [{
            'source': 'file',
            'path': short_path_template + '.nlu',
        }],
        'nlg': {'path': short_path_template + '.nlg'},
    }
    if subintent:
        del vp_intent_template['description']
        del vp_intent_template['examples']
        del vp_intent_template['types']
        del vp_intent_template['nlu']
        vp_intent_template['nlg']['path'] = short_path_template.split('__')[0] + '.nlg'
    vp_file = 'uhura/vins_project/VinsProjectfile.json'
    with open(vp_file) as f:
        vins_project = json.load(f)
    vins_project['intents'].append(vp_intent_template)

    vp_intent_sort_order = [
        'entity', 'path', 'intent', 'description', 'examples', 'types', 'yamb', 'reset_form',
        'fallback_threshold', 'total_fallback', 'trainable_classifiers', 'dm', 'nlu', 'nlg'
    ]
    for (k, v) in vins_project.iteritems():
        if isinstance(v, list):
            vins_project[k] = [_get_ordered_dict(val, vp_intent_sort_order) for val in v]
    vp_file_sort_order = ['name', 'entities', 'microintents', 'intents']
    _dump_ordered_dict(vp_file, vins_project, vp_file_sort_order, 2)


def _list_index(list_obj, idx):
    try:
        return list_obj.index(idx)
    except ValueError:
        return -1


def _get_ordered_dict(dict_obj, sort_order):
    return collections.OrderedDict(sorted(
        dict_obj.iteritems(), key=lambda (k, v): _list_index(sort_order, k)
    ))


def _dump_ordered_dict(file_path, dict_obj, sort_order, indent=2):
    ordered_dict = _get_ordered_dict(dict_obj, sort_order)
    with codecs.open(file_path, 'w', encoding='utf-8') as f:
        json.dump(ordered_dict, f, indent=indent, ensure_ascii=False, separators=(',', ': '))
        f.write('\n')


def _sorted_walk(top):
    names = os.listdir(top)
    names.sort()
    dirs, nondirs = [], []

    for name in names:
        if os.path.isdir(os.path.join(top, name)):
            dirs.append(name)
        else:
            nondirs.append(name)

    for name in dirs:
        path = os.path.join(top, name)
        if not os.path.islink(path):
            for x in _sorted_walk(path):
                yield x
    yield top, dirs, nondirs


def _get_vins_project_hash():
    root_dir = 'uhura/vins_project'
    possible_types = ['.nlu', '.json', '.txt', '.yaml']
    chunk_size = 65536
    digest = md5()
    for root, dirnames, filenames in _sorted_walk(root_dir):
        for filename in filenames:
            name, extension = os.path.splitext(filename)
            if extension in possible_types and not filename.startswith('transition'):
                with open(os.path.join(root, filename), 'rb') as f:
                    [digest.update(chunk) for chunk in iter(functools.partial(f.read, chunk_size), '')]
    with open('deps/install-vins.sh', 'r') as f:
        digest.update(re.findall(VINS_COMMIT_HASH_RE, f.read())[0])
    return digest.hexdigest()


def _download_model():
    local('docker create --name tmp_uhura_model_container {project}-model:{hash} sh'.format(**conf))
    local('docker cp tmp_uhura_model_container:/app/uhura/uhura_model.tar.gz ./uhura')
    local('docker cp tmp_uhura_model_container:/app/uhura/Vinsfile.json ./uhura')
    local(
        'docker cp '
        'tmp_uhura_model_container:/app/uhura/vins_project/intents/utils/get_commands_list/get_commands_list.nlg '
        './uhura/vins_project/intents/utils/get_commands_list'
    )
    local('docker rm tmp_uhura_model_container')
