# coding: utf-8

from __future__ import unicode_literals

import json
import logging
import os
import re
import shutil
from datetime import datetime, timedelta

import dateparser
import jinja2
import requests
import waffle
import yenv
from django import db as django_db
from django.conf import settings
from django.db.models import Q
from django.forms import model_to_dict
from requests.packages.urllib3.exceptions import InsecureRequestWarning

from uhura.models import TelegramUsername

if os.environ.get('USE_TENSORFLOW'):
    # Так мы не будем триггерить инициализацию тензорфлоу на каждую команду ухуры
    from vins_core import logger as vins_logger
    from vins_core.nlu.flow_nlu_factory.transition_model import register_transition_model
    from vins_core.utils.data import find_vinsfile
    from vins_sdk.app import VinsApp, callback_method
else:
    VinsApp = object
    callback_method = staticmethod

from uhura import models
from uhura.external import bot
from uhura.external import calendar
from uhura.external import forms as yandex_forms
from uhura.external import gap
from uhura.external import intranet
from uhura.external import staff
from uhura.external import startrek
from uhura.external import suggest
from uhura.external import wiki
from uhura.external import yavision
from uhura.external.botan import botan
from uhura.external.jing import jing
from uhura.idm.utils import can_use_uhura
from uhura.lib import cache_storage
from uhura.lib import callbacks
from uhura.lib import mail
from uhura.lib import session_storage
from uhura.lib import tasha
from uhura.lib.vins import connectors
from uhura.lib.vins.transition import transition_model
from uhura.lib.vins.response import UhuraVinsResponse
from uhura.utils import datetimes
from uhura.utils import exceptions
from uhura.utils import strings
from uhura.utils import word_assert

jinja_env = jinja2.environment.Environment()
logger = logging.getLogger(__name__)
requests.packages.urllib3.disable_warnings(InsecureRequestWarning)
word_regexp = re.compile(r'[\w\s-]', re.UNICODE)
COMMANDS_LIST_KEYBOARD = [
    'Столовая', 'Парковка', 'Стартрек', 'Стафф', 'Wifi', 'Календарь/Гэп', 'Поиск', 'ДМС', 'Другое'
]
YES_NO_KEYBOARD = ['Да', 'Нет']


class TelegramRetry(Exception):
    pass


class UhuraApp(VinsApp):
    def __init__(self, vins_file=None, **kwargs):
        register_transition_model('uhura', transition_model.create_transition_model)
        session_storage_obj = session_storage.PostgressSessionStorage()
        vins_file = vins_file or find_vinsfile('uhura')
        self.connector = None
        super(UhuraApp, self).__init__(
            vins_file=vins_file,
            app_id='uhura',
            session_storage=session_storage_obj,
            **kwargs
        )

    def authenticate_and_return_user(self, req_info):
        raise NotImplementedError()

    def get_person_data(self, req_info):
        raise NotImplementedError()

    def get_person_data_from_staff(self, req_info):
        raise NotImplementedError()

    def change_intent_to_empty(self, session, req_info, response):
        vins_logger.log_dialog_state(session, req_info, response)
        self.save_session(session, req_info)
        if waffle.switch_is_active('use_botan'):
            self.track_message(req_info)
        new_form = self.new_form('empty_intent', 'utils')
        self.change_form(session=session, form=new_form, req_info=req_info, response=response)

    def new_form(self, form_name, module_name):
        return super(UhuraApp, self).new_form('uhura.{}.{}'.format(module_name, form_name))

    def track_message(self, req_info):
        session = self._load_session(req_info)
        form = session.form
        form_name = form.name
        intent_name = form_name.split('.')[-1]
        message = req_info.additional_options.get('message', {})
        if intent_name != 'empty_intent':
            botan.track(req_info.uuid, message, intent_name)

    def update_user_table(self, req_info):
        raise NotImplementedError()

    @django_db.transaction.atomic
    def handle_request(self, req_info, **kwargs):
        try:
            user = self.authenticate_and_return_user(req_info)
        except Exception:
            logger.exception('Can\'t check user authorization:')
            user = None

        try:
            session = self.load_or_create_session(req_info)
        except django_db.DatabaseError:
            raise TelegramRetry('Session is locked')

        response = UhuraVinsResponse()
        new_intent_name = None
        phrase_id = None
        if user:
            put_user(req_info, user)
            if not can_use_uhura(user):
                new_intent_name = 'external_affiliation'
            elif word_assert.is_cancel(req_info.utterance.text):
                new_intent_name = 'empty_intent'
                phrase_id = 'cancel'
            else:
                response = super(UhuraApp, self).handle_request(req_info, response_class=UhuraVinsResponse, **kwargs)
        else:
            if user is None:
                new_intent_name = 'staff_timeout'
            else:
                new_intent_name = 'not_permitted'
                req_info.additional_options['not_permitted'] = True
                if word_assert.is_cancel(req_info.utterance.text):
                    req_info.additional_options['not_permitted_cancel'] = True

        if new_intent_name is not None:
            new_form = self.new_form('{}'.format(new_intent_name), 'utils')

            if phrase_id is not None:
                text = self.render_phrase(phrase_id=phrase_id, form=new_form).text
                response.reply(text)
            self.change_form(session=session, form=new_form, req_info=req_info, response=response)
            self.save_session(session, req_info)

        if waffle.switch_is_active('use_botan'):
            self.track_message(req_info)
        return response

    @callback_method
    def handle_nlg_with_cancel_button(self, req_info, form, session, response, **kwargs):
        phrase_id = kwargs['phrase_id']
        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text, cancel_button=True)

    @callback_method
    def not_permitted(self, req_info, form, session, response, **kwargs):
        raise NotImplementedError()

    def _format_label_and_helptext(self, label, help_text):
        raise NotImplementedError()

    @callback_method
    def normalize_login(self, req_info, form, session, response, **kwargs):
        login = form.login.value
        if len(login.split()) > 1:
            if '_'.join(login.split()) in req_info.utterance.text:
                form.login.set_value('_'.join(login.split()), 'string')
            else:
                form.login.set_value(login.replace(' - ', '-'), 'string')
            return
        min_dist = len(login)
        min_word = None
        text = ''.join(word_regexp.findall(req_info.utterance.text))
        for x in text.split():
            cur_dist = strings.edit_distance(login, x)
            if cur_dist < min_dist:
                min_dist = cur_dist
                min_word = x
        form.login.set_value(min_word, 'string')

    @callback_method
    def check_aurora_parking(self, req_info, form, session, response, **kwargs):
        params = {
            'target': 'eng.office.avrora.common.parking.one_min',
            'until': 'now',
            'from': 'now',
            'format': 'json'
        }
        try:
            if intranet.get_request(settings.GRAPHITE_URL, params=params)[0]['datapoints'][0][0] == 0:
                phrase_id = 'available'
            else:
                phrase_id = 'not_available'
        except Exception:
            phrase_id = 'error'
            logger.exception('check_aurora_parking raised exception')

        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text)

    @callback_method
    def prepare_car_plate(self, req_info, form, session, response, **kwargs):
        cancel_button = False
        files = self.connector.download_message_files(req_info.additional_options['message'])
        if files is None:
            return
        plates = yavision.get_car_numbers_from_image(files[0])
        os.remove(files[0])
        if plates is None:
            phrase_id = 'recognition_error'
        elif not plates:
            phrase_id = 'not_recognized'
        else:
            phrase_id = 'suggest_plates'
            cancel_button = True
        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text, plates, cancel_button=cancel_button)

    @callback_method
    def create_gap_shortcut(self, req_info, form, session, response, **kwargs):
        new_form = self.new_form('create_gap__type', 'gap')
        intent_name = form.name.split('.')[-1]
        gap_type = gap.FORMAT_DICT[intent_name[intent_name.find('_') + 1:]]
        new_form.type.set_value(gap_type, 'string')
        new_form.from_create_gap_shortcut.set_value(True, 'bool')
        self.change_form(session=session, form=new_form, req_info=req_info, response=response)

    @callback_method
    def dont_understand(self, req_info, form, session, response, **kwargs):
        text = self.render_phrase(phrase_id='dont_understand', form=form).text
        keyboard = ['Help', 'Feedback']
        response.reply(text, keyboard, cancel_button=True)

    @callback_method
    def get_changelog(self, req_info, form, session, response, **kwargs):
        try:
            grid = wiki.get_wiki_grid(settings.RELEASE_GRID)
            changelog = grid['rows'][0][2]['raw'].replace('\n', '\\n')
            date = grid['rows'][0][1]['raw']
            number = grid['rows'][0][0]['raw']
            phrase_id = 'success'
            context = {'changelog': changelog, 'date': date, 'number': number}
        except (KeyError, TypeError):
            phrase_id = 'error'
            context = {}
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text)

    @callback_method
    def get_mobcert_password(self, req_info, form, session, response, **kwargs):
        context = {}
        try:
            password = bot.get_mobcert_password()
            phrase_id = 'wifi'
            context['password'] = password
        except Exception:
            logger.exception('get_mobcert_password raised exception')
            phrase_id = 'error'
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text.split('\n'))

    @callback_method
    def get_ticket(self, req_info, form, session, response, **kwargs):
        context = {}
        try:
            key = '-'.join(form.key.value.split('/')[-1].split()).upper()
            ticket = startrek.get_ticket(key)
        except startrek.WrongTicketKey:
            phrase_id = 'wrong_format'
        except Exception:
            phrase_id = 'error'
        else:
            context = ticket
            if ticket is None:
                phrase_id = 'forbidden'
            else:
                phrase_id = 'ticket'
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text)
        self.change_intent_to_empty(session, req_info, response)

    @callback_method
    def get_commands_list(self, req_info, form, session, response, **kwargs):
        keyboard = COMMANDS_LIST_KEYBOARD
        message = self.render_phrase(phrase_id='what_do_you_know', form=form).text
        response.reply(message, keyboard)

    @callback_method
    def get_shuttle_schedule(self, req_info, form, session, response, **kwargs):
        if form.place.value == 'benua':
            fin_from_layer_id = settings.CALENDAR_LAYERS['shuttle_benua_from_fin']
            fin_to_layer_id = settings.CALENDAR_LAYERS['shuttle_benua_to_fin']
            nov_from_layer_id = settings.CALENDAR_LAYERS['shuttle_benua_from_nov']
            nov_to_layer_id = settings.CALENDAR_LAYERS['shuttle_benua_to_nov']

            fin_from_schedule, fin_from_day = callbacks.get_shuttle_schedule(fin_from_layer_id)
            fin_to_schedule, fin_to_day = callbacks.get_shuttle_schedule(fin_to_layer_id)
            nov_from_schedule, nov_from_day = callbacks.get_shuttle_schedule(nov_from_layer_id)
            nov_to_schedule, nov_to_day = callbacks.get_shuttle_schedule(nov_to_layer_id)

            if all((fin_from_schedule, fin_to_schedule, nov_from_schedule, nov_to_schedule)):
                phrase_id = 'benua'
            else:
                phrase_id = 'error'
            context = {
                'fin_from': fin_from_schedule,
                'fin_from_day': fin_from_day,
                'fin_to': fin_to_schedule,
                'fin_to_day': fin_to_day,
                'nov_from': nov_from_schedule,
                'nov_from_day': nov_from_day,
                'nov_to': nov_to_schedule,
                'nov_to_day': nov_to_day,
            }
        else:
            from_layer_id = settings.CALENDAR_LAYERS['shuttle_avrora_from']
            to_layer_id = settings.CALENDAR_LAYERS['shuttle_avrora_to']

            from_schedule, from_day = callbacks.get_shuttle_schedule(from_layer_id)
            to_schedule, to_day = callbacks.get_shuttle_schedule(to_layer_id)

            if all((from_schedule, to_schedule)):
                phrase_id = 'avrora'
            else:
                phrase_id = 'error'
            context = {
                'from': from_schedule,
                'from_day': from_day,
                'to': to_schedule,
                'to_day': to_day,
            }

        text = self.render_phrase(form=form, phrase_id=phrase_id, context=context).text
        response.reply(text)

    @callback_method
    def get_wifi_password(self, req_info, form, session, response, **kwargs):
        context = {}
        try:
            password = bot.get_wifi_password()
            context['password'] = password
            phrase_id = 'wifi'
        except Exception:
            logger.exception('get_wifi_password raised exception')
            phrase_id = 'error'
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text.split('\n'))

    @callback_method
    def get_last_online(self, req_info, form, session, response, **kwargs):
        context = {}
        keyboard = None
        cancel_button = False
        try:
            suggested_people, phrase_id = suggest.find_person(form.login.value)
            if phrase_id != 'person_answer':
                context['people'] = suggested_people
                keyboard = [x['login'] for x in suggested_people]
                if keyboard:
                    cancel_button = True
            else:
                login = suggested_people[0]['login']
                response.reply(staff.get_last_online(login))
                return
        except ValueError:
            phrase_id = 'wrong_format'
        except Exception:
            phrase_id = 'error'
            logger.exception('get_last_online raised exception')
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text, keyboard, cancel_button=cancel_button)

    def missing_token_context(self, credentials):
        return {
            'user_has_token': credentials.user_has_token,
            'redirect_url': settings.OAUTH_REDIRECT_URL,
            'oauth_client_id': settings.OAUTH_CLIENT_ID,
        }

    @callback_method
    def get_current_meeting(self, req_info, form, session, response, **kwargs):
        person_data = self.get_person_data(req_info)
        credentials = calendar.CalendarRequestCredentials(person_data)
        uid = person_data['uid']
        events = callbacks.get_current_meeting(uid, credentials)
        if events is None:
            phrase_id = 'error'
        else:
            if not events:
                events = callbacks.get_next_meeting(uid, credentials)
                if events is None:
                    phrase_id = 'error'
                else:
                    phrase_id = 'next_meeting'
            else:
                phrase_id = 'current_meeting'
        calendar.cast_timestamps_to_readable_format(events)
        context = {'events': events}
        context.update(self.missing_token_context(credentials))
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text)

    @callback_method
    def get_next_meeting(self, req_info, form, session, response, **kwargs):
        person_data = self.get_person_data(req_info)
        credentials = calendar.CalendarRequestCredentials(person_data)
        events = callbacks.get_next_meeting(person_data['uid'], credentials)
        if events is None:
            phrase_id = 'error'
        else:
            phrase_id = 'meeting'
        calendar.cast_timestamps_to_readable_format(events)
        context = {'events': events}
        context.update(self.missing_token_context(credentials))
        render_result = self.render_phrase(phrase_id=phrase_id, form=form, context=context)
        response.reply(render_result.text)

    @callback_method
    def get_meetings_by_date(self, req_info, form, session, response, **kwargs):
        context = {}
        request_user = self.get_person_data(req_info)
        credentials = calendar.CalendarRequestCredentials(request_user)
        if form.login.value:
            suggested_people, suggest_phrase_id = suggest.find_person(form.login.value)
            if suggest_phrase_id == 'person_answer':
                login = suggested_people[0]['login']
                uid = staff.get_uid_by_login(login)
                context['login'] = login
            else:
                cancel_button = False
                context = {'people': suggested_people}
                text = self.render_phrase(phrase_id=suggest_phrase_id, form=form, context=context).text
                keyboard = [x['login'] for x in suggested_people]
                if keyboard:
                    cancel_button = True
                response.reply(text, keyboard, cancel_button=cancel_button, buttons_in_row_count=3)
                return
        else:
            uid = request_user['uid']

        from_date, to_date = datetimes.process_date_range(form.date_range.value, form.date_range.value_type)
        try:
            request = calendar.MeetingsRequest(uid, from_date, to_date, credentials)
            events = callbacks.get_meetings_by_date(request)
        except ValueError:
            phrase_id = 'wrong_dates'
        else:
            if events is None:
                phrase_id = 'error'
            else:
                phrase_id = 'meetings'
            calendar.cast_timestamps_to_readable_format(events)
            context.update({
                'events': events,
                'from_date': from_date,
                'to_date': to_date,
            })

        context.update(self.missing_token_context(credentials))
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply([x.strip() for x in text.split('\n\n') if x])

        self.change_intent_to_empty(session, req_info, response)

    @callback_method
    def get_food_balance(self, req_info, form, session, response, **kwargs):
        context = {}
        try:
            login = self.get_person_data(req_info)['login']
            food_balance_data = staff.get_food_balance(login)
            context['food_limit_month'] = float(food_balance_data['food_limit_month'])
            context['food_balance_month'] = float(food_balance_data['food_balance_month'])
            context['food_balance_day'] = float(food_balance_data['food_balance_day'])
            context['remaining_work_days'] = food_balance_data['food_balance_remaining_work_days']
            context['food_balance_recommended'] = float(food_balance_data['food_balance_recommended'])
            context['food_overspending'] = float(food_balance_data['food_overspending'])
            phrase_id = 'success'
        except (TypeError, KeyError, AttributeError):
            phrase_id = 'error'

        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text)

    @callback_method
    def get_vacation(self, req_info, form, session, response, **kwargs):
        context = {}
        try:
            login = self.get_person_data(req_info)['login']
            staff_meta_profile = staff.get_profile(login)
            context['vacation'] = int(float(staff_meta_profile.get('vacation', 0)))
            context['paid_day_off'] = int(float(staff_meta_profile.get('paid_day_off', 0)))
            phrase_id = 'success'
        except (TypeError, KeyError, AttributeError):
            phrase_id = 'error'

        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text)

    @callback_method
    def get_password_expired_at(self, req_info, form, session, response, **kwargs):
        context = {}
        try:
            login = self.get_person_data(req_info)['login']
            staff_meta_profile = staff.get_profile(login)
            expired_at_date = datetimes.str_to_datetime(staff_meta_profile['password_expired_at'])
            context['date'] = datetimes.datetime_to_readable_format(expired_at_date)
            context['diff'] = staff_meta_profile['password_days_reminder']
            phrase_id = 'success'
        except (TypeError, KeyError, AttributeError):
            phrase_id = 'error'

        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text)

    @callback_method
    def feedback__message(self, req_info, form, session, response, **kwargs):
        form.message.set_value(req_info.utterance.text, 'string')
        text = self.render_phrase(phrase_id='success', form=form).text
        response.reply(text)
        self.change_intent_to_empty(session, req_info, response)

    @callback_method
    def find_car_owner(self, req_info, form, session, response, **kwargs):
        plate = form.plate.value
        context = {}
        keyboard = None
        cancel_button = False
        owners = suggest.find_car_owner(plate)

        if owners is None:
            phrase_id = 'error'
        elif len(owners) == 1:
            phrase_id = 'answer'
        elif len(owners) > 1:
            plates = [x['plate'] for x in owners]
            if len(set(plates)) == 1:
                phrase_id = 'answer'
            else:
                phrase_id = 'choose_plate'
                keyboard = plates
                cancel_button = True
        else:
            phrase_id = 'car_not_found'
        context['owners'] = owners
        if phrase_id == 'answer':
            request_user = self.get_person_data(req_info)
            credentials = calendar.CalendarRequestCredentials(request_user)

            for i in range(len(owners)):
                login = owners[i]['login']
                owners[i]['online'] = staff.get_last_online(login)
                owners[i]['format_dict'] = gap.FORMAT_DICT
                owners[i].update(callbacks.get_person_info(login, credentials))

        context['plate'] = form.plate.value
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        if phrase_id == 'answer':
            text = [x for x in text.split('\n\n') if x]
        response.reply(text, keyboard, buttons_in_row_count=3, cancel_button=cancel_button)

    @callback_method
    def find_table(self, req_info, form, session, response, **kwargs):
        context = {}
        keyboard = None
        cancel_button = False
        request_user = self.get_person_data(req_info)
        credentials = calendar.CalendarRequestCredentials(request_user)
        try:
            suggested_people, phrase_id = suggest.find_person(form.login.value)
            if phrase_id != 'person_answer':
                context['people'] = suggested_people
                if phrase_id == 'person_choose':
                    keyboard = [x['login'] for x in suggested_people]
                    cancel_button = True
            else:
                login = suggested_people[0]['login']
                info = callbacks.get_person_info(login, credentials)
                if info is None:
                    phrase_id = 'person_not_found'
                else:
                    phrase_id = 'person_answer'
                    context['person'] = info
                    context['person']['format_dict'] = gap.FORMAT_DICT
                    context['person']['online'] = staff.get_last_online(login)
        except ValueError:
            phrase_id = 'wrong_format'
        except Exception:
            logger.exception('find_table raised exception')
            phrase_id = 'error'
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text, keyboard, buttons_in_row_count=3, cancel_button=cancel_button)

    @callback_method
    def find_meeting_room(self, req_info, form, session, response, **kwargs):
        suggested_rooms, phrase_id = suggest.find_meeting_room(form['name'])
        context = {'rooms': suggested_rooms}
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text)

    @callback_method
    def find_meeting_room_or_table(self, req_info, form, session, response, **kwargs):
        context = {}
        keyboard = None
        cancel_button = False
        request_user = self.get_person_data(req_info)
        credentials = calendar.CalendarRequestCredentials(request_user)
        try:
            suggest_response, phrase_id = suggest.find_meeting_room_or_table(form.login.value)
            if phrase_id.startswith('person'):
                if not phrase_id.endswith('answer'):
                    context['people'] = suggest_response
                    if phrase_id.endswith('choose'):
                        keyboard = [x['login'] for x in suggest_response]
                        cancel_button = True
                else:
                    login = suggest_response[0]['login']
                    info = callbacks.get_person_info(login, credentials)
                    if info is None:
                        phrase_id = 'not_found'
                    else:
                        context['person'] = info
                        context['person']['format_dict'] = gap.FORMAT_DICT
                        context['person']['online'] = staff.get_last_online(login)
            elif phrase_id == 'choose':
                keyboard = [x.get('login') or 'Переговорка ' + x.get('title') for x in suggest_response]
                context['result'] = suggest_response
                cancel_button = True
            else:
                context = {'rooms': suggest_response}
        except ValueError:
            phrase_id = 'wrong_format'
        except Exception:
            logger.exception('find_person_or_meeting_room raised exception')
            phrase_id = 'error'
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text, keyboard, buttons_in_row_count=3, cancel_button=cancel_button)

    @callback_method
    def get_top5_tickets(self, req_info, form, session, response, **kwargs):
        login = self.get_person_data(req_info)['login']
        tickets = callbacks.get_top5_tickets(login)
        phrase_id = 'tickets' if tickets is not None else 'error'
        text = self.render_phrase(phrase_id=phrase_id, form=form, context={'tickets': tickets}).text
        response.reply(text)

    @callback_method
    def get_tickets_inProgress(self, req_info, form, session, response, **kwargs):
        login = self.get_person_data(req_info)['login']
        tickets = callbacks.get_tickets_inProgress(login)
        phrase_id = 'tickets' if tickets is not None else 'error'
        text = self.render_phrase(phrase_id=phrase_id, form=form, context={'tickets': tickets}).text
        response.reply(text)

    @callback_method
    def ask_queue(self, req_info, form, session, response, **kwargs):
        message = self.render_phrase(phrase_id='ask_queue', form=form).text
        response.reply(message, cancel_button=True)

    @callback_method
    def create_ticket(self, req_info, form, session, response, **kwargs):
        keyboard = None
        cancel_button = False
        if form.queue.value is not None and form.dont_skip.value is None:
            new_form = self._dm.new_form(
                'uhura.startrek.create_ticket__ellipsis',
                response=response,
                req_info=req_info,
                app=self
            )
            new_form.queue.set_value(form.queue.value, 'string')
            new_form.dont_skip.set_value('true', 'string')
            self.change_form(session=session, form=new_form, req_info=req_info, response=response)
            return
        else:
            queue_name = form.queue.value.encode('utf-8', 'ignore')
            try:
                can_access_queue = startrek.check_queue(queue_name.upper())
                if can_access_queue is None:
                    phrase_id = 'doesnt_exist'
                elif can_access_queue:
                    phrase_id = 'ask_ticket_title'
                else:
                    phrase_id = 'no_access'
            except Exception:
                phrase_id = 'error'

            if phrase_id == 'ask_ticket_title':
                cancel_button = True
            else:
                keyboard = YES_NO_KEYBOARD
                form.queue.reset_value()
                form.wrong_queue.set_value('1', 'string')
        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text, keyboard, cancel_button=cancel_button)

    @callback_method
    def create_ticket__summary(self, req_info, form, session, response, **kwargs):
        cancel_button = False
        change_to_empty = False
        if form.wrong_queue.value:
            if word_assert.is_yes(req_info.utterance.text):
                new_form = self._dm.new_form(
                    'uhura.startrek.create_ticket', response=response, req_info=req_info, app=self
                )
                self.change_form(session=session, form=new_form, req_info=req_info, response=response)
                return
            else:
                phrase_id = 'cancel_button'
                change_to_empty = True
        else:
            form.summary.set_value(req_info.utterance.text, 'string')
            phrase_id = 'ask_ticket_description'
            cancel_button = True
        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text, cancel_button=cancel_button)
        if change_to_empty:
            self.change_intent_to_empty(session, req_info, response)

    @callback_method
    def create_ticket__summary__description(self, req_info, form, session, response, **kwargs):
        form.description.set_value(req_info.utterance.text, 'string')
        phrase_id = 'ask_confirmation'
        keyboard = YES_NO_KEYBOARD
        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text, keyboard)

    @callback_method
    def create_ticket__summary__description__confirmed(self, req_info, form, session, response, **kwargs):
        context = {}
        if word_assert.is_yes(req_info.utterance.text):
            login = self.get_person_data(req_info)['login']
            ticket_link = startrek.create_ticket(
                login,
                form.queue.value,
                form.summary.value,
                form.description.value
            )
            if ticket_link is not None:
                context['ticket_link'] = ticket_link
                phrase_id = 'show_ticket_link'
            else:
                phrase_id = 'final_error'
        else:
            phrase_id = 'do_not_create'
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text)

    @callback_method
    def get_gaps(self, req_info, form, session, response, **kwargs):
        try:
            suggested_people, phrase_id = suggest.find_person(form.login.value)
        except ValueError:
            render_result = self.render_phrase(phrase_id='wrong_format', form=form)
            response.reply(render_result.text)
            return

        if phrase_id != 'person_answer':
            context = {'people': suggested_people}
            text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
            keyboard = [x['login'] for x in suggested_people]
            response.reply(text, keyboard, buttons_in_row_count=3)
            return
        else:
            login = suggested_people[0]['login']

        from_date, to_date = datetimes.process_date_range(form.date_range.value, form.date_range.value_type)

        try:
            gaps_list = callbacks.get_gaps(login, from_date, to_date)
        except gap.GapWrongDatesError:
            phrase_id = 'wrong_dates'
            render_result = self.render_phrase(phrase_id=phrase_id, form=form)
        else:
            if gaps_list is None:
                phrase_id = 'error'
            else:
                phrase_id = 'gaps'
            gap.cast_timestamps_to_readable_format(gaps_list)
            render_result = self.render_phrase(
                phrase_id=phrase_id,
                form=form,
                context={'format_dict': gap.FORMAT_DICT, 'gaps': gaps_list}
            )
        messages = [x for x in render_result.text.split('\n\n') if x]
        response.reply(messages)

    @callback_method
    def empty_callback(self, req_info, form, session, response, **kwargs):
        pass

    @callback_method
    def work_from_home(self, req_info, form, session, response, **kwargs):
        message = self.render_phrase(phrase_id='ask_confirmation', form=form, context={'date': datetime.now()}).text
        keyboard = YES_NO_KEYBOARD
        response.reply(message, keyboard)

    @callback_method
    def work_from_home__confirmed(self, req_info, form, session, response, **kwargs):
        if word_assert.is_yes(form.confirmed.value):
            login = self.get_person_data(req_info)['login']
            try:
                r = callbacks.work_from_home(login, date=datetime.now())
            except gap.GapIntersectionError:
                phrase_id = 'gap_exists'
            else:
                if r is None:
                    phrase_id = 'error'
                else:
                    phrase_id = 'success'
        else:
            phrase_id = 'cancel'
        render_result = self.render_phrase(phrase_id=phrase_id, form=form)
        response.reply(render_result.text)

    @callback_method
    def create_gap__type__start__end__comment__work__confirmed(self, req_info, form, session, response, **kwargs):
        if word_assert.is_no(req_info.utterance.text):
            phrase_id = 'cancel'
        else:
            staff_obj = self.get_person_data(req_info)
            login = staff_obj['login']
            try:
                from_date = datetimes.str_to_datetime(form.start.value)
                to_date = datetimes.str_to_datetime(form.end.value)
                r = gap.create_gap(
                    login=login,
                    from_date=from_date,
                    to_date=to_date,
                    message=form.comment.value,
                    workflow=form.type.value.lower(),
                    has_sicklist=form.sicklist.value.lower() == 'да' if form.sicklist.value else None,
                    work_in_absence=word_assert.is_yes(form.work.value)
                )
            except gap.GapIntersectionError:
                phrase_id = 'intersection'
            except gap.GapWrongDatesError:
                phrase_id = 'wrong_dates'
            except gap.GapPaidDayOffCountError:
                phrase_id = 'paid_day_off_limit_exceeded'
            except Exception:
                logger.exception('create_gap raised exception')
                phrase_id = 'error'
            else:
                if r is not None:
                    phrase_id = gap.REVERSED_FORMAT_DICT[form.type.value.lower()]
                else:
                    phrase_id = 'error'
        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text)
        self.change_intent_to_empty(session, req_info, response)

    @callback_method
    def get_nearest_free_time(self, req_info, form, session, response, **kwargs):
        now = datetime.now()
        context = {}
        keyboard = None
        cancel_button = False
        request_user = self.get_person_data(req_info)
        credentials = calendar.CalendarRequestCredentials(request_user)

        try:
            suggested_people, phrase_id = suggest.find_person(form.login.value)
            if phrase_id != 'person_answer':
                context['people'] = suggested_people
                keyboard = [x['login'] for x in suggested_people]
                if keyboard:
                    cancel_button = True
            else:
                person_info = callbacks.get_person_info(suggested_people[0]['login'], credentials)
                context['gender'] = person_info['gender'][0]
                context['name'] = suggested_people[0]['name']
                uid = suggested_people[0]['uid']

                request = calendar.MeetingsRequest(uid, now, now + timedelta(days=1), credentials)
                events = calendar.get_meetings(request)
                if events is None:
                    phrase_id = 'error'
                elif not events:
                    phrase_id = 'always_free'
                else:
                    tomorrow_str = datetimes.datetime_to_str(now + timedelta(days=1))
                    events.append({'startTs': tomorrow_str, 'endTs': tomorrow_str})
                    timestamps = []
                    for x in events:
                        timestamps += [
                            {'startTs': datetimes.str_to_datetime(x['startTs'])},
                            {'endTs': datetimes.str_to_datetime(x['endTs'])}
                        ]
                    timestamps = sorted(timestamps, key=lambda x: x.values()[0])
                    started_events_count = 0
                    free_time_start = None
                    prev_end = now
                    for obj in timestamps:
                        if started_events_count == 0:
                            if 'startTs' in obj:
                                new_start = obj['startTs']
                                free_time_duration = new_start - prev_end
                                if free_time_duration >= timedelta(minutes=15):
                                    free_time_end = new_start
                                    free_time_start = prev_end
                                    break
                                else:
                                    started_events_count += 1
                            else:
                                prev_end = obj['endTs']
                        else:
                            if 'startTs' in obj:
                                started_events_count += 1
                            elif 'endTs' in obj:
                                started_events_count -= 1
                                prev_end = obj['endTs']
                    if free_time_start:
                        context.update({
                            'start': datetimes.datetime_to_readable_format(free_time_start),
                            'end': datetimes.datetime_to_readable_format(free_time_end),
                            'tomorrow': datetimes.datetime_str_to_readable_format(tomorrow_str)
                        })
                        phrase_id = 'success'
                    else:
                        phrase_id = 'not_soon'
        except Exception:
            phrase_id = 'error'
            logger.exception('get_nearest_free_time raised exception')

        context.update(self.missing_token_context(credentials))
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text, keyboard, cancel_button=cancel_button)

    @callback_method
    def unblock_car(self, req_info, form, session, response, **kwargs):
        uid = self.get_person_data(req_info)['uid']
        context = {}
        keyboard = None
        try:
            plates = models.User.objects.get(uid=uid).blocked_cars_plates
            if plates:
                keyboard = YES_NO_KEYBOARD
                context['owners'] = []
                for plate in plates:
                    context['owners'] += suggest.find_car_owner(plate)
                phrase_id = 'confirm'
            else:
                phrase_id = 'no_blocked_cars'
        except Exception:
            logger.exception('unblock_car raised exception')
            phrase_id = 'error'

        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text, keyboard)

    @callback_method
    def unblock_car__confirmed(self, req_info, form, session, response, **kwargs):
        if word_assert.is_yes(req_info.utterance.text):
            try:
                phrase_id = 'success'
                person_data = self.get_person_data(req_info)
                credentials = calendar.CalendarRequestCredentials(person_data)
                uid = person_data['uid']
                user = models.User.objects.get(uid=uid)
                from_login = person_data['login']
                person_info = callbacks.get_person_info(from_login, credentials)
                context = {
                    'name': person_info['name'], 'login': person_info['login'], 'gender': person_info['gender'][0]
                }
                _from = from_login + '@yandex-team.ru'
                for plate in user.blocked_cars_plates:
                    owners = suggest.find_car_owner(plate)
                    context['plate'] = plate
                    subject = self.render_phrase(phrase_id='mail_subject', form=form, context=context).text
                    for owner in owners:
                        to = owner['login'] + '@yandex-team.ru'
                        mail.send_mail(_from, to, subject, template_name='unblock_car', context=context)
                user.blocked_cars_plates = []
                user.save()
            except Exception:
                logger.exception('unblock_car raised exception')
                phrase_id = 'error'
        else:
            phrase_id = 'cancel'

        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text)

    @callback_method
    def block_car(self, req_info, form, session, response, **kwargs):
        context = {}
        keyboard = None
        cancel_button = False
        owners = suggest.find_car_owner(form.plate.value)
        if owners is None:
            phrase_id = 'error'
        elif not owners:
            phrase_id = 'car_not_found'
        elif len(owners) == 1:
            phrase_id = 'confirmation'
            form.login.set_value(owners[0]['login'], 'string')
            form.plate.set_value(owners[0]['plate'], 'string')
            keyboard = YES_NO_KEYBOARD
        else:
            plates = [x['plate'].upper() for x in owners]
            if len(set(plates)) == 1:
                logins = [x['login'] for x in owners]
                form.login.set_value(','.join(logins), 'string')
                form.plate.set_value(owners[0]['plate'], 'string')
                phrase_id = 'confirmation'
                keyboard = YES_NO_KEYBOARD
            else:
                phrase_id = 'choose_plate'
                keyboard = plates
                cancel_button = True
        context['owners'] = owners
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text, keyboard, buttons_in_row_count=3, cancel_button=cancel_button)

    @callback_method
    def block_car__confirmed(self, req_info, form, session, response, **kwargs):
        if not word_assert.is_yes(req_info.utterance.text):
            phrase_id = 'do_not_send_mail'
        else:
            phrase_id = 'send_mail'
            try:
                user = models.User.objects.get(uid=self.get_person_data(req_info)['uid'])
                if form.plate.value in user.blocked_cars_plates:
                    phrase_id = 'duplicate'
                else:
                    to_login = form.login.value
                    person_data = self.get_person_data(req_info)
                    credentials = calendar.CalendarRequestCredentials(person_data)
                    from_login = person_data['login']
                    _from = from_login + '@yandex-team.ru'
                    context = callbacks.get_person_info(from_login, credentials)
                    to = [x + '@yandex-team.ru' for x in to_login.split(',')]
                    subject = self.render_phrase(phrase_id='mail_subject', form=form).text
                    user.blocked_cars_plates.append(form.plate.value)
                    user.save()
                    mail.send_mail(_from, to, subject, template_name='block_car', context=context)
            except Exception:
                phrase_id = 'staff_timeout'
                logger.exception('Can\'t get person_info')

        response.reply(self.render_phrase(phrase_id=phrase_id, form=form).text.split('\n\n'))

    @callback_method
    def dms_find_clinic(self, req_info, form, session, response, **kwargs):
        if form['city']:
            person_data = self.get_person_data(req_info)
            models.User.objects.update_or_create(uid=person_data['uid'], defaults={'city': form['city']})
        else:
            city = self.get_person_city(req_info)
            if not city:
                text = self.render_phrase(phrase_id='ask_city', form=form).text
                response.reply(text, cancel_button=True)
                return

            form.city.set_value(city, 'string')

        clinics, total_count, phrase_id = suggest.find_clinics(form['name'], form['city'])
        context = {'clinics': clinics, 'total_count': total_count, 'dms_clinics_url': settings.DMS_CLINICS_URL}
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text)

    def get_person_city(self, req_info):
        person_data = self.get_person_data_from_staff(req_info)
        if person_data['location']['office']['code'] == 'home':
            try:
                city = models.User.objects.values_list('city', flat=True).get(uid=person_data['uid'])
            except models.User.DoesNotExist:
                city = None
        else:
            city = person_data['location']['office']['city']['name']['ru']
        return city

    @callback_method
    def submit_form(self, req_info, form, session, response, **kwargs):
        context = {}
        keyboard = None
        cancel_button = False
        change_to_empty = False
        person_data = self.get_person_data(req_info)

        try:
            # Initializing form
            form_id = settings.FORM_ID_MAPPING[kwargs['form_name']]
            constructors_form = yandex_forms.Form(form_id, form.form_data.value)
            form.form_data.set_value(constructors_form.get_json(), 'string')
            if not form.answers.value:
                form.answers.set_value({field_id: [] for (field_id, _) in constructors_form}, 'dict')

            # We asked user about a retry on the previous step
            if form.retry.value:
                if not word_assert.is_yes(req_info.utterance.text):
                    raise exceptions.CancelException()
                else:
                    form.retry.reset_value()
                    form['answers'][form.prev_field_id.value] = form['answers'][form.prev_field_id.value][:-1]
                    constructors_form.fill(form['answers'], person_data)

            # Form is filled => submit it
            if form.submitted.value:
                change_to_empty = True
                if word_assert.is_yes(req_info.utterance.text):
                    constructors_form.fill(form['answers'], person_data, raise_filled=False)
                    constructors_form.submit(person_data['login'], self.connector)
                else:
                    raise exceptions.CancelException()

            if form.prev_field_id.value:
                prev_field = constructors_form[form.prev_field_id.value]
                if (
                    word_assert.is_skip(req_info.utterance.text)
                    and (
                        not prev_field.is_required()
                        or (
                                prev_field.is_allow_multiple_choice()
                                and len(form['answers'][form.prev_field_id.value]) > 0
                        )
                    )
                ):
                    form['answers'][form.prev_field_id.value].append('_skip')
                else:
                    files = self.connector.get_message_files(req_info.additional_options.get('message'), only_images=False)
                    if files:
                        if isinstance(prev_field, yandex_forms.FileInput):
                            form['answers'][form.prev_field_id.value].extend(files)
                        else:
                            raise yandex_forms.WrongInputError()
                    else:
                        form['answers'][form.prev_field_id.value].append(req_info.utterance.text)

            constructors_form.fill(form['answers'], person_data)

        except exceptions.CancelException:
            change_to_empty = True
            phrase_id = 'cancel'
        except yandex_forms.FilledError:
            phrase_id = 'ask_confirmation'
            context['form'] = str(constructors_form).decode('utf-8')
            form.submitted.set_value(True, 'bool')
            keyboard = YES_NO_KEYBOARD
            form.prev_field_id.reset_value()
        except yandex_forms.NotFilledError as exc:
            phrase_id = 'question'
            next_field = exc.field
        except yandex_forms.SubmitError:
            phrase_id = 'error'
        except yandex_forms.SuccessfulSubmitError:
            phrase_id = 'submitted'
            context['final_text'] = constructors_form.get_final_text()
        except yandex_forms.SuggestNotFoundError:
            phrase_id = 'not_found'
            cancel_button = True
            form['answers'][form.prev_field_id.value].pop()
        except yandex_forms.SuggestNotPerfectlyMatchedError as exc:
            phrase_id = 'suggest'
            keyboard = exc.suggested_items
            cancel_button = True
            form['answers'][form.prev_field_id.value].pop()
        except yandex_forms.WrongInputError:
            phrase_id = 'wrong_input'
            keyboard = YES_NO_KEYBOARD
            form.retry.set_value(True, 'bool')
        except Exception:
            phrase_id = 'error'
            logger.exception('submit_form raised exception')
        else:
            raise AssertionError('This code is unreachable')

        if phrase_id == 'question':
            if isinstance(next_field, yandex_forms.EmptyInput):
                self.change_form(session, form, req_info, response)
            context['question'] = next_field.get_label()
            context['help_text'] = next_field.get_help_text()
            form.prev_field_id.set_value(next_field.id, 'string')
            keyboard = next_field.get_options() or []
            if (
                    not next_field.is_required()
                    or (next_field.is_allow_multiple_choice() and next_field.has_answer())
            ):
                keyboard.append('Перейти к следующему вопросу')
            cancel_button = True

        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text, keyboard, cancel_button=cancel_button)
        if change_to_empty:
            self.change_intent_to_empty(session, req_info, response)

    @callback_method
    def ask_sicklist(self, req_info, form, session, response, **kwargs):
        form.sicklist.set_value('asked', 'string')
        keyboard = YES_NO_KEYBOARD
        message = self.render_phrase(phrase_id='ask', form=form).text
        response.reply(message, keyboard, cancel_button=True)

    @callback_method
    def create_gap(self, req_info, form, session, response, **kwargs):
        keyboard = [x.capitalize() for x in gap.IMPLEMENTED_GAP_TYPES]
        message = self.render_phrase(phrase_id='ask_type', form=form).text
        response.reply(message, keyboard, cancel_button=True)

    @callback_method
    def create_gap__type(self, req_info, form, session, response, **kwargs):
        cancel_button = False
        change_to_empty = False
        if (
                req_info.utterance.text.lower() not in gap.IMPLEMENTED_GAP_TYPES and
                form.retry.value != 'retry' and
                form.from_create_gap_shortcut.value is None
        ):
            phrase_id = 'cancel'
            change_to_empty = True
        else:
            if form.type.value is None:
                form.type.set_value(req_info.utterance.text, 'string')
            if form.type.value.lower() in gap.FULL_DAY_GAP_TYPES:
                phrase_id = 'ask_start'
            else:
                phrase_id = 'ask_start_time'
            cancel_button = True
        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text, cancel_button=cancel_button)
        if change_to_empty:
            self.change_intent_to_empty(session, req_info, response)

    @callback_method
    def create_gap__type__start(self, req_info, form, session, response, sample, **kwargs):
        cancel_button = False
        keyboard = None
        if form.retry.value == 'retry':
            cancel_button = True
            if form.type.value.lower() in gap.FULL_DAY_GAP_TYPES:
                phrase_id = 'ask_end'
            else:
                phrase_id = 'ask_end_time'
        else:
            start = dateparser.parse(sample.text, languages=['ru'])
            if start is None:
                keyboard = YES_NO_KEYBOARD
                phrase_id = 'wrong_format'
            else:
                if (
                        form.type.value.lower() in gap.FULL_DAY_GAP_TYPES
                        and start.microsecond == 0
                        and any((start.hour, start.minute))
                ):
                    keyboard = YES_NO_KEYBOARD
                    phrase_id = 'wrong_format_time'
                else:
                    if start.microsecond != 0:
                        start = start.replace(hour=0, minute=0, second=0)
                    form.start.set_value(datetimes.datetime_to_str(start), 'string')
                    cancel_button = True
                    if form.type.value.lower() in gap.FULL_DAY_GAP_TYPES:
                        phrase_id = 'ask_end'
                    else:
                        phrase_id = 'ask_end_time'
        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text, keyboard, cancel_button=cancel_button)

    @callback_method
    def create_gap__type__start__end(self, req_info, form, session, response, sample, **kwargs):
        keyboard = None
        cancel_button = False
        change_to_empty = False
        if word_assert.is_no(req_info.utterance.text):
            keyboard = None
            phrase_id = 'cancel'
            change_to_empty = True
        elif word_assert.is_yes(req_info.utterance.text):
            new_form = self._dm.new_form('uhura.gap.create_gap__type', previous_form=form)
            new_form.retry.set_value('retry', 'string')
            new_form.type.set_value(form.type.value, 'string')
            self.change_form(session=session, form=new_form, req_info=req_info, response=response)
            return
        else:
            end = dateparser.parse(sample.text, languages=['ru'])
            if end is None:
                keyboard = YES_NO_KEYBOARD
                phrase_id = 'wrong_format'
            else:
                if (
                        form.type.value.lower() in gap.FULL_DAY_GAP_TYPES
                        and end.microsecond == 0
                        and any((end.hour, end.minute))
                ):
                    keyboard = YES_NO_KEYBOARD
                    phrase_id = 'wrong_format_time'
                else:
                    if end.microsecond != 0:
                        end = end.replace(hour=0, minute=0, second=0)
                    form.end.set_value(datetimes.datetime_to_str(end), 'string')
                    cancel_button = True
                    phrase_id = 'ask_comment'
        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text, keyboard, cancel_button=cancel_button)
        if change_to_empty:
            self.change_intent_to_empty(session, req_info, response)

    @callback_method
    def create_gap__type__start__end__comment(self, req_info, form, session, response, **kwargs):
        keyboard = None
        change_to_empty = False
        if word_assert.is_no(req_info.utterance.text):
            phrase_id = 'cancel'
            cancel_button = False
            change_to_empty = True
        elif word_assert.is_yes(req_info.utterance.text):
            new_form = self._dm.new_form('uhura.gap.create_gap__type__start', previous_form=form)
            new_form.type.set_value(form.type.value, 'string')
            new_form.retry.set_value('retry', 'string')
            self.change_form(session=session, form=new_form, req_info=req_info, response=response)
            return
        else:
            form.comment.set_value(req_info.utterance.text, 'string')
            keyboard = YES_NO_KEYBOARD
            cancel_button = True
            phrase_id = 'ask_will_work'
        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text, keyboard, cancel_button=cancel_button)
        if change_to_empty:
            self.change_intent_to_empty(session, req_info, response)

    @callback_method
    def create_gap__type__start__end__comment__work(self, req_info, form, session, response, **kwargs):
        context = {}
        if form.type.value.lower() == 'болезнь':
            if form.sicklist.value is None:
                form.work.set_value(req_info.utterance.text.lower(), 'string')
                new_form = self._dm.new_form('uhura.gap.create_gap__ask_sicklist', previous_form=form)
                self.change_form(
                    session=session,
                    form=new_form,
                    req_info=req_info,
                    response=response
                )
                return
            else:
                if word_assert.is_yes(req_info.utterance.text) or word_assert.is_no(req_info.utterance.text):
                    form.sicklist.set_value(req_info.utterance.text.lower(), 'string')
                else:
                    text = self.render_phrase(phrase_id='cancel', form=form).text
                    response.reply(text)
                    self.change_intent_to_empty(session, req_info, response)
                    return
        else:
            form.work.set_value(req_info.utterance.text.lower(), 'string')
        keyboard = YES_NO_KEYBOARD
        phrase_id = 'confirmation'
        context['start'] = datetimes.datetime_str_to_readable_format(form.start.value)
        context['end'] = datetimes.datetime_str_to_readable_format(form.end.value)
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text, keyboard)

    @callback_method
    def ask_image_to_upload(self, req_info, form, session, response, **kwargs):
        text = self.render_phrase(phrase_id='ask_file', form=form).text
        response.reply(text, cancel_button=True)

    @callback_method
    def upload_image_to_jing(self, req_info, form, session, response, **kwargs):
        context = {}
        try:
            files = self.connector.download_message_files(req_info.additional_options['message'])
            if files is None:
                phrase_id = 'wrong_input'
            else:
                login = self.get_person_data(req_info)['login']
                links = []
                for file_name in files:
                    links.append(jing.upload(file_name, login))
                    os.remove(file_name)
                phrase_id = 'success'
                context['links'] = links
        except Exception:
            logger.exception('Uploading file to jing raised exception')
            phrase_id = 'error'

        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text)
        self.change_intent_to_empty(session, req_info, response)

    def _check_canteen_working_day(self, form, response):
        if calendar.is_working_day(datetime.now()):
            return True
        else:
            text = self.render_phrase(phrase_id='not_working', form=form).text
            response.reply(text)
            return False

    def _get_menu_type(self, req_info, form):
        if not form.office.value:
            login = self.get_person_data(req_info)['login']
            office_info = staff.get_office_by_login(login)
            if office_info['code'] == 'spb':
                office = 'benua'
            else:
                office = 'morozov'
        else:
            office = form.office.value

        menu_type = form.menu_type.value
        if menu_type is None:
            menu_type = 'obed'

        return office, menu_type


class TelegramApp(UhuraApp):
    def generate_notification(self, intent_name, module_name, phrase_id, context):
        form = self.new_form(intent_name, module_name)
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response = UhuraVinsResponse()
        response.reply(text)
        return response

    def get_person_data(self, req_info):
        return req_info.additional_options['user']

    def get_person_data_from_staff(self, req_info):
        user_id = req_info.uuid
        username = req_info.additional_options['telegram_username']
        data = staff.get_person_data_by_telegram_username(username)
        if data is not None:
            return data

        try:
            phone = models.User.objects.values_list('phone', flat=True).get(telegram_usernames__telegram_id=user_id)
        except models.User.DoesNotExist:
            return None
        else:
            data = staff.get_person_data_by_userphone(phone)
            return data

    def update_user_table(self, req_info):
        person_data = self.get_person_data(req_info)
        user, created = models.User.objects.get_or_create(uid=person_data['uid'])
        return user

    def authenticate_and_return_user(self, req_info):
        user_id = req_info.uuid
        username = req_info.additional_options['telegram_username']

        if user_id is None:
            logger.error('Suspicious request')
            return False
        try:
            return models.User.objects.get(telegram_usernames__telegram_id=user_id)
        except models.User.DoesNotExist:
            pass

        if username is None:
            return False
        username = username.lower()
        try:
            user = models.User.objects.get(telegram_usernames__username=username)
        except models.User.DoesNotExist:
            return False
        else:
            TelegramUsername.objects.filter(telegram_id=user_id).delete()
            TelegramUsername.objects.filter(username=username).update(telegram_id=user_id)
            return user

    def render_phrase(self, phrase_id, form=None, context=None, req_info=None):
        context = context or {}
        context.update({'yamb': False, 'telegram': True})
        return super(TelegramApp, self).render_phrase(phrase_id, form, context, req_info)

    def _format_label_and_helptext(self, label, help_text):
        if help_text:
            return '{}\\n<i>{}</i>'.format(label, help_text)
        else:
            return label

    @callback_method
    def get_canteen_queue(self, req_info, form, session, response, **kwargs):
        try:
            cached_value = cache_storage.get(settings.CANTEEN_QUEUE_CACHE_KEY)
            file_id = cached_value['file_id']
            file_timestamp = cached_value['file_timestamp']
            image_data = (file_id, 'file_id', None, None, None)
            text = self.render_phrase(
                phrase_id='image_timestamp_info',
                form=form,
                context={'timestamp': file_timestamp}
            ).text
            response.reply(text, images=[image_data])
        except Exception:
            response.reply(self.render_phrase(phrase_id='error', form=form).text)

    @callback_method
    def get_canteen_menu(self, req_info, form, session, response, **kwargs):
        if not self._check_canteen_working_day(form, response):
            return
        office, menu_type = self._get_menu_type(req_info, form)

        date = datetimes.datetime_to_str(datetime.now().replace(hour=0, minute=0, second=1, microsecond=0))
        cache_key = office + '_' + menu_type

        cached_value = cache_storage.get(cache_key)
        if cached_value == cache_storage.EXPIRED_VALUE:
            last_menu_date = None
            menu_file_id = None
        else:
            last_menu_date = cached_value['last_menu_date']
            menu_file_id = cached_value['file_id']

        if last_menu_date != date:
            try:
                temp_path = wiki.download_canteen_menu(datetimes.str_to_datetime(date), office, menu_type)
                if temp_path is None:
                    response.reply(self.render_phrase(phrase_id='menu_unavailable', form=form).text)
                    return

                cache_value = {'last_menu_date': date}
                image_data = (temp_path, 'path', cache_key, cache_value, None)

                response.reply(images=[image_data])
            except (requests.exceptions.HTTPError, requests.exceptions.ReadTimeout):
                response.reply(self.render_phrase(phrase_id='error', form=form).text)
        else:
            image_data = (menu_file_id, 'file_id', None, None, None)
            response.reply(images=[image_data])

    @callback_method
    def get_my_subscriptions(self, req_info, form, session, response, **kwargs):
        notifications = models.PeriodicNotification.objects.filter(
            user__telegram_usernames__telegram_id=req_info.uuid
        )
        messages = []
        inline_keyboards = [None]
        for notification in notifications:
            messages.append(
                jinja_env.from_string(
                    notification.subscription.description_template
                ).render(time=notification.time, **notification.params)
            )
            inline_keyboards.append({
                'inline_keyboard': [[{
                    'callback_data': 'unsubscribe from {}'.format(notification.id), 'text': 'Удалить'
                }]]
            })
        if messages:
            result_text = [self.render_phrase(phrase_id='my', form=form).text]
        else:
            result_text = [self.render_phrase(phrase_id='no_subscriptions', form=form).text]
        result_text.extend(messages)
        response.reply(result_text, inline_keyboards=inline_keyboards)

    @callback_method
    def get_subscriptions(self, req_info, form, session, response, **kwargs):
        text = self.render_phrase(phrase_id='my_or_all', form=form).text
        keyboard = ['Создать подписку']
        if models.PeriodicNotification.objects.filter(user__telegram_usernames__telegram_id=req_info.uuid).count():
            keyboard.append('Удалить подписку')
        response.reply(text, keyboard, cancel_button=True)

    @callback_method
    def get_all_subscriptions(self, req_info, form, session, response, **kwargs):
        messages = []
        inline_keyboards = [None]
        for subscription in models.Subscription.objects.all():
            messages.append('{}\n\n{}'.format(subscription.name, subscription.description))
            inline_keyboards.append({
                'inline_keyboard': [[{
                    'callback_data': 'Подписка на {}'.format(subscription.name), 'text': 'Добавить'
                }]]
            })
        if messages:
            result_text = [self.render_phrase(phrase_id='all', form=form).text]
        else:
            result_text = [self.render_phrase(phrase_id='error', form=form).text]
        result_text.extend(messages)
        response.reply(result_text, inline_keyboards=inline_keyboards)

    @callback_method
    def unsubscribe(self, req_info, form, session, response, **kwargs):
        try:
            (deleted, _) = models.PeriodicNotification.objects.filter(id=form.id.value).delete()
        except Exception:
            phrase_id = 'error'
        else:
            if deleted:
                phrase_id = 'success'
            else:
                phrase_id = 'doesnt_exist'

        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text)

    @callback_method
    def meetings_digest__ask_template(self, req_info, form, session, response, **kwargs):
        keyboard = [
            '9:00 (встречи на тот же день)',
            '21:00 (встречи на следующий день)',
            '2:00 (встречи на тот же день)',
            'Другое'
        ]
        text = self.render_phrase(phrase_id='ask_template', form=form).text
        response.reply(text, keyboard)

    @callback_method
    def meetings_digest__ask_day(self, req_info, form, session, response, **kwargs):
        utterance_text = req_info.utterance.text
        time = None
        _type = None
        if utterance_text == '9:00 (встречи на тот же день)':
            time = '9:00'
            _type = 'today'
        elif utterance_text == '21:00 (встречи на следующий день)':
            time = '21:00'
            _type = 'tomorrow'
        elif utterance_text == '2:00 (встречи на тот же день)':
            time = '2:00'
            _type = 'today'

        if time and _type:
            new_form = self.new_form('meetings_digest__template__day__time', 'subscriptions')
            new_form.type.set_value(_type, 'string')
            new_form.time.set_value(time, 'string')
            self.change_form(session=session, form=new_form, req_info=req_info, response=response)
        else:
            text = self.render_phrase(phrase_id='ask_day', form=form).text
            keyboard = ['На текущий день', 'На следующий день']
            response.reply(text, keyboard, cancel_button=True)

    @callback_method
    def meetings_digest__ask_time(self, req_info, form, session, response, **kwargs):
        cancel_button = True
        phrase_id = 'ask_time'
        change_to_empty = False
        if req_info.utterance.text == 'На текущий день':
            form.type.set_value('today', 'string')
        elif req_info.utterance.text == 'На следующий день':
            form.type.set_value('tomorrow', 'string')
        else:
            phrase_id = 'wrong_input'
            cancel_button = False
            change_to_empty = True
        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text, cancel_button=cancel_button)
        if change_to_empty:
            self.change_intent_to_empty(session, req_info, response)

    @callback_method
    def meetings_digest__ask_confirmation(self, req_info, form, session, response, **kwargs):
        if isinstance(form.time.value, dict):
            form.time.set_value(datetimes.dict_to_time(form.time.value).strftime('%H:%M'), 'string')
        text = self.render_phrase(
            phrase_id='confirm_' + form.type.value, form=form, context={'time': form.time.value}
        ).text
        response.reply(text, YES_NO_KEYBOARD)

    @callback_method
    def meetings_digest__finish(self, req_info, form, session, response, **kwargs):
        if word_assert.is_yes(form.confirmed.value):
            try:
                user_tz = staff.get_timezone_by_uid(self.get_person_data(req_info)['uid'])
                hour, minute = map(int, form.time.value.split(':'))
                converted_dt = datetimes.convert_datetime(
                    datetime.now().replace(hour=hour, minute=minute), user_tz, 'Europe/Moscow'
                )
                converted_now = datetimes.convert_datetime(datetime.now(), user_tz, 'Europe/Moscow')
                if converted_now > converted_dt:
                    last_success_date = datetime.now()
                else:
                    last_success_date = datetime.now() - timedelta(days=1)

                (_, created) = models.PeriodicNotification.objects.get_or_create(
                    user=models.User.objects.get(telegram_usernames__telegram_id=req_info.uuid),
                    subscription=models.Subscription.objects.get(task_name='meetings_digest'),
                    time=converted_dt.strftime('%H:%M'),
                    params={
                        '_type': form.type.value,
                        'uid': self.get_person_data(req_info)['uid'],
                        'chat_id': req_info.uuid
                    },
                    last_success_date=last_success_date
                )
                if created:
                    phrase_id = 'success'
                else:
                    phrase_id = 'duplicate'
            except Exception:
                logger.exception('Creating meeting digest raised exception')
                phrase_id = 'error'
        else:
            phrase_id = 'cancel'
        context = {'time': form.time.value, 'type': form.type.value}
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text)

    @callback_method
    def not_permitted(self, req_info, form, session, response, **kwargs):
        request_user_contact = False
        cancel_button = False
        change_to_empty = False
        if req_info.additional_options.get('not_permitted_cancel'):
            phrase_id = 'not_permitted_cancel'
            change_to_empty = True
        elif req_info.additional_options.get('contact_info'):
            contact_info = req_info.additional_options['contact_info']
            logger.debug('Contact info received: %r', contact_info)

            contact_id = contact_info['user_id']
            if contact_id != int(req_info.uuid):
                phrase_id = 'not_permitted_hack'
                request_user_contact = True
                cancel_button = True
            else:
                phone = contact_info['phone_number']
                user = get_user_by_userphone(phone)

                if user is None:
                    phrase_id = 'phone_auth_fail'
                else:
                    phrase_id = 'phone_auth_success'
                    try:
                        user.register_telegram_id(req_info.uuid)
                    except Exception:
                        phrase_id = 'error'
                change_to_empty = True
        else:
            if req_info.additional_options.get('not_permitted'):
                phrase_id = 'not_permitted'
            else:
                phrase_id = 'auth_requested'
            request_user_contact = True
            cancel_button = True

        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text, request_user_contact=request_user_contact, cancel_button=cancel_button)
        if change_to_empty:
            self.change_intent_to_empty(session, req_info, response)

    @callback_method
    def get_commands_list__ellipsis(self, req_info, form, session, response, **kwargs):
        text = self.render_phrase(phrase_id='tg_' + form.menu_item.value, form=form).text
        response.reply(text)

    @callback_method
    def who_is_it(self, req_info, form, session, response, **kwargs):
        info = None
        forward_from = req_info.additional_options.get('forward_from')
        if forward_from:
            username = forward_from.get('username')
            user_id = forward_from.get('id')
            staff_obj = None
            found_by_username = False
            if username:
                try:
                    staff_obj = staff.get_person_data_by_telegram_username(username)
                    if staff_obj is None:
                        phrase_id = 'not_found'
                    else:
                        found_by_username = True
                except Exception:
                    logger.exception('who_is_it raised exception')
                    phrase_id = 'error'

            if user_id and not found_by_username:
                try:
                    phone = (
                        models.User.objects.values_list('phone', flat=True)
                        .get(telegram_usernames__telegram_id=user_id)
                    )
                    staff_obj = staff.get_person_data_by_userphone(phone)
                    if staff_obj is None:
                        phrase_id = 'not_found'
                except models.User.DoesNotExist:
                    phrase_id = 'not_found'
                except Exception:
                    logger.exception('who_is_it raised exception')
                    phrase_id = 'error'

            if staff_obj:
                try:
                    staff_login = staff_obj['login']
                    person_data = self.get_person_data(req_info)
                    credentials = calendar.CalendarRequestCredentials(person_data)
                    info = callbacks.get_person_info(staff_login, credentials)
                    info['format_dict'] = gap.FORMAT_DICT
                    info['online'] = staff.get_last_online(staff_login)
                    phrase_id = 'success'
                except Exception:
                    logger.exception('who_is_it raised exception')
                    phrase_id = 'error'
        else:
            phrase_id = 'not_forwarded'
        text = self.render_phrase(phrase_id=phrase_id, form=form, context={'person': info}).text
        response.reply(text)
        self.change_intent_to_empty(session, req_info, response)

    @callback_method
    def work_from_home(self, req_info, form, session, response, **kwargs):
        text = self.render_phrase(phrase_id='ask_confirmation', form=form, context={'date': datetime.now()}).text
        response.reply(text, YES_NO_KEYBOARD)

    @callback_method
    def create_telegram_chat(self, req_info, form, session, response, **kwargs):
        text = self.render_phrase(phrase_id='ask_chat_title', form=form).text
        response.reply(text)

    @callback_method
    def create_telegram_chat__title(self, req_info, form, session, response, **kwargs):
        form.title.set_value(req_info.utterance.text, 'string')
        phrase_id = 'ask_confirmation'
        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        keyboard = YES_NO_KEYBOARD
        response.reply(text, keyboard)

    @callback_method
    def create_telegram_chat__title__confirmed(self, req_info, form, session, response, **kwargs):
        if word_assert.is_yes(req_info.utterance.text):
            username = req_info.additional_options['telegram_username']
            try:
                tasha_response = tasha.create_chat(form.title.value, req_info.uuid, username)
            except Exception:
                phrase_id = 'error'
                logger.exception('Telegram chat create error')
            else:
                if tasha_response is None or 'status' not in tasha_response:
                    phrase_id = 'error'
                elif tasha_response['status'] == 'success':
                    phrase_id = 'chat_created'
                elif tasha_response['status'] == 'privacy_error':
                    phrase_id = 'privacy_error'
                else:
                    logger.error('Unknown status in tasha_response')
                    phrase_id = 'error'
        else:
            phrase_id = 'cancel'
        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text)
        self.change_intent_to_empty(session, req_info, response)

    @callback_method
    def invite_tasha_bot_to_chat(self, req_info, form, session, response, **kwargs):
        text = self.render_phrase(phrase_id='ask_invite_link', form=form).text
        response.reply(text)

    @callback_method
    def invite_tasha_bot_to_chat__link(self, req_info, form, session, response, **kwargs):
        invite_link = req_info.utterance.text
        bot_username = None
        try:
            (phrase_id, bot_username) = tasha.invite_to_chat_by_link(invite_link)
        except Exception:
            logger.exception('Invite by link error')
            phrase_id = 'error'
        context = {'bot_username': bot_username}
        text = self.render_phrase(phrase_id=phrase_id, form=form, context=context).text
        response.reply(text)
        self.change_intent_to_empty(session, req_info, response)

    @callback_method
    def quest_start(self, req_info, form, session, response, **kwargs):
        keyboard = []
        cancel_button = False
        new_form = None
        if yenv.type == 'production':
            phrase_id = 'not_in_production_yet'
        else:
            try:
                (progress, _) = models.QuestProgress.objects.get_or_create(
                    user=models.User.objects.get(telegram_usernames__telegram_id=req_info.uuid)
                )
                passed_tasks_count = progress.passed_tasks_count
            except Exception:
                logger.exception('show_quest_tasks raised exception')
                phrase_id = 'error'
                new_form = self.new_form('empty_intent', 'utils')
            else:
                if passed_tasks_count == 0:
                    phrase_id = 'intro'
                    new_form = self.new_form(settings.QUEST_TASKS[passed_tasks_count], 'quest')
                elif passed_tasks_count < len(settings.QUEST_TASKS):
                    phrase_id = 'continue_or_restart'
                    keyboard = ['Начать сначала', 'Продолжить']
                    cancel_button = True
                else:
                    phrase_id = 'complete'
                    keyboard = ['Начать сначала']
                    cancel_button = True

        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text.split('\n\n'), keyboard, cancel_button=cancel_button)
        if new_form:
            self.change_form(session=session, form=new_form, req_info=req_info, response=response)

    @callback_method
    def quest_clear_progress(self, req_info, form, session, response, **kwargs):
        new_form = None
        try:
            (
                models.QuestProgress.objects
                .filter(user__telegram_usernames__telegram_id=req_info.uuid)
                .update(passed_tasks_count=0)
            )
            phrase_id = 'cleared'
            new_form = self.new_form(settings.QUEST_TASKS[0], 'quest')
        except Exception:
            phrase_id = 'error'
            logger.exception('clear_quest_progress raised exception')

        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text)
        if new_form:
            self.change_form(session=session, form=new_form, req_info=req_info, response=response)

    @callback_method
    def quest_continue(self, req_info, form, session, response, **kwargs):
        new_form = None
        try:
            phrase_id = 'task'
            passed_tasks_count = models.QuestProgress.objects.get(
                user__telegram_usernames__telegram_id=req_info.uuid
            ).passed_tasks_count
            new_form = self.new_form(settings.QUEST_TASKS[passed_tasks_count], 'quest')
        except Exception:
            phrase_id = 'error'
            logger.exception('quest_continue raised exception')

        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text)
        if new_form:
            self.change_form(session=session, form=new_form, req_info=req_info, response=response)

    @callback_method
    def quest_ask_for_retry(self, req_info, form, session, response, **kwargs):
        new_form = None
        if word_assert.is_yes(req_info.utterance.text):
            try:
                passed_tasks_count = models.QuestProgress.objects.get(
                    user__telegram_usernames__telegram_id=req_info.uuid
                ).passed_tasks_count
            except Exception:
                phrase_id = 'error'
                logger.exception('quest_ask_for_retry raised exception')
            else:
                phrase_id = 'retry'
                new_form = self.new_form(settings.QUEST_TASKS[passed_tasks_count], 'quest')
        else:
            phrase_id = 'cancel'

        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text)
        if new_form:
            self.change_form(session=session, form=new_form, req_info=req_info, response=response)

    def _update_quest_progress(self, req_info):
        quest_progress = models.QuestProgress.objects.get(
            user__telegram_usernames__telegram_id=req_info.uuid
        )
        quest_progress.passed_tasks_count += 1
        quest_progress.save(update_fields=['passed_tasks_count'])
        if quest_progress.passed_tasks_count < len(settings.QUEST_TASKS):
            return (self.new_form(settings.QUEST_TASKS[quest_progress.passed_tasks_count], 'quest'), 'next_task')
        else:
            return (self.new_form('empty_intent', 'utils'), 'finish')

    @callback_method
    def quest_where_is_hural(self, req_info, form, session, response, **kwargs):
        text = self.render_phrase(phrase_id='ask', form=form).text
        keyboard = ['Синий Кит', 'Седьмое Небо', 'Еще переговорка', 'Еще одна переговорка']
        response.reply(text, keyboard, cancel_button=True)

    @callback_method
    def quest_where_is_hural__check_answer(self, req_info, form, session, response, **kwargs):
        new_form = None
        keyboard = None
        if req_info.utterance.text.lower() == 'синий кит':
            try:
                (new_form, phrase_id) = self._update_quest_progress(req_info)
            except Exception:
                phrase_id = 'error'
                logger.exception('quest_where_is_hural__check_answer raised exception')
                new_form = self.new_form('empty_intent', 'utils')
        else:
            phrase_id = 'wrong_answer'
            keyboard = YES_NO_KEYBOARD
        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text, keyboard)
        if new_form:
            self.change_form(session=session, form=new_form, req_info=req_info, response=response)

    @callback_method
    def quest_create_ticket__check_answer(self, req_info, form, session, response, **kwargs):
        new_form = None
        keyboard = None
        key = '-'.join(req_info.utterance.text.split('/')[-1].split()).upper()
        if not re.match('^UHURAQUEST-\d+$', key):
            phrase_id = 'wrong_format'
            keyboard = YES_NO_KEYBOARD
        else:
            try:
                ticket = startrek.get_ticket(key)
                if ticket is None:
                    phrase_id = 'not_found'
                    keyboard = YES_NO_KEYBOARD
                elif self.get_person_data(req_info)['login'] == ticket['author']:
                    (new_form, phrase_id) = self._update_quest_progress(req_info)
                else:
                    phrase_id = 'wrong_author'
                    keyboard = YES_NO_KEYBOARD
            except Exception:
                phrase_id = 'error'
                new_form = self.new_form('empty_intent', 'utils')
                logger.exception('quest_create_ticket__check_answer raised exception')

        text = self.render_phrase(phrase_id=phrase_id, form=form).text
        response.reply(text, keyboard)
        if new_form:
            self.change_form(session=session, form=new_form, req_info=req_info, response=response)


class YambApp(UhuraApp):
    last_menu_date = None
    last_menu_url = None

    def authenticate_and_return_user(self, req_info):
        return self.update_user_table(req_info)

    def get_person_data(self, req_info):
        return req_info.additional_options['user']

    def update_user_table(self, req_info):
        uid = req_info.additional_options['uid']
        user = models.User.objects.get(uid=uid)
        if user.yamb_id != req_info.uuid:
            user.yamb_id = req_info.uuid
            user.save(update_fields=['yamb_id'])

        return user

    def render_phrase(self, phrase_id, form=None, context=None, req_info=None):
        context = context or {}
        context.update({'yamb': True, 'telegram': False})
        return super(YambApp, self).render_phrase(phrase_id, form, context, req_info)

    def render_yamb_lapki(self, req_info, form, session, response, **kwargs):
        response.reply(self.render_phrase(phrase_id='yamb_lapki', form=form).text)
        self.change_intent_to_empty(session, req_info, response)

    def _format_label_and_helptext(self, label, help_text):
        if help_text:
            return '\\n'.join([label, help_text])
        else:
            return label

    @callback_method
    def get_all_subscriptions(self, req_info, form, session, response, **kwargs):
        self.render_yamb_lapki(req_info, form, session, response, **kwargs)

    @callback_method
    def get_my_subscriptions(self, req_info, form, session, response, **kwargs):
        self.render_yamb_lapki(req_info, form, session, response, **kwargs)

    @callback_method
    def get_subscriptions(self, req_info, form, session, response, **kwargs):
        self.render_yamb_lapki(req_info, form, session, response, **kwargs)

    @callback_method
    def meetings_digest__ask_template(self, req_info, form, session, response, **kwargs):
        self.render_yamb_lapki(req_info, form, session, response, **kwargs)

    @callback_method
    def get_canteen_queue(self, req_info, form, session, response, **kwargs):
        try:
            path = os.path.join(settings.FILE_CACHE_PATH, settings.CANTEEN_QUEUE_CACHE_FILE)
            unix_timestamp = os.path.getmtime(path)
            file_timestamp = datetime.fromtimestamp(unix_timestamp).strftime('%H:%M')
            image_data = (path, 'path', None, None, None)
            text = self.render_phrase(
                phrase_id='image_timestamp_info',
                form=form,
                context={'timestamp': file_timestamp}
            ).text
            response.reply(text, images=[image_data])
        except Exception:
            response.reply(self.render_phrase(phrase_id='error', form=form).text)

    @callback_method
    def quest_start(self, req_info, form, session, response, **kwargs):
        self.render_yamb_lapki(req_info, form, session, response, **kwargs)

    @callback_method
    def who_is_it(self, req_info, form, session, response, **kwargs):
        self.render_yamb_lapki(req_info, form, session, response, **kwargs)

    @callback_method
    def create_telegram_chat(self, req_info, form, session, response, **kwargs):
        self.render_yamb_lapki(req_info, form, session, response, **kwargs)

    @callback_method
    def invite_tasha_bot_to_chat(self, req_info, form, session, response, **kwargs):
        self.render_yamb_lapki(req_info, form, session, response, **kwargs)

    @callback_method
    def get_commands_list__ellipsis(self, req_info, form, session, response, **kwargs):
        text = self.render_phrase(phrase_id='yamb_' + form.menu_item.value, form=form).text
        response.reply(text)

    @callback_method
    def get_canteen_menu(self, req_info, form, session, response, **kwargs):
        if not self._check_canteen_working_day(form, response):
            return
        office, menu_type = self._get_menu_type(req_info, form)

        date = datetimes.datetime_to_str(datetime.now().replace(hour=0, minute=0, second=1, microsecond=0))
        cache_key = office + '_' + menu_type
        meta_path = os.path.join(
            settings.FILE_CACHE_PATH,
            '{}_{}'.format(cache_key, settings.CANTEEN_MENU_CACHE_META_FILE)
        )
        menu_path = os.path.join(
            settings.FILE_CACHE_PATH,
            '{}_{}'.format(cache_key, settings.CANTEEN_MENU_CACHE_FILE)
        )

        if os.path.exists(meta_path):
            with open(meta_path) as meta:
                last_menu_date = json.load(meta)['date']
        else:
            last_menu_date = None

        if last_menu_date != date:
            try:
                temp_path = wiki.download_canteen_menu(datetimes.str_to_datetime(date), office, menu_type)
                if temp_path is None:
                    response.reply(self.render_phrase(phrase_id='menu_unavailable', form=form).text)
                    return

                shutil.copy(temp_path, menu_path)
                temp_meta_path = meta_path + '-tmp.jpg'
                with open(temp_meta_path, 'w') as meta:
                    json.dump({'date': date}, meta)
                shutil.move(temp_meta_path, meta_path)

                image_data = (menu_path, 'path', None, None, None)

                response.reply(images=[image_data])
            except (requests.exceptions.HTTPError, requests.exceptions.ReadTimeout):
                response.reply(self.render_phrase(phrase_id='error', form=form).text)
        else:
            image_data = (menu_path, 'path', None, None, None)
            response.reply(images=[image_data])


def create_app_connector():
    if settings.BOT_TYPE == 'telegram':
        app_connector = connectors.TelegramConnector(settings.TELEGRAM_TOKEN, TelegramApp())
    elif settings.BOT_TYPE == 'yamb':
        app_connector = connectors.YambConnector(settings.YAMB_TOKEN, YambApp(), base_url=settings.YAMB_API_BASE_URL)
    else:
        raise ValueError('settings.BOT_TYPE must be "telegram" or "yamb"')
    return app_connector


def get_user_by_userphone(phone):
    phone = ''.join(c for c in phone if c.isdigit())
    phone = r'^\+%s$' % phone
    if not phone:
        return None
    try:
        return models.User.objects.get(phone__regex=phone)
    except models.User.DoesNotExist:
        return None


def put_user(req_info, user):
    user = model_to_dict(user)
    user['login'] = user['username']
    del user['leave_at']
    del user['quit_at']
    req_info.additional_options['user'] = user


if os.environ.get('CREATE_UHURA_APP'):
    # Application instance singletone
    app_connector = create_app_connector()
